package io.aether.clickhouse;

import com.clickhouse.client.*;
import com.clickhouse.data.ClickHouseFormat;
import com.clickhouse.data.ClickHouseRecord;

import java.net.URI;
import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * High-performance asynchronous logger for ClickHouse.
 * It dynamically adds new columns to the target table using ALTER TABLE IF NOT EXISTS.
 */
public class ClickHouseLogger implements AutoCloseable {

    private static final int DEFAULT_HTTP_PORT = 8123;
    private static final int DEFAULT_HTTPS_PORT = 8443;
    private final ClickHouseNode server;
    private final String databaseName;
    private final String tableName;
    private final List<Map<String, Object>> buffer; // Изменено на Object для поддержки разных типов
    private final ScheduledExecutorService scheduler;
    // Set to track all column names and their types encountered so far.
    private final Map<String, String> knownColumnTypes; // Храним типы колонок
    private final long batchIntervalMs = 5000;
    private final int batchSize = 10000;

    /**
     * Initializes the logger from a URL in the format http(s)://user:password@host[:port]/database/table
     *
     * @param clickHouseUrl The full URL to ClickHouse including database and table
     */
    public ClickHouseLogger(String clickHouseUrl) {

        URI uri;
        try {
            uri = new URI(clickHouseUrl);
        } catch (Exception e) {
            throw new IllegalArgumentException("Invalid ClickHouse URL format: " + e.getMessage(), e);
        }

        // 1. Check scheme and determine protocol/port
        String scheme = uri.getScheme();
        ClickHouseProtocol protocol;
        int usedPort;

        if ("https".equalsIgnoreCase(scheme)) {
            protocol = ClickHouseProtocol.HTTP;
            usedPort = (uri.getPort() == -1) ? DEFAULT_HTTPS_PORT : uri.getPort();
        } else if ("http".equalsIgnoreCase(scheme)) {
            protocol = ClickHouseProtocol.HTTP;
            usedPort = (uri.getPort() == -1) ? DEFAULT_HTTP_PORT : uri.getPort();
        } else {
            throw new IllegalArgumentException("Unsupported scheme. Please use 'http://' or 'https://'.");
        }

        // 2. Extract components from the URI
        String host = uri.getHost();
        String userInfo = uri.getUserInfo();
        String path = uri.getPath();

        if (host == null || userInfo == null || path == null || path.length() <= 1) {
            throw new IllegalArgumentException("ClickHouse URL is missing required components (host, user:password, or /database/table path).");
        }

        String[] userPass = userInfo.split(":");
        String user = userPass[0];
        String password = (userPass.length > 1) ? userPass[1] : "";

        String[] pathSegments = path.substring(1).split("/");
        if (pathSegments.length != 2) {
            throw new IllegalArgumentException("URL path must be in format /database/table.");
        }
        this.databaseName = pathSegments[0];
        this.tableName = pathSegments[1];

        // 3. Configure ClickHouseNode
        this.server = ClickHouseNode.builder()
                .host(host)
                .port(protocol, usedPort)
                .database(databaseName)
                .credentials(ClickHouseCredentials.fromUserAndPassword(user, password))
                .build();

        // 4. Initialize buffer, column set, and scheduler
        this.buffer = new CopyOnWriteArrayList<>();
        this.knownColumnTypes = Collections.synchronizedMap(new HashMap<>());
        this.scheduler = Executors.newSingleThreadScheduledExecutor(r -> new Thread(r, "ClickHouse-Log-Sender"));
        this.scheduler.scheduleAtFixedRate(this::flush, batchIntervalMs, batchIntervalMs, TimeUnit.MILLISECONDS);

        // 5. Проверяем/создаем таблицу при инициализации
        ensureTableExists();
    }

    /**
     * Determines ClickHouse data type based on Java object type
     */
    private String determineColumnType(Object value) {
        if (value == null) {
            return "String"; // По умолчанию String для null
        }

        if (value instanceof Number) {
            if (value instanceof Integer || value instanceof Long || value instanceof Short || value instanceof Byte) {
                return "Int64";
            } else if (value instanceof Float || value instanceof Double) {
                return "Float64";
            } else if (value instanceof java.math.BigDecimal) {
                return "Decimal(18,6)";
            }
        } else if (value instanceof Boolean) {
            return "Bool";
        } else if (value instanceof java.time.LocalDate) {
            return "Date";
        } else if (value instanceof java.time.LocalDateTime || value instanceof java.util.Date) {
            return "DateTime";
        }

        return "String"; // По умолчанию String
    }

    /**
     * Saves a single log record to the buffer and updates the set of known column names and types.
     * Triggers an immediate flush if the buffer limit is reached.
     *
     * @param record The log record as a Map&lt;String, Object&gt;
     */
    public void save(Map<String, Object> record) {
        if (record == null || record.isEmpty()) return;

        // Update the set of known column names and types
        synchronized (knownColumnTypes) {
            for (Map.Entry<String, Object> entry : record.entrySet()) {
                String columnName = entry.getKey();
                Object value = entry.getValue();

                // Определяем тип для новой колонки или проверяем совместимость для существующей
                String newType = determineColumnType(value);
                String existingType = knownColumnTypes.get(columnName);

                if (existingType == null) {
                    // Новая колонка
                    knownColumnTypes.put(columnName, newType);
                } else if (!existingType.equals(newType)) {
                    // Конфликт типов - используем более широкий тип
                    String widerType = getWiderType(existingType, newType);
                    if (!existingType.equals(widerType)) {
                        knownColumnTypes.put(columnName, widerType);
                        System.out.printf("Column '%s' type changed from %s to %s%n",
                                columnName, existingType, widerType);
                    }
                }
            }
        }

        buffer.add(record);

        // Immediate send if limit is exceeded
        if (buffer.size() >= batchSize) {
            scheduler.submit(this::flush);
        }
    }

    /**
     * Determines wider type when there's a type conflict
     */
    private String getWiderType(String type1, String type2) {
        // Если один из типов String, возвращаем String (самый широкий)
        if ("String".equals(type1) || "String".equals(type2)) {
            return "String";
        }

        // Приоритет числовых типов
        Set<String> numericTypes = Set.of("Int64", "Float64", "Decimal(18,6)");
        if (numericTypes.contains(type1) && numericTypes.contains(type2)) {
            // Float64 шире Int64
            if ("Float64".equals(type1) || "Float64".equals(type2)) {
                return "Float64";
            }
            return "Int64";
        }

        // По умолчанию возвращаем String как самый безопасный тип
        return "String";
    }

    /**
     * Ensures that the table exists and has required structure
     */
    private void ensureTableExists() {
        try (ClickHouseClient client = ClickHouseClient.newInstance()) {
            // Проверяем существование таблицы
            String checkTableSql = String.format(
                    "SELECT name FROM system.tables WHERE database = '%s' AND name = '%s'",
                    databaseName, tableName);

            boolean tableExists = false;
            try (ClickHouseResponse response = client.read(server).query(checkTableSql).executeAndWait()) {
                tableExists = response.records().iterator().hasNext();
            }

            if (!tableExists) {
                // Создаем таблицу с минимальной структурой
                String createTableSql = String.format(
                        "CREATE TABLE IF NOT EXISTS %s.%s (" +
                        "    id UUID DEFAULT generateUUIDv4(), " + // Добавляем первичный ключ
                        "    timestamp DateTime DEFAULT now()" +
                        ") ENGINE = MergeTree() " +
                        "PRIMARY KEY (id) " +
                        "ORDER BY (id, timestamp)",
                        databaseName, tableName);

                try (ClickHouseResponse response = client.read(server).query(createTableSql).executeAndWait()) {
                    System.out.printf("Created table %s.%s with primary key%n", databaseName, tableName);
                }
            }

        } catch (Exception e) {
            System.err.printf("Failed to ensure table exists: %s%n", e.getMessage());
            e.printStackTrace();
        }
    }

    /**
     * Sends the current contents of the buffer to ClickHouse.
     */
    private void flush() {
        if (buffer.isEmpty()) {
            return;
        }

        // 1. Check and create new columns before inserting data
        ensureColumnsExist();

        // 2. Atomically retrieve buffer contents
        List<Map<String, Object>> recordsToSend = new ArrayList<>();
        synchronized (buffer) {
            recordsToSend.addAll(buffer);
            buffer.clear();
        }

        if (recordsToSend.isEmpty()) {
            return;
        }

        // 3. Perform batch insert
        try (ClickHouseClient client = ClickHouseClient.newInstance()) {

            // Формируем данные в формате JSONEachRow с учетом типов
            StringBuilder dataBuilder = new StringBuilder();
            for (Map<String, Object> record : recordsToSend) {
                dataBuilder.append("{");
                boolean first = true;
                for (Map.Entry<String, Object> entry : record.entrySet()) {
                    if (!first) {
                        dataBuilder.append(",");
                    }
                    String key = entry.getKey();
                    Object value = entry.getValue();

                    dataBuilder.append("\"").append(key).append("\":");

                    // Форматируем значение в зависимости от типа
                    if (value == null) {
                        dataBuilder.append("null");
                    } else if (value instanceof Number || value instanceof Boolean) {
                        // Числа и булевы значения без кавычек
                        dataBuilder.append(value);
                    } else {
                        // Строки и другие типы - в кавычках с экранированием
                        String stringValue = value.toString()
                                .replace("\\", "\\\\")
                                .replace("\"", "\\\"")
                                .replace("\n", "\\n")
                                .replace("\r", "\\r")
                                .replace("\t", "\\t");
                        dataBuilder.append("\"").append(stringValue).append("\"");
                    }
                    first = false;
                }
                dataBuilder.append("}\n");
            }

            // Используем write() для мутации (INSERT) и data() для передачи данных
            ClickHouseRequest.Mutation request = client.read(server)
                    .write()
                    .table(tableName)
                    .format(ClickHouseFormat.JSONEachRow);

            try (ClickHouseResponse response = request.data(dataBuilder.toString()).executeAndWait()) {
                System.out.printf("Successfully flushed %d logs to ClickHouse%n", recordsToSend.size());
            }

        } catch (Exception e) {
            System.err.printf("ClickHouse flush error. Dropping %d records. Message: %s%n",
                    recordsToSend.size(), e.getMessage());
            e.printStackTrace();
        }
    }

    /**
     * Retrieves the current schema from ClickHouse and sends ALTER TABLE commands
     * for any unknown columns in the 'knownColumnTypes' map.
     */
    private void ensureColumnsExist() {
        if (knownColumnTypes.isEmpty()) {
            return;
        }

        try (ClickHouseClient client = ClickHouseClient.newInstance()) {

            synchronized (knownColumnTypes) {
                // Получаем существующие колонки из ClickHouse
                Set<String> existingColumns = new HashSet<>();
                String getColumnsSql = String.format(
                        "SELECT name FROM system.columns WHERE database = '%s' AND table = '%s'",
                        databaseName, tableName);

                try (ClickHouseResponse response = client.read(server).query(getColumnsSql).executeAndWait()) {
                    for (ClickHouseRecord record : response.records()) {
                        existingColumns.add(record.getValue(0).asString());
                    }
                }

                // Добавляем только новые колонки
                StringBuilder alterSql = new StringBuilder("ALTER TABLE ")
                        .append(databaseName).append(".").append(tableName);

                boolean first = true;
                for (Map.Entry<String, String> columnEntry : knownColumnTypes.entrySet()) {
                    String columnName = columnEntry.getKey();
                    String columnType = columnEntry.getValue();

                    if (!existingColumns.contains(columnName)) {
                        if (!first) {
                            alterSql.append(",");
                        }
                        alterSql.append(" ADD COLUMN IF NOT EXISTS `").append(columnName)
                                .append("` ").append(columnType);
                        first = false;
                    }
                }

                if (!first) {
                    // Execute the ALTER TABLE command
                    String sql = alterSql.toString();
                    try (ClickHouseResponse response = client.read(server).query(sql).executeAndWait()) {
                        System.out.printf("Executed ALTER TABLE: %s%n", sql);
                    }
                }
            }

        } catch (Exception e) {
            System.err.printf("Failed to execute ALTER TABLE for new columns. Message: %s%n",
                    e.getMessage());
            e.printStackTrace();
        }
    }

    /**
     * Clears any remaining buffer contents and stops the scheduler upon application shutdown.
     */
    @Override
    public void close() {
        // Stop the scheduler
        scheduler.shutdown();
        try {
            if (!scheduler.awaitTermination(60, TimeUnit.SECONDS)) {
                scheduler.shutdownNow();
            }
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            scheduler.shutdownNow();
        }

        // Perform final buffer flush, which includes the final column check
        flush();
    }
}