package io.aether.api;

import io.aether.api.common.*;
import io.aether.api.common.Sign;
import io.aether.api.common.SignedKey;
import io.aether.crypto.*;
import io.aether.utils.RU;

public class CryptoUtils {
    public static PairAsymKeysSigned of(PairKeysAsymSigned k) {
        return new PairAsymKeysSigned(of(k.getPublicKey()), of(k.getPrivateKey()));
    }

    public static io.aether.crypto.SignedKey of(SignedKey key) {
        return new io.aether.crypto.SignedKey(of(key.getKey()), of(key.getSign()));
    }

    public static boolean verifySign(io.aether.crypto.SignedKey key, Iterable<SignChecker> checkers) {
        var k = key.getKey();
        var s = key.getSign();
        for (var c : checkers) {
            if (c.getCryptoProvider() == k.getCryptoProvider() && c.checkSign(k.getData(), s)) return true;
        }
        return false;
    }

    public static SignedKey of(io.aether.crypto.SignedKey key) {
        return new SignedKey(of(key.key), of(key.sign));
    }

    public static io.aether.crypto.Sign of(Sign sign) {
        if (sign instanceof SignAE_ED25519) {
            return CryptoProviderFactory.getProvider("SODIUM").createSign(sign.getData());
        } else if (sign instanceof SignHYDROGEN) {
            return CryptoProviderFactory.getProvider("HYDROGEN").createSign(sign.getData());
        } else {
            throw new UnsupportedOperationException();
        }
    }

    public static Sign of(io.aether.crypto.Sign sign) {
        switch (sign.getProviderName().toLowerCase()){
            case "sodium":
                return new SignAE_ED25519(sign.getSignData());
            case "hydrogen":
                return new SignHYDROGEN(sign.getSignData());
        }
        throw new UnsupportedOperationException();
    }

    public static <T extends AKey> T of(Key key) {
        if (key instanceof SodiumCurvePrivate) {
            return CryptoProviderFactory.getProvider("SODIUM").createKey(KeyType.ASYMMETRIC_PRIVATE, key.getData());
        } else if (key instanceof SodiumSignPublic) {
            return CryptoProviderFactory.getProvider("SODIUM").createKey(KeyType.SIGN_PUBLIC, key.getData());
        } else if (key instanceof SodiumSignPrivate) {
            return CryptoProviderFactory.getProvider("SODIUM").createKey(KeyType.SIGN_PRIVATE, key.getData());
        } else if (key instanceof SodiumCurvePublic) {
            return CryptoProviderFactory.getProvider("SODIUM").createKey(KeyType.ASYMMETRIC_PUBLIC, key.getData());
        } else if (key instanceof HydrogenCurvePrivate) {
            return CryptoProviderFactory.getProvider("HYDROGEN").createKey(KeyType.ASYMMETRIC_PRIVATE, key.getData());
        } else if (key instanceof HydrogenCurvePublic) {
            return CryptoProviderFactory.getProvider("HYDROGEN").createKey(KeyType.ASYMMETRIC_PUBLIC, key.getData());
        } else if (key instanceof HydrogenSignPublic) {
            return CryptoProviderFactory.getProvider("HYDROGEN").createKey(KeyType.SIGN_PUBLIC, key.getData());
        } else if (key instanceof HydrogenSignPrivate) {
            return CryptoProviderFactory.getProvider("HYDROGEN").createKey(KeyType.SIGN_PRIVATE, key.getData());
        } else if (key instanceof SodiumChacha20Poly1305) {
            return CryptoProviderFactory.getProvider("SODIUM").createKey(KeyType.SYMMETRIC, key.getData());
        } else if (key instanceof HydrogenSecretBox) {
            return CryptoProviderFactory.getProvider("HYDROGEN").createKey(KeyType.SYMMETRIC, key.getData());
        } else {
            throw new UnsupportedOperationException();
        }
    }

    public static CryptoEngine makeProvider(Key key) {
        var k = of(key);
        switch (k.getKeyType()) {
            case ASYMMETRIC_PUBLIC:
                return ((AKey.AsymmetricPublic) k).toCryptoEngine();
            case SYMMETRIC:
                return ((AKey.Symmetric) k).toCryptoEngine();
            default:
                throw new UnsupportedOperationException();
        }
    }

    public static <T extends Key> T of(AKey key) {
        switch (key.getProviderName().toLowerCase()) {
            case "sodium":
                switch (key.getKeyType()) {
                    case SYMMETRIC:
                        return RU.cast(new SodiumChacha20Poly1305(key.getData()));
                    case ASYMMETRIC_PRIVATE:
                        return RU.cast(new SodiumCurvePrivate(key.getData()));
                    case ASYMMETRIC_PUBLIC:
                        return RU.cast(new SodiumCurvePublic(key.getData()));
                    case SIGN_PRIVATE:
                        return RU.cast(new SodiumSignPrivate(key.getData()));
                    case SIGN_PUBLIC:
                        return RU.cast(new SodiumSignPublic(key.getData()));
                    default:
                        throw new UnsupportedOperationException();
                }
            case "hydrogen":
                switch (key.getKeyType()) {
                    case SYMMETRIC:
                        return RU.cast(new HydrogenSecretBox(key.getData()));
                    case ASYMMETRIC_PRIVATE:
                        return RU.cast(new HydrogenCurvePrivate(key.getData()));
                    case ASYMMETRIC_PUBLIC:
                        return RU.cast(new HydrogenCurvePublic(key.getData()));
                    case SIGN_PRIVATE:
                        return RU.cast(new HydrogenSignPrivate(key.getData()));
                    case SIGN_PUBLIC:
                        return RU.cast(new HydrogenSignPublic(key.getData()));
                    default:
                        throw new UnsupportedOperationException();
                }
        }
        throw new UnsupportedOperationException();
    }

}
