package io.aether.crypto;

/**
 * Interface for an object that performs encryption and decryption
 * with a stateful component, such as a nonce.
 * Implementations of this interface will hold keys and other
 * internal state necessary for the cryptographic process.
 */
public interface CryptoEngine extends CryptoProviderUnit {
    /**
     * Encrypts the provided data.
     *
     * @param data The data to be encrypted.
     * @return The encrypted data.
     */
    byte[] encrypt(byte[] data);

    /**
     * Decrypts the provided data.
     *
     * @param data The data to be decrypted.
     * @return The decrypted data.
     */
    byte[] decrypt(byte[] data);

    static CryptoEngine of(CryptoEngine encoder, CryptoEngine decoder) {
        return new CryptoEngine(){
            @Override
            public byte[] encrypt(byte[] data) {
                return encoder.encrypt(data);
            }

            @Override
            public byte[] decrypt(byte[] data) {
                return decoder.decrypt(data);
            }

            @Override
            public String getProviderName() {
                return encoder.getProviderName();
            }

            @Override
            public CryptoProvider getCryptoProvider() {
                return encoder.getCryptoProvider();
            }
        };
    }
}