package io.aether.crypto;

import io.aether.logger.Log;
import io.aether.utils.HexUtils;

/**
 * Interface defining the contract for a cryptographic provider (e.g., SODIUM, HYDROGEN).
 * It provides methods for key creation, key pairing, and cryptographic operations.
 */
public interface CryptoProvider {

    /**
     * Gets the unique name of the cryptographic library implemented by this provider.
     *
     * @return The name of the crypto library (e.g., "SODIUM", "HYDROGEN").
     */
    String getCryptoLibName();

    /**
     * Generates a new pair of asymmetric keys (public and private).
     *
     * @return A PairAsymKeys object containing the generated keys.
     */
    PairAsymKeys createAsymmetricKeys();

    /**
     * Generates a new random symmetric key.
     *
     * @return A new AKey.Symmetric instance.
     */
    AKey.Symmetric createSymmetricKey();

    /**
     * Generates a new pair of signing keys (public and private).
     *
     * @return A PairSignKeys object containing the generated keys.
     */
    PairSignKeys createSignKeys();

    /**
     * Parses a signed key string into a SignedKey object.
     * The method is flexible, supporting both ':' and '|' as the separator
     * between KEY_DATA and SIGN_DATA. The key provider and sign provider are
     * assumed to be the current provider.
     * <p>
     * Expected formats (4 parts):
     * 1. PROVIDER:KEY_TYPE:KEY_DATA:SIGN_DATA
     * 2. PROVIDER:KEY_TYPE:KEY_DATA|SIGN_DATA
     *
     * @param data The signed key string.
     * @return A new SignedKey instance.
     * @throws IllegalArgumentException if the format is invalid or provider mismatch.
     */
    default SignedKey createSignedKey(String data) {
        // 1. Unify separator: replace '|' with ':' if present, to ensure 4 parts can be split by ':'
        String unifiedData = data.replace('|', ':');

        var parts = unifiedData.split(":");
        String currentProviderName = getCryptoLibName();

        // 2. Check format (must be 4 parts)
        if (parts.length != 4) {
            throw new IllegalArgumentException("Invalid signed key string format. Expected 4 parts: PROVIDER:TYPE:KEY_DATA:SIGN_DATA.");
        }

        // 3. Check provider mismatch
        // Require the key's provider (parts[0]) to match the current provider.
        if (!parts[0].equals(currentProviderName)) {
            throw new IllegalArgumentException("Key provider '" + parts[0] + "' must match the current provider: " + currentProviderName);
        }

        // 4. Parse Key (Part 0, 1, 2)
        // Reconstruct key string as "PROVIDER:TYPE:DATA" for createKey(String).
        String keyString = parts[0] + ":" + parts[1] + ":" + parts[2];
        AKey key = createKey(keyString);

        // 5. Parse Sign (Part 0, 3)
        // Reconstruct sign string as "PROVIDER:DATA" for createSign(String).
        // Sign provider is parts[0], which must match the current provider.
        String signString = parts[0] + ":" + parts[3];
        Sign sign = createSign(signString);

        // 6. Create SignedKey
        return new SignedKey(key, sign);
    }

    /**
     * Creates a pair of signing keys from raw byte arrays.
     *
     * @param publicKey  The byte array of the public key.
     * @param privateKey The byte array of the private key.
     * @return A new PairSignKeys instance.
     */
    default PairSignKeys createSignKeys(byte[] publicKey, byte[] privateKey) {
        return new PairSignKeys(createKey(KeyType.SIGN_PUBLIC, publicKey), createKey(KeyType.SIGN_PRIVATE, privateKey));
    }

    /**
     * Creates a pair of signing keys from hexadecimal strings.
     *
     * @param publicKey  The hexadecimal string of the public key.
     * @param privateKey The hexadecimal string of the private key.
     * @return A new PairSignKeys instance.
     */
    default PairSignKeys createSignKeys(String publicKey, String privateKey) {
        return createSignKeys(HexUtils.hexToBytes(publicKey), HexUtils.hexToBytes(privateKey));
    }

    //Для проверки подписи и создания подписи
    Signer createSigner(PairSignKeys keys);

    /**
     * Creates a Signer object for signature verification and creation from explicit keys.
     *
     * @param publicKey  The public signing key.
     * @param privateKey The private signing key.
     * @return A new Signer instance.
     */
    Signer createSigner(AKey.SignPublic publicKey, AKey.SignPrivate privateKey);

    /**
     * Creates a Signer object solely for signature verification.
     *
     * @param publicKey The public signing key.
     * @return A new Signer instance configured only for verification.
     */
    Signer createSigner(AKey.SignPublic publicKey);

    /**
     * Creates a CryptoEngine for symmetric encryption and decryption.
     * This engine will handle its own internal state, such as a nonce.
     *
     * @param key The AKey.SymmetricKey to be used.
     * @return A new CryptoEngine instance.
     */
    CryptoEngine createSymmetricEngine(AKey.Symmetric key);

    /**
     * Creates a CryptoEngine for asymmetric encryption.
     *
     * @param key The AKey.AsymmetricPublicKey to be used.
     * @return A new CryptoEngine instance.
     */
    CryptoEngine createAsymmetricEngine(AKey.AsymmetricPublic key);

    /**
     * Creates a CryptoEngine for asymmetric decryption.
     *
     * @param privateKey The AKey.AsymmetricPrivateKey to be used.
     * @param publicKey  The AKey.AsymmetricPublicKey of the sender.
     * @return A new CryptoEngine instance.
     */
    CryptoEngine createAsymmetricEngine(AKey.AsymmetricPrivate privateKey, AKey.AsymmetricPublic publicKey);

    /**
     * Creates a CryptoEngine for asymmetric decryption from a key pair.
     *
     * @param keys The pair of asymmetric keys.
     * @return A new CryptoEngine instance.
     */
    CryptoEngine createAsymmetricEngine(PairAsymKeys keys);

    /**
     * Creates a concrete AKey implementation based on the specified key type and data.
     *
     * @param keyType The type of key to create.
     * @param data    The byte array of the key.
     * @return A new AKey instance.
     */
    <T extends AKey> T createKey(KeyType keyType, byte[] data);

    /**
     * Creates a key from its textual representation, which is generated by AKey.toString.
     * The expected format is "PROVIDER:KEY_TYPE:KEY_DATA".
     *
     * @param data The textual key representation.
     * @return A new AKey instance.
     */
    <T extends AKey> T createKey(String data);

    /**
     * Creates a SignedKey from key type, raw key bytes, and raw signature bytes.
     *
     * @param keyType The type of key.
     * @param key     The raw key bytes.
     * @param sign    The raw signature bytes.
     * @return A new SignedKey instance.
     */
    default SignedKey createSignedKey(KeyType keyType, byte[] key, byte[] sign) {
        var k = createKey(keyType, key);
        return k.toSignedKey(createSign(sign));
    }

    /**
     * Creates a Sign object from its textual representation, which is generated by Sign.toString.
     * The expected format is "PROVIDER:SIGN_DATA".
     *
     * @param data The textual sign representation.
     * @return A new Sign instance.
     */
    Sign createSign(String data);

    /**
     * Creates a Sign object from raw byte data.
     *
     * @param data The raw signature bytes.
     * @return A new Sign instance.
     */
    Sign createSign(byte[] data);

    /**
     * Derives a pair of symmetric keys (for client-to-server and server-to-client communication)
     * using Key Derivation Function (KDF) from a master key and session/key identifiers.
     *
     * @param masterKey The master symmetric key.
     * @param serverId  The server identifier (32-bit).
     * @param keyNumber The key number/index (32-bit).
     * @return A PairSymmetricKeys object containing client and server keys.
     */
    PairSymKeys deriveSymmetricKeys(AKey.Symmetric masterKey, int serverId, int keyNumber);

    /**
     * Creates a symmetric key for server-side usage from a master key and session ID.
     * Default implementation returns the master key itself.
     *
     * @param masterKey The master symmetric key.
     * @param sid       The session ID.
     * @return The symmetric key for the server.
     */
    default PairSymKeys createKeyForServer(AKey.Symmetric masterKey, int sid) {
        var res = deriveSymmetricKeys(masterKey, sid, 0);
        Log.debug("Generate pair sym keys: client:$keyClient <-> server: $keyServer",
                "keyClient", res.serverToClient,
                "keyServer", res.clientToServer,
                "masterKey", masterKey,
                "sid", sid
                );
        return res;
//        return new PairSymKeys(masterKey, masterKey);
    }

    /**
     * Creates a public signing key from raw byte data.
     *
     * @param data The raw public key bytes.
     * @return A new AKey.SignPublic instance.
     */
    AKey.SignPublic createSignPublicKey(byte[] data);

    /**
     * Creates a private signing key from raw byte data.
     *
     * @param data The raw private key bytes.
     * @return A new AKey.SignPrivate instance.
     */
    AKey.SignPrivate createSignPrivateKey(byte[] data);

    /**
     * Creates a pair of signing keys from a textual representation.
     * The method handles two formats:
     * 1. FULL_PUB_KEY_STRING|FULL_PRIV_KEY_STRING (for rootSigners.key parsing).
     * 2. PROVIDER:PUBLIC_KEY_HEX[:PRIVATE_KEY_HEX] (2 or 3 sections).
     * 3. PUBLIC_KEY_HEX[:PRIVATE_KEY_HEX] (Legacy format, 1 or 2 sections).
     *
     * @param text The textual key pair representation.
     * @return A new PairSignKeys instance.
     */
    default PairSignKeys createSignKeys(String text) {
        // 1. Handle the "FULL_KEY_STRING|FULL_KEY_STRING" format (for rootSigners.key)
        if (text.contains("|")) {
            var parts = text.split("\\|");
            if (parts.length != 2) {
                throw new IllegalArgumentException("Invalid PairSignKeys format with '|' separator. Expected: PUBLIC_KEY_STRING|PRIVATE_KEY_STRING.");
            }

            // Public and Private key strings already contain PROVIDER:KEY_TYPE:HEX_DATA
            AKey.SignPublic publicKey = createKey(parts[0]); // Uses createKey(String)
            AKey.SignPrivate privateKey = createKey(parts[1]); // Uses createKey(String)

            return new PairSignKeys(publicKey, privateKey);
        }

        // 2. Handle formats separated by ":"
        var tt = text.split(":");
        String currentProviderName = getCryptoLibName();

        // Check for the user's requested 2 or 3 section format: PROVIDER:PUBLIC_HEX[:PRIVATE_HEX]
        if (tt.length >= 2 && tt.length <= 3 && tt[0].equals(currentProviderName)) {
            // New format: PROVIDER:PUBLIC_HEX[:PRIVATE_HEX]
            // tt[0] = Provider Name (already checked)
            // tt[1] = Public Key (HEX)
            // tt[2] = Private Key (HEX, optional)

            var keyPublic = createSignPublicKey(HexUtils.hexToBytes(tt[1]));

            if (tt.length == 3) {
                var keyPrivate = createSignPrivateKey(HexUtils.hexToBytes(tt[2]));
                return new PairSignKeys(keyPublic, keyPrivate);
            } else { // tt.length == 2 (Public key only)
                return new PairSignKeys(keyPublic, null);
            }
        }

        // 3. Handle the legacy raw hex format: PUBLIC_KEY_HEX[:PRIVATE_KEY_HEX] (1 or 2 sections)
        // This is the original logic for raw hex parts (provider name stripped by caller, e.g., PairSignKeysUtils.of)
        if (tt.length < 1 || tt.length > 2) {
            throw new IllegalArgumentException("Invalid raw hex key pair format. Expected 1 or 2 parts, found " + tt.length);
        }

        var keyPublic = createSignPublicKey(HexUtils.hexToBytes(tt[0]));
        if (tt.length > 1) { // 2 parts
            var keyPrivate = createSignPrivateKey(HexUtils.hexToBytes(tt[1]));
            return new PairSignKeys(keyPublic, keyPrivate);
        } else { // 1 part
            return new PairSignKeys(keyPublic, null);
        }
    }

    /**
     * Creates a symmetric key from raw byte data.
     *
     * @param bytes The raw symmetric key bytes.
     * @return A new AKey.Symmetric instance.
     */
    AKey.Symmetric createSymmetricKey(byte[] bytes);

    /**
     * Creates a Signer object using newly generated signing keys.
     *
     * @return A new Signer instance.
     */
    default Signer createSigner() {
        return createSigner(createSignKeys());
    }

    /**
     * Creates a key from its type and public key's hexadecimal string.
     *
     * @param keyType   The type of key to create.
     * @param publicKey The hexadecimal string of the public key data.
     * @return A new AKey instance.
     */
    default <T extends AKey> T createKey(KeyType keyType, String publicKey) {
        return createKey(keyType, HexUtils.hexToBytes(publicKey));
    }
}