package io.aether.crypto;

import io.aether.utils.HexUtils;

public class PairAsymKeysSigned implements PairKeys {
    public final SignedKey publicKey;
    public final AKey.AsymmetricPrivate privateKey;

    public PairAsymKeysSigned(SignedKey publicKey, AKey.AsymmetricPrivate privateKey) {
        if (!(publicKey.key instanceof AKey.AsymmetricPublic))
            throw new IllegalArgumentException("Public key is bad: " + publicKey);
        this.publicKey = publicKey;
        this.privateKey = privateKey;
    }

    @Override
    public String getProviderName() {
        return publicKey.getProviderName();
    }

    @Override
    public CryptoProvider getCryptoProvider() {
        return publicKey.getCryptoProvider();
    }

    public AKey.AsymmetricPrivate getPrivateKey() {
        return privateKey;
    }

    public SignedKey getPublicKeySigned() {
        return publicKey;
    }

    public AKey.AsymmetricPublic getPublicKey() {
        return (AKey.AsymmetricPublic) publicKey.key;
    }


    public static PairAsymKeysSigned of(String k) {
        try {
            var kk = k.split(":");
            if (kk.length < 3) {
                throw new IllegalArgumentException("Invalid key string format. Expected at least 3 parts (Provider:PublicKey:Signature) but got " + kk.length);
            }

            var cpName = kk[0];
            var cp = CryptoProviderFactory.getProvider(cpName);

            // 1. Public Key is kk[1] (32 bytes)
            byte[] publicKeyData = HexUtils.hexToBytes(kk[1]);

            // 2. Signature is kk[2] (64 bytes)
            byte[] signatureData = HexUtils.hexToBytes(kk[2]);

            // Create SignedKey (Public Key + Signature)
            // Note: The createSignedKey method must handle the validation of public key length (32) and signature length (64).
            // Since the current implementation expects SIGNATURE length to be 64, this should now pass.
            var signedKey = cp.createSignedKey(KeyType.ASYMMETRIC_PUBLIC, publicKeyData, signatureData);

            // 3. Private Key is kk[3] (32 bytes) - Optional
            AKey.AsymmetricPrivate privateKey = null;
            if (kk.length > 3) {
                byte[] privateKeyData = HexUtils.hexToBytes(kk[3]);
                // Note: The createKey method for ASYMMETRIC_PRIVATE must handle the validation of private key length (32).
                privateKey = (AKey.AsymmetricPrivate) cp.createKey(KeyType.ASYMMETRIC_PRIVATE, privateKeyData);
            }

            return new PairAsymKeysSigned(signedKey, privateKey);

        } catch (IllegalArgumentException e) {
            // Re-throw IllegalArgumentException directly to show clear error messages (like those from SodiumSign.<init>)
            throw new IllegalStateException("Parse exception for: " + k + " (Invalid data length or format)", e);
        } catch (Exception e) {
            throw new IllegalStateException("Parse exception for: " + k, e);
        }
    }
}