package io.aether.utils.consoleCanonical;

import io.aether.logger.Log;
import io.aether.logger.LogFilter;
import io.aether.net.fastMeta.FastFutureContext;
import io.aether.net.fastMeta.FastMeta;
import io.aether.utils.AString;
import io.aether.utils.CTypeI;
import io.aether.utils.Mods;
import io.aether.utils.RU;
import io.aether.utils.dataio.DataInOut;
import io.aether.utils.flow.Flow;
import io.aether.utils.futures.AFuture;
import io.aether.utils.futures.ARFuture;
import io.aether.utils.interfaces.ABiFunction;
import io.aether.utils.interfaces.AFunction;
import io.aether.utils.slots.EventConsumer;
import it.unimi.dsi.fastutil.objects.ObjectArrayList;
import it.unimi.dsi.fastutil.objects.ObjectOpenHashSet;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.lang.annotation.*;
import java.lang.reflect.Array;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

public class ConsoleMgrCanonical {
    final Map<String, Boolean> flags = new ConcurrentHashMap<>();
    final Map<String, String> vals = new ConcurrentHashMap<>();
    private final Map<CTypeI<?>, AFunction<String, Object>> converters = new ConcurrentHashMap<>();
    private final Map<String, Map<CTypeI<?>, ABiFunction<ResCtx, Object, byte[]>>> resultConverters = new ConcurrentHashMap<>();
    public String footer;
    List<String> args;
    List<String> argsOrigin;
    boolean helpMode;
    InputStream stdin = System.in;
    Object rootApi;

    // Добавлено: Future для возврата результата выполнения
    private ARFuture<Object> executionResultFuture=ARFuture.make();

    public ConsoleMgrCanonical(String... args) {
        this.args = new ObjectArrayList<>(args);
        if (this.args.isEmpty()) {
            this.args.add("help");
        }
        argsOrigin = new ObjectArrayList<>(args);
        regConverter(CTypeI.of(UUID.class), UUID::fromString);
        regConverter(CTypeI.of(File.class), File::new);
        regConverter(CTypeI.of(int.class), Integer::parseInt);
        regConverter(CTypeI.of(long.class), Long::parseLong);
        regConverter(CTypeI.of(short.class), Short::parseShort);
        regConverter(CTypeI.of(byte.class), Byte::parseByte);
        regConverter(CTypeI.of(float.class), Float::parseFloat);
        regConverter(CTypeI.of(double.class), Double::parseDouble);
        regConverter(CTypeI.of(URI.class), URI::create);
        regConverter(CTypeI.BOOLEAN, Boolean::parseBoolean);
        regResultConverter("bin", CTypeI.of(UUID.class), v -> {
            if (v == null) return null;
            var out = new DataInOut();
            FastMeta.META_UUID.serialize(FastFutureContext.STUB, v, out);
            return out.toArray();
        });
        regResultConverter("json", CTypeI.of(UUID.class), v -> {
            if (v == null) return null;
            return ("\"" + v + "\"").getBytes(StandardCharsets.UTF_8);
        });
        regResultConverter("json", CTypeI.STRING, v -> {
            if (v == null) return null;
            return ("\"" + v + "\"").getBytes(StandardCharsets.UTF_8);
        });
    }

    public <T> void regResultConverter(String format, CTypeI<T> type, AFunction<T, byte[]> f) {
        regResultConverterCtx(format, type, (c, v) -> f.apply(RU.cast(v)));
    }

    public <T> void regResultConverterCtx(String format, CTypeI<T> type, ABiFunction<ResCtx, T, byte[]> f) {
        resultConverters.computeIfAbsent(format, k -> new ConcurrentHashMap<>()).put(type, RU.cast(f));
    }

    String getByKey(String key) {
        key = key.toLowerCase();
        if (vals.containsKey(key) && vals.get(key) == null) {
            return null;
        }
        return vals.computeIfAbsent(key, kk -> {
            var key2 = "--" + kk.toLowerCase();
            var key3 = "--" + kk.toLowerCase() + '=';
            var it = args.iterator();
            while (it.hasNext()) {
                var k = it.next().toLowerCase();
                if (k.equals(key2)) {
                    if (!it.hasNext()) {
                        throw new UnsupportedOperationException("Value not set for key: " + key2);
                    }
                    it.remove();
                    var r = it.next();
                    it.remove();
                    return r;
                } else if (k.startsWith(key3)) {
                    it.remove();
                    return k.split("=")[1];
                }
            }
            return null;
        });
    }

    boolean getFlag(String flag) {
        return flags.computeIfAbsent(flag.toLowerCase(), key -> {
            var key2 = "--" + key.toLowerCase();
            var it = args.iterator();
            while (it.hasNext()) {
                var k = it.next().toLowerCase();
                if (k.equals(key2)) {
                    it.remove();
                    return true;
                }
            }
            return false;
        });
    }

    public void regConverter(CTypeI<?> type, AFunction<String, Object> f) {
        converters.put(type, f);
    }

    public Object convert(CTypeI<?> type, String s) {
        if (type.instanceOf(String.class)) return s;
        var f = converters.get(type);
        if (f == null) {
            if (type.isEnum()) {
                f = v -> Enum.valueOf(RU.cast(type.getRaw()), v);
                converters.put(type, f);
                return f;
            } else if (type.instanceOf(Collection.class)) {
                var c = type.getComponent();
                f = v -> {
                    var vv = v.split("(?<=[^\\\\]),");
                    Collection<?> res;
                    if (type.instanceOf(List.class)) {
                        res = new ObjectArrayList<>();
                    } else if (type.instanceOf(Set.class)) {
                        res = new ObjectOpenHashSet<>();
                    } else {
                        throw new UnsupportedOperationException();
                    }
                    for (var e : vv) {
                        res.add(RU.cast(convert(c, e)));
                    }
                    return RU.cast(res);
                };
                converters.put(type, f);
                return f;
            } else if (type.isArray()) {
                var c = type.getComponent();
                f = v -> {
                    var vv = v.split(",");
                    Object res = Array.newInstance(c.getRaw2(), vv.length);
                    int i = 0;
                    for (var e : vv) {
                        Array.set(res, i++, convert(c, e));
                    }
                    return RU.cast(res);
                };
                converters.put(type, f);
                return f;
            }
            throw new IllegalStateException("converter not found for " + type);
        }
        return f.apply(s);
    }

    String convertMethodName(String s) {
        StringBuilder sb = new StringBuilder();
        boolean flag = true;
        for (int i = 0; i < s.length(); i++) {
            var ch = s.charAt(i);
            if (flag) {
                if (Character.isUpperCase(ch) || Character.isDigit(ch)) {
                    sb.append(Character.toLowerCase(ch));
                } else {
                    flag = false;
                    sb.append(ch);
                }
            } else {
                if (Character.isUpperCase(ch) || Character.isDigit(ch)) {
                    sb.append("-");
                    sb.append(Character.toLowerCase(ch));
                    flag = true;
                } else {
                    sb.append(ch);
                }
            }
        }
        return sb.toString();
    }

    String buildHelpForMethod(CTypeI<?> owner, Method m) {
        var s = AString.of();
        s.add("Method: ");
        if (m.isAnnotationPresent(Alias.class)) {
            s.add(m.getAnnotation(Alias.class).value()).add(", ");
        }
        s.add(convertMethodName(m.getName())).add("\n");
        var d = m.getAnnotation(Doc.class);
        if (d != null) {
            s.add(d.value()).add("\n");
        }

        // --- ОБНОВЛЕНО: Чтение повторяющихся аннотаций @Example ---
        Example[] examples = m.getAnnotationsByType(Example.class);
        for (Example ex : examples) {
            s.styleForeground(AString.Style.BRIGHT, 150, 255, 150)
                    .add(buildExampleString(ex.value())).styleClear();
        }
        // ---------------------------------------------------------

        Parameter[] parameters = m.getParameters();
        var pt = owner.resolveTypeArgs(m);
        s.add("\n");
        boolean withTypes = getFlag("with-types");
        for (int i = 0; i < parameters.length; i++) {
            var p = parameters[i];
            var len = s.length();
            s.addSpace(4);
            var option = p.getAnnotation(Optional.class);
            if (p.isAnnotationPresent(StdIn.class)) {
                s.add("#");
            }
            if (option != null) {
                s.style(AString.Style.DIM).add("[");
            } else {
                s.add("<");
            }
            var alias = p.getAnnotation(Alias.class);
            if (alias != null) {
                if (option != null) s.add("--");
                s.add(alias.value());
                if (option != null) s.add("|");
            }
            if (option != null) s.add("--");
            if (option != null || alias == null) s.add(convertMethodName(p.getName()));
            if (withTypes) {
                s.style(AString.Style.DIM);
                s.add(":").add(pt[i]);
                s.styleClear();
            }
            if (option != null) {
                if (!option.value().isEmpty()) {
                    s.color(AString.Color.GREEN).add(" (").add(option.value()).add(")").styleClear();
                }
                s.add("]").styleClear();
            } else {
                s.add(">");
            }
            var lenD = s.calcVisibleSymbols(len);
            var dd = p.getAnnotation(Doc.class);
            if (dd != null) {
                s.addWithAlign(70, lenD, 30, dd.value());
            }
            // --- ОБНОВЛЕНО: Чтение повторяющихся аннотаций @Example на параметре ---
            Example[] paramExamples = p.getAnnotationsByType(Example.class);
            if (paramExamples.length > 0) {
                for (Example example : paramExamples) {
                    s.styleForeground(AString.Style.BRIGHT, 150, 255, 150)
                            .addWithAlign(70, dd == null ? lenD : 0, 30, buildExampleString(example.value())).styleClear();
                }
            }
            // ---------------------------------------------------------------------

            if (dd == null && paramExamples.length == 0) {
                s.add("\n");
            }
            if (pt[i].isEnum()) {
                s.style(AString.Style.ITALIC).addSpace(4).add("Possible values: ").add(
                        Flow.flow(pt[i].getEnumValues()).map(v -> AString.of().style(AString.Style.BRIGHT).style(AString.Style.ITALIC).add(v).styleClear()).join(", ")).styleClear().add("\n");
            }
            if (p.isAnnotationPresent(StdIn.class)) {
                s.style(AString.Style.ITALIC).addSpace(4).add("The argument can accept data from standard input\n").styleClear();
            }
            s.add("\n");
        }
        addRequiredHelp(s);
        return s.toString();
    }

    // Изменено: возвращает ARFuture<Object> вместо AFuture
    public ARFuture<Object> execute(CTypeI<?> type, Object api, Method m) {
        executionResultFuture = ARFuture.make();

        var rt = type.resolveReturnType(m);
        String fileOutFormat = getByKey("file-out-format");
        String fileOut = getByKey("file-out");
        String consoleFormat = getByKey("console");
        String fileAppend = getByKey("file-out-append");
        if (helpMode) {
            if (m.isAnnotationPresent(Api.class)) {
                return execute(rt, null);
            } else {
                result(CTypeI.STRING, buildHelpForMethod(type, m), fileOut, fileOutFormat, consoleFormat, fileAppend);
                executionResultFuture.tryDone(null);
            }
            return executionResultFuture;
        }
        Object[] aa = new Object[m.getParameterCount()];
        var pt = type.resolveTypeArgs(m);
        int ia = 0;
        while (ia < args.size()) {
            var a = args.get(ia);
            if (a.startsWith("--")) {
                if (a.matches("--\\w+=.*")) {
                    ia++;
                } else {
                    ia += 2;
                }
            } else {
                break;
            }
        }

        for (int i = 0; i < aa.length; i++) {
            var p = m.getParameters()[i];
            var alias = p.getAnnotation(Alias.class);
            String pn1 = convertMethodName(p.getName());
            String pn2 = alias == null ? null : alias.value();
            var optional = p.getAnnotation(Optional.class);
            String val;
            if (optional == null) {
                if (getFlag("stdin") && p.isAnnotationPresent(StdIn.class)) {
                    try {
                        val = new String(stdin.readAllBytes());
                    } catch (IOException e) {
                        RU.error(e);
                        executionResultFuture.error(e);
                        return executionResultFuture;
                    }
                } else {
                    val = args.remove(ia);
                }
                if (val == null) {
                    var ex = new IllegalStateException("Value is not specified for: " + convertMethodName(p.getName()));
                    executionResultFuture.error(ex);
                    return executionResultFuture;
                }
                aa[i] = convert(pt[i], val);
            } else {
                if (pt[i].getRaw2() == boolean.class || pt[i].getRaw2() == Boolean.class) {
                    val = String.valueOf(getFlag(pn1));
                } else {
                    val = getByKey(pn1);
                }
                if (val == null) {
                    if (pn2 != null) {
                        if (pt[i].getRaw2() == boolean.class || pt[i].getRaw2() == Boolean.class) {
                            val = AString.of().add(getFlag(pn2)).toString();
                        } else {
                            val = getByKey(pn2);
                        }
                    }
                }
                if (val == null) {
                    if (!optional.value().isEmpty()) {
                        val = optional.value();
                    }
                }
                if (val == null) {
                    if (p.isAnnotationPresent(StdIn.class)) {
                        try {
                            val = new String(stdin.readAllBytes());
                        } catch (IOException e) {
                            RU.error(e);
                            executionResultFuture.error(e);
                            return executionResultFuture;
                        }
                    }
                }
                if (Objects.equals(pn1, "file-out") || Objects.equals(pn2, "file-out")) {
                    fileOut = val;
                }
                if (Objects.equals(pn1, "file-out-format") || Objects.equals(pn2, "file-out-format")) {
                    fileOutFormat = val;
                }
                if (Objects.equals(pn1, "console") || Objects.equals(pn2, "console")) {
                    consoleFormat = val;
                }
                if (Objects.equals(pn1, "file-out-append") || Objects.equals(pn2, "file-out-append")) {
                    fileAppend = val;
                }
                if (val != null) {
                    aa[i] = convert(pt[i], val);
                }
            }
        }
        Object res;
        String fileOutFinal = fileOut;
        String fileOutFormatFinal = fileOutFormat;
        String consoleFormatFinal = consoleFormat;
        String fileAppendFinal = fileAppend;
        try {
            res = m.invoke(api, aa);
        } catch (InvocationTargetException e) {
            e.getCause().printStackTrace();
            executionResultFuture.error(e.getCause());
            return executionResultFuture;
        } catch (Exception e) {
            executionResultFuture.error(e);
            return executionResultFuture;
        }
        if (res != null) {
            var rt2 = CTypeI.of(res.getClass());
            if (rt2.instanceOf(rt)) {
                rt = rt2;
            }
        }
        if (rt.instanceOf(void.class)) {
            executionResultFuture.tryDone(null);
        } else if (rt.instanceOf(AFuture.class)) {
            RU.<AFuture>cast(res).to(() -> {
                executionResultFuture.tryDone(null);
            }).onError(executionResultFuture::error);
        } else if (rt.instanceOf(EventConsumer.class)) {
            CTypeI<?> finalRt = rt;
            RU.<EventConsumer<Object>>cast(res).add(v -> {
                var c = finalRt.getComponent();
                if (c == null) c = CTypeI.of(Object.class);
                if (v != null) {
                    var rt2 = CTypeI.of(v.getClass());
                    if (rt2.instanceOf(c)) {
                        c = RU.cast(rt2);
                    }
                }
                if (m.isAnnotationPresent(Api.class)) {
                    execute(c, v).to(executionResultFuture);
                } else {
                    result(c, v, fileOutFinal, fileOutFormatFinal, consoleFormatFinal, fileAppendFinal);
                    executionResultFuture.tryDone(v);
                }
            });
        } else if (rt.instanceOf(ARFuture.class)) {
            CTypeI<?> finalRt = rt;
            RU.<ARFuture<Object>>cast(res).to(v -> {
                var c = finalRt.getComponent();
                if (c == null) c = CTypeI.of(Object.class);
                if (v != null) {
                    var rt2 = CTypeI.of(v.getClass());
                    if (rt2.instanceOf(c)) {
                        c = RU.cast(rt2);
                    }
                }
                if (m.isAnnotationPresent(Api.class)) {
                    execute(c, v).to(executionResultFuture);
                } else {
                    result(c, v, fileOutFinal, fileOutFormatFinal, consoleFormatFinal, fileAppendFinal);
                    executionResultFuture.tryDone(v);
                }
            }).onError(executionResultFuture::error);
        } else if (m.isAnnotationPresent(Api.class)) {
            execute(rt, res).to(executionResultFuture);
        } else {
            result(rt, res, fileOut, fileOutFormat, consoleFormat, fileAppend);
            executionResultFuture.tryDone(res);
        }
        return executionResultFuture;
    }

    public void result(CTypeI<?> type, Object val, String fileOutStr, String fileOutFormatStr, String consoleFormatStr, String fileAppend) {
        if (fileOutFormatStr == null) {
            fileOutFormatStr = "human";
        } else {
            fileOutFormatStr = fileOutFormatStr.toLowerCase();
        }
        ResCtx ctx = new ResCtx();
        if (fileAppend != null) {
            ctx.append = Boolean.parseBoolean(fileAppend);
        }
        if (fileOutStr != null) {
            File f = new File(fileOutStr);
            var fomatMap = resultConverters.get(fileOutFormatStr);
            byte[] data;
            if (fomatMap == null || fomatMap.get(type) == null) {
                if (fileOutFormatStr.equals("human")) {
                    data = AString.of().add(val).getBytes();
                } else {
                    throw new UnsupportedOperationException("unsupported format [" + fileOutFormatStr + "] for: " + val);
                }
            } else {
                var ff = fomatMap.get(type);
                data = ff.apply(ctx, val);
                if (ctx.fileName != null) {
                    f = new File(fileOutStr + "-" + ctx.fileName);
                }
            }
            try {
                Log.debug("Write file $fileName", "fileName", f);
                Files.write(Paths.get(f.toURI()), data, f.exists() ? (getFlag("file-out-append") ? StandardOpenOption.APPEND : StandardOpenOption.TRUNCATE_EXISTING) : StandardOpenOption.CREATE);
            } catch (Exception e) {
                RU.error(e);
            }
        }
        if (consoleFormatStr == null) {
            consoleFormatStr = "human";
        }
        var fomatMap = resultConverters.get(consoleFormatStr);
        if (fomatMap == null || fomatMap.get(type) == null) {
            if (consoleFormatStr.equals("human")) {
                System.out.println(AString.of().add(val));
                return;
            } else {
                throw new UnsupportedOperationException("unsupported format [" + fileOutFormatStr + "] for: " + val);
            }
        }
        var data = fomatMap.get(type).apply(ctx, val);
        try {
            System.out.write(data);
            System.out.flush();
        } catch (IOException e) {
            RU.error(e);
        }
    }

    public String getAppName() {
        return "app";
    }

    // Изменено: возвращает ARFuture<Object> вместо AFuture
    public ARFuture<Object> execute(Object api) {
        if (getFlag("loggerConsole")) {
            Log.printConsoleColored(new LogFilter());
        }
        if (rootApi == null) {
            rootApi = api;
        }
        return execute(CTypeI.of(api.getClass()), api);
    }

    private Flow<Method> getMethods(CTypeI<?> type) {
        return type.getAllMethods().filter(Mods::isPublic).filter(m -> m.getDeclaringClass() != Object.class);
    }

    private void printHelpApi(CTypeI<?> type) {
        var s = AString.of();
        s.style(AString.Style.BRIGHT, AString.Color.ORANGE);
        s.add("List methods ");
        s.add(type.getRawSimpleName());
        s.add("\n");
        var d = type.getAnnotation(Doc.class);
        if (d != null) {
            s.add(d.value()).add("\n\n");
        }
        s.styleClear();
        boolean withTypes = getFlag("with-types");
        for (var m : getMethods(type)) {
            int len = s.length();
            s.add("<");
            var cmdName = convertMethodName(m.getName());
            s.add(cmdName);
            var alias = m.getAnnotation(Alias.class);
            if (alias != null) {
                s.add("|").add(alias.value());
            }
            s.add(">");
            AString s1 = AString.of();
            AString s2 = AString.of();
            AString ss;
            Parameter[] parameters = m.getParameters();
            var pt = type.resolveTypeArgs(m);
            for (int i = 0; i < parameters.length; i++) {
                var p = parameters[i];
                var option = p.getAnnotation(Optional.class);
                if (option == null) {
                    ss = s1;
                } else {
                    ss = s2;
                }
                ss.add(" ");
                var aliasParam = p.getAnnotation(Alias.class);
                if (option == null) {
                    ss.add("<");
                    if (aliasParam == null) {
                        ss.add(convertMethodName(p.getName()));
                    } else {
                        ss.add(aliasParam.value());
                    }
                    if (withTypes) {
                        ss.style(AString.Style.DIM);
                        ss.add(":");
                        pt[i].getRawSimpleNameWithParameters(ss);
                        ss.styleClear();
                    }
                    ss.add(">");
                } else {
                    ss.style(AString.Style.DIM);
                    ss.add("[");
                    ss.add("--");
                    ss.add(convertMethodName(p.getName()));
                    if (aliasParam != null) {
                        ss.add("|--");
                        ss.add(aliasParam.value());
                    }
                    if (!option.value().isBlank()) {
                        ss.color(AString.Color.GREEN).add(" (").add(option.value()).add(")").styleClear().style(AString.Style.DIM);
                    }
                    if (withTypes) {
                        ss.add(":");
                        pt[i].getRawSimpleNameWithParameters(ss);
                    }
                    ss.add("]");
                    ss.style(AString.Style.CLEAR);
                }
            }
            s.add(s1).add(s2);
            var doc = m.getAnnotation(Doc.class);
            var lend = s.calcVisibleSymbols(len);
            if (doc != null) {
                s.styleForeground(AString.Style.ITALIC, 250, 250, 250)
                        .addWithAlign(70, lend, 60, doc.value()).styleClear();
                lend = 0;
            }

            // --- ОБНОВЛЕНО: Чтение повторяющихся аннотаций @Example ---
            Example[] examples = m.getAnnotationsByType(Example.class);
            for (Example example : examples) {
                s.styleForeground(AString.Style.BRIGHT, 150, 255, 150)
                        .addWithAlign(70, lend, 60, buildExampleString(example.value())).styleClear();
                lend = 0;
            }
            // ---------------------------------------------------------

            if (m.isAnnotationPresent(Api.class)) {
                s.addWithAlign(70, lend, 60,
                        AString.of().add("To view details, see ").style(AString.Style.BRIGHT)
                                .add(getAppName()).add(" help " + Flow.flow(args).filterNot(a -> a.startsWith("--")).join(" ") + " " + cmdName).styleClear().toString());
            } else if (doc == null && examples.length == 0) s.add("\n");
            s.repeat(150, "⎼").add("\n");
        }
        addRequiredHelp(s);
        result(CTypeI.STRING, s.toString(), getByKey("file-out"), getByKey("file-out-format"), getByKey("console"), getByKey("file-out-append"));
    }

    public ARFuture<Object> execute(CTypeI<?> type, Object api) {
        var executionResultFuture = ARFuture.make();

        int argIndex = 0;
        while (argIndex < args.size()) {
            var a = args.get(argIndex);
            if (a.startsWith("--")) {
                if (a.matches("--\\w+=.*")) {
                    argIndex++;
                } else {
                    argIndex += 2;
                }
            } else {
                break;
            }
        }
        if (argIndex >= args.size()) {
            if (helpMode) {
                printHelpApi(type);
                executionResultFuture.tryDone(null);
            } else {
                System.err.println("Method is not specified: " + AString.of().add(args));
                showHelp();
                executionResultFuture.tryDone(null);
            }
            return executionResultFuture;
        }
        var a = args.get(argIndex).toLowerCase();
        if (a.equals("help")) {
            helpMode = true;
            args.remove(a);
            execute(type, api).to(executionResultFuture);
        } else {
            for (var m : getMethods(type)) {
                var aliasAnnotation = m.getAnnotation(Alias.class);
                String alias = aliasAnnotation == null ? null : aliasAnnotation.value();
                if (convertMethodName(m.getName()).equals(a) || alias != null && alias.equals(a)) {
                    args.remove(a);
                    execute(type, api, m).to(executionResultFuture);
                    return executionResultFuture;
                }
            }
            if (helpMode) {
                printHelpApi(type);
                executionResultFuture.tryDone(null);
            } else {
                System.out.println("Method is not found: " + AString.of().add(args));
                argsOrigin.remove(a);
                showHelp();
                executionResultFuture.tryDone(null);
            }
        }
        return executionResultFuture;
    }

    private void showHelp() {
        helpMode = true;
        args = new ObjectArrayList<>(argsOrigin);
        execute(rootApi);
    }

    private void addRequiredHelp(AString s) {
        s.add("\n");
        s.add("The options are available for all commands and allow you to specify the output format for the command execution.\n");
        s.style(AString.Style.BRIGHT);
        s.add("--console <format>\n");
        s.add("--file-out <file-name>\n");
        s.add("--file-out-append <true|false>\n");
        s.add("--file-out-format <format>\n\n");
        s.add("--loggerConsole\n\n");
        s.styleClear();
        s.add("Possible values for the format: ")
                .style(AString.Style.BRIGHT).add("human").styleClear().add(", ")
                .style(AString.Style.BRIGHT).add("json").styleClear().add(", ")
                .style(AString.Style.BRIGHT).add("bin").styleClear().add("\n");
        s.add("if the command supports an additional output format, this will be indicated in the help information.\n\n");
        s.add("If the argument is marked with the ")
                .style(AString.Style.BRIGHT).add("#").styleClear()
                .add(" symbol in the help information, you can use the ")
                .style(AString.Style.BRIGHT).add("--stdin").styleClear()
                .add(" option to specify the need to read the argument value from the standard input.\n");
        s.add("Use the ")
                .style(AString.Style.BRIGHT).add("--with-types").styleClear()
                .add(" option to display the background information with the types of arguments.");
        s.add("\n");
        if (footer != null) {
            s.add(footer);
        }
    }

    private String buildExampleString(String s) {
        var ss = AString.of();
        ss.add("Example: ").addVars(s, c -> {
            switch (c.toString()) {
                case "exCmd":
                    return getAppName();
            }
            return "???";
        });
        return ss.toString();
    }

    public void setStd(ByteArrayInputStream stdin) {
        this.stdin = stdin;
    }

    @Retention(RetentionPolicy.RUNTIME)
    public @interface Doc {
        String value();
    }

    // --- ОБНОВЛЕНО: СДЕЛАНО ПОВТОРЯЕМЫМ ---
    @Retention(RetentionPolicy.RUNTIME)
    @Repeatable(Examples.class) // <-- ДОБАВЛЕНО
    public @interface Example {
        String value();
    }
    // ------------------------------------

    // --- ДОБАВЛЕНО: КОНТЕЙНЕР ДЛЯ @Example ---
    @Retention(RetentionPolicy.RUNTIME)
    @Target({ElementType.METHOD, ElementType.PARAMETER, ElementType.TYPE, ElementType.FIELD})
    public @interface Examples {
        Example[] value();
    }
    // --------------------------------------

    @Retention(RetentionPolicy.RUNTIME)
    public @interface Alias {
        String value();
    }

    @Retention(RetentionPolicy.RUNTIME)
    public @interface Optional {
        String value() default "";
    }

    @Retention(RetentionPolicy.RUNTIME)
    public @interface StdIn {
    }

    @Retention(RetentionPolicy.RUNTIME)
    public @interface ResultFormats {
        String[] value() default {};
    }

    @Retention(RetentionPolicy.RUNTIME)
    @Target({ElementType.METHOD})
    public @interface Api {
    }

    public class ResCtx {
        String fileName;
        StandardOpenOption fileMode;
        boolean toFile;
        boolean append;

        public void setFileModeAppend(boolean f) {
            this.fileMode = f ? StandardOpenOption.APPEND : StandardOpenOption.TRUNCATE_EXISTING;
        }

        public boolean isAppend() {
            return append;
        }

        public String getFileName() {
            return fileName;
        }

        public void setFileName(String name) {
            fileName = name;
        }

        public boolean isToFile() {
            return toFile;
        }
    }
}