package io.aether.api.common;

import io.aether.common.NetworkConfigurator;
import io.aether.common.Protocol;
import io.aether.net.fastMeta.SerializerPackNumber;
import io.aether.net.serialization.DeserializerSizeStream;
import io.aether.utils.ConcurrentHashSet;
import io.aether.utils.dataio.DataInOutStatic;
import io.aether.utils.interfaces.AConsumer;
import io.aether.utils.streams.Value;

import java.net.URI;
import java.util.Collection;

public interface AetherCodecUtil {
    static AetherCodec of(URI uri) {
        return AetherCodec.valueOf(uri.getScheme().toUpperCase());
    }

    static NetworkConfigurator getNetworkConfigurator(URI uri) {
        return getNetworkConfigurator(of(uri));
    }

    static NetworkConfigurator getNetworkConfigurator(AetherCodec codec) {
        switch (codec) {
            case TCP:
                return new NetworkConfigurator() {
                    @Override
                    public Protocol initConnectionServer() {
                        return new ProtocolTCP();
                    }

                    @Override
                    public Protocol initConnectionClient(URI host) {
                        return new ProtocolTCP();
                    }

                    @Override
                    public String getName() {
                        return "tcp";
                    }
                };
            case UDP:
                return new NetworkConfigurator() {
                    @Override
                    public Protocol initConnectionServer() {
                        return new ProtocolUDP();
                    }

                    @Override
                    public Protocol initConnectionClient(URI host) {
                        return new ProtocolUDP();
                    }

                    @Override
                    public String getName() {
                        return "udp";
                    }
                };
            case WEBSOCKET:
                throw new UnsupportedOperationException();
        }
        throw new UnsupportedOperationException();
    }

    class ProtocolUDP implements Protocol {
        @Override
        public void fromSocket(Value<byte[]> data, AConsumer<Value<byte[]>> result) {
            result.accept(data);
        }

        @Override
        public void toSocket(Value<byte[]> data, AConsumer<Value<byte[]>> result) {
            result.accept(data);
        }
    }

    class ProtocolTCP implements Protocol {
        final DeserializerSizeStream deserializerSizeStream = new DeserializerSizeStream();
        volatile DataInOutStatic pkg;
        private volatile Collection<Value<?>> values;

        @Override
        public void fromSocket(Value<byte[]> value, AConsumer<Value<byte[]>> result) {
            value.enter(this);
            if (value.isOnlyRequestData()) {
                result.accept(value);
                return;
            }
            assert !value.isRequestData();
            if (value.isData()) {
//                Log.trace("acc down receive data: $data", "data", value.data());
                var b = new DataInOutStatic(value.data());
                if (values == null) {
                    values = new ConcurrentHashSet<>();
                }
                values.add(value);
                while (b.isReadable()) {
                    if (pkg == null) {
                        if (!deserializerSizeStream.put(b)) {
                            return;
                        }
                        int size = (int) deserializerSizeStream.getValue();
                        assert size != 0;
                        pkg = new DataInOutStatic(new byte[size], 0, 0);
                        deserializerSizeStream.reset();
                    }
                    if (b.isReadable() && b.getSizeForRead() >= pkg.getSizeForWrite()) {
                        b.read(pkg.data, pkg.writePos, pkg.getSizeForWrite());
                        pkg.writePos = pkg.data.length;
                        var p = pkg;
                        var vv = values;
                        values = null;
                        pkg = null;
                        var arr = p.toArray();
//                        Log.trace("acc down send data to up: $data", "data", arr);
                        if (vv == null) {
                            result.accept(Value.of(arr));
                        } else {
                            result.accept(Value.of(arr, false, vv));
                        }
                    } else {
                        var s = b.getSizeForRead();
                        b.read(pkg.data, pkg.writePos, s);
                        pkg.writePos += s;
                    }
                }
            } else if (value.isForce() || value.isClose()) {
                result.accept(value);
            }
        }

        Value<byte[]> convert(Value<byte[]> value) {
            return value.map(d -> {
                var b = new DataInOutStatic(value.data());
                var sizeHead = SerializerPackNumber.INSTANCE.calcSize(b.getSizeForRead());
                var buf = new byte[sizeHead + b.getSizeForRead()];
                var bb = new DataInOutStatic(buf);
                bb.setWritePos(0);
                SerializerPackNumber.INSTANCE.put(bb, b.getSizeForRead());
                b.read(buf, sizeHead, b.getSizeForRead());
                return buf;
            });
        }

        @Override
        public void toSocket(Value<byte[]> value, AConsumer<Value<byte[]>> result) {
            value.enter(this);
            if (value.isData()) {
                if (value.data().length == 0) {
                    throw new IllegalStateException();
                }
                result.accept(convert(value));
            } else {
                result.accept(value);
            }
        }
    }
}
