package io.aether.crypto;

import io.aether.api.common.CryptoLib;
import io.aether.utils.HexUtils;
import io.aether.utils.ModuleAutoRun;
import io.aether.utils.flow.Flow;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

/**
 * The CryptoProviderFactory is a central registry for all cryptographic providers.
 * It provides a way to register and retrieve specific CryptoProvider implementations
 * by their name, ensuring a complete decoupling from concrete libraries.
 */
public final class CryptoProviderFactory {

    private static final Map<String, CryptoProvider> providers = new ConcurrentHashMap<>();

    private CryptoProviderFactory() {
        // Private constructor to prevent instantiation
    }

    /**
     * Registers a CryptoProvider implementation with the factory.
     * This method is typically called by a module's static initializer.
     *
     * @param provider The CryptoProvider instance to register.
     */
    public static void register(CryptoProvider provider) {
        providers.put(provider.getCryptoLibName().toLowerCase(), provider);
    }

    /**
     * Retrieves a registered CryptoProvider by its name.
     *
     * @param libName The name of the crypto library (e.g., "SODIUM", "HYDROGEN").
     * @return The corresponding CryptoProvider instance.
     * @throws IllegalArgumentException if the provider is not registered.
     */
    public static CryptoProvider getProvider(String libName) {
        ModuleAutoRun.initialize("CryptoProviderFactory");
        var res = providers.get(libName.toLowerCase());
        if (res == null) {
            throw new IllegalArgumentException("Provider not registered for library: " + libName);
        }
        return res;
    }

    /**
     * Retrieves a CryptoProvider instance based on an AKey object.
     *
     * @param key The AKey object.
     * @return The corresponding CryptoProvider instance.
     */
    public static CryptoProvider getProvider(AKey key) {
        return getProvider(key.getProviderName());
    }

    /**
     * Retrieves a CryptoProvider instance based on a Sign object.
     *
     * @param sign The Sign object.
     * @return The corresponding CryptoProvider instance.
     */
    public static CryptoProvider getProvider(Sign sign) {
        return getProvider(sign.getProviderName());
    }

    /**
     * Creates an AKey instance from raw data and provider details.
     * This method delegates the key creation to the specific CryptoProvider.
     *
     * @param providerName The name of the cryptographic provider.
     * @param keyType      The type of the key.
     * @param data         The byte array of the key.
     * @return A concrete implementation of AKey.
     */
    public static AKey createKey(String providerName, KeyType keyType, byte[] data) {
        CryptoProvider provider = getProvider(providerName);
        return provider.createKey(keyType, data);
    }

    /**
     * Creates an AKey instance from a serialized string representation.
     * The format is "PROVIDER_NAME:KEY_TYPE:DATA".
     *
     * @param s The serialized key string.
     * @return A concrete implementation of AKey.
     */
    public static AKey createKey(String s) {
        try {
            if (s == null || s.isBlank()) {
                return null;
            }
            var parts = s.split(":");
            if (parts.length != 3) {
                throw new IllegalArgumentException("Invalid admin key string format. Expected 'PROVIDER_NAME:KEY_TYPE:DATA' actual: " + s + "");
            }
            String providerName = parts[0];
            KeyType keyType = KeyType.valueOf(parts[1]);
            byte[] data = HexUtils.hexToBytes(parts[2]);
            return createKey(providerName, keyType, data);
        } catch (Exception e) {
            throw new IllegalArgumentException("Invalid admin key: " + s, e);
        }
    }

    public static Flow<CryptoProvider> allFlow() {
        return Flow.flow(all());
    }

    public static Set<CryptoProvider> all() {
        return Set.copyOf(providers.values());
    }

    public static SignChecker createSignChecker(String s) {
        if (s == null || s.isBlank()) {
            return null;
        }
        var i = s.indexOf(":");
        String providerName = s.substring(0, i);
        s = s.substring(i + 1);
        var cp = getProvider(providerName);
        if (s.contains(":")) throw new IllegalStateException(s);
        return cp.createSigner(cp.createKey(KeyType.SIGN_PUBLIC, s).asSignPublicKey());
    }

    public static Set<Signer> makeSigners() {
        Set<Signer> res = new HashSet<>();
        for (var p : all()) {
            res.add(p.createSigner());
        }
        return res;
    }

    public static Signer createSigner(String providerName, String publicKey, String privateKey) {
        var p = getProvider(providerName);
        return p.createSigner(p.createSignKeys(publicKey, privateKey));
    }

    public static PairAsymKeysSigned createPairKeysSigned(String providerName, String publicKey, String privateKey, Signer signer) {
        var p = getProvider(providerName);
        var pk = p.createKey(KeyType.ASYMMETRIC_PUBLIC, publicKey);
        return new PairAsymKeysSigned(new SignedKey(pk, signer), p.createKey(KeyType.ASYMMETRIC_PRIVATE, privateKey));
    }
}