package io.aether.net.fastMeta.nio;

import io.aether.logger.LNode;
import io.aether.logger.Log;
import io.aether.net.fastMeta.FastApiContextLocal;
import io.aether.net.fastMeta.FastMetaApi;
import io.aether.net.fastMeta.RemoteApi;
import io.aether.net.fastMeta.SerializerPackNumber;
import io.aether.net.serialization.DeserializerSizeStream;
import io.aether.utils.RU;
import io.aether.utils.dataio.DataInOut;
import io.aether.utils.dataio.DataInOutStatic;
import io.aether.utils.futures.AFuture;
import io.aether.utils.interfaces.AFunction;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.*;
import java.util.Iterator;
import java.util.Objects;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicBoolean;

public class FastApiContextClientNIO<LT, RT extends RemoteApi> extends FastApiContextLocal<LT> implements Runnable {

    public final FastMetaApi<LT, ?> localApiMeta;
    public final FastMetaApi<?, RT> remoteApiMeta;
    public final AFuture connectedFuture = AFuture.make();

    private final SocketChannel channel;
    private final Selector selector;
    private final SelectionKey key;

    private final ByteBuffer readBuffer = ByteBuffer.allocate(65536);
    private final ByteBuffer internalWriteBuffer = ByteBuffer.allocate(65536);
    private final AtomicBoolean isClosed = new AtomicBoolean(false);

    private final DeserializerSizeStream deserializerSizeStream = new DeserializerSizeStream();
    private final LNode logContext;
    // -------------------------

    public FastApiContextClientNIO(SocketChannel channel, FastMetaApi<LT, ?> localApiMeta,
                                   FastMetaApi<?, RT> remoteApiMeta, AFunction<RT, LT> localApi, LNode logContext) throws IOException {
        super(c -> localApi.apply(remoteApiMeta.makeRemote(c)));
        this.logContext = logContext;
        this.channel = Objects.requireNonNull(channel);
        this.localApiMeta = localApiMeta;
        this.remoteApiMeta = remoteApiMeta;

        this.selector = Selector.open();
        this.channel.configureBlocking(false);

        this.key = this.channel.register(this.selector, SelectionKey.OP_CONNECT);
        this.key.attach(this);


        var executor=Executors.newSingleThreadExecutor(r -> {
            try {
                Thread t = new Thread(r, "NIO-Client-Worker-" + ((InetSocketAddress) channel.getRemoteAddress()).getPort());
                t.setDaemon(true);
                return t;
            } catch (ClosedChannelException e) {
                destroyChannel();
                return null;
            } catch (IOException e) {
                throw new RuntimeException("Failed to get remote address for thread name.", e);
            }
        });
        executor.execute(this);
        connectedFuture.onCancel(executor::shutdown);
    }

    @Override
    public void run() {
        try (var ln = Log.context(logContext)) {
//            Log.debug("selector loop started for channel: $channel", "socket", "nio client", "channel", channel);
            while (channel.isOpen() && !isClosed.get()) {

                if (!isEmpty()) {
                    selector.wakeup();
                }

                selector.select();
                Iterator<SelectionKey> keys = selector.selectedKeys().iterator();
                while (keys.hasNext()) {
                    SelectionKey key = keys.next();
                    keys.remove();

                    if (!key.isValid()) continue;

                    try {
                        if (key.isConnectable()) {
                            handleConnect(key);
                        }
                        if (key.isReadable()) {
                            readFromChannel(key);
                        }
                        if (key.isWritable()) {
                            writeToChannel(key);
                        }
                    } catch (CancelledKeyException e) {
                        destroyChannel();
                        break;
                    } catch (ClosedChannelException e) {
                        throw e; // Propagate to outer catch for graceful close message
                    } catch (IOException e) {
                        Log.warn("I/O error on channel: $channel", e, "socket", "nio client", "channel", channel);
                        destroyChannel();
                    }
                }

                // Check and set OP_WRITE interest if the queue is not empty
                try {
                    if (key.isValid()) {
                        if (!isEmpty() && (key.interestOps() & SelectionKey.OP_WRITE) == 0) {
                            key.interestOps(key.interestOps() | SelectionKey.OP_WRITE);
                        }
                    }
                } catch (CancelledKeyException e) {
                    // This is the expected exception when 'key' is cancelled by destroyChannel().
                    // We catch it here to prevent the worker thread from crashing
                    // but we rely on the main loop condition 'channel.isOpen()' to exit.
                    Log.trace("SelectionKey was cancelled during interestOps modification. Exiting loop condition check.", "socket", "nio client", "channel", channel);
                }
            }
        } catch (ClosedChannelException e) {
            // Channel closed by the selector or external force. Graceful exit.
            Log.info("Client channel closed gracefully: $channel", "socket", "nio client", "channel", channel);
            connectedFuture.error(e);
        } catch (ClosedSelectorException e) {
            // Selector closed by destroyChannel(). Graceful exit.
            Log.info("Client selector closed gracefully: $channel", "socket", "nio client", "channel", channel);
            connectedFuture.error(e);
        } catch (Exception e) {
            // General worker error. Non-graceful exit.
            Log.warn("Client worker error for channel: $channel", e, "socket", "nio client", "channel", channel);
            connectedFuture.error(e);
        } finally {
            destroyChannel();
//            Log.debug("selector loop finished for channel: $channel", "socket", "nio client", "channel", channel);
        }
    }

    private void handleConnect(SelectionKey key) throws IOException {
        SocketChannel ch = (SocketChannel) key.channel();
        if (ch.isConnectionPending()) {
            ch.finishConnect();
        }
        key.interestOps(SelectionKey.OP_READ);
        connectedFuture.done();
        Log.info("client connected successfully to: $address", "socket", "nio client", "address", ch.getRemoteAddress());
    }

    private void readFromChannel(SelectionKey key) throws IOException {
        int bytesRead = channel.read(readBuffer);

//        Log.trace("read attempt: $bytesRead bytes", "socket", "nio client", "bytesRead", bytesRead, "channel", channel);

        if (bytesRead == -1) {
            // The channel reached end-of-stream. The remote side closed the connection.
            throw new ClosedChannelException();
        }

        if (bytesRead == 0) return;

        readBuffer.flip();

        byte[] rawBytes = new byte[readBuffer.remaining()];
        readBuffer.get(rawBytes);
        readBuffer.clear();

        var b = new DataInOutStatic(rawBytes);
        int totalBytes = b.getWritePos();

        while (b.isReadable()) {
            if (!deserializerSizeStream.put(b)) {
//                Log.trace("insufficient data to read size header", "socket", "nio client", "remaining", b.getSizeForRead(), "channel", channel);
                return;
            }
            long size = deserializerSizeStream.getValue();
            deserializerSizeStream.reset();

//            Log.debug("received packet size: $size", "socket", "nio client", "size", size, "channel", channel);

            if (size == 0) continue;

            if (size > Integer.MAX_VALUE) {
                Log.error("Packet size too large: $size", "socket", "nio client", "size", size);
                throw new IOException("Packet too large");
            }

            if (b.getSizeForRead() < size) {
                Log.warn("received incomplete packet body.", "socket", "nio client", "expected", size, "available", b.getSizeForRead(), "channel", channel);
                // Return to keep the remaining data in the buffer for the next read
                return;
            }

            byte[] pkgBody = b.readBytes((int) size);
//            Log.trace("processing packet body of size: $size", "socket", "nio client", "size", size, "channel", channel);

            try {
                localApiMeta.makeLocal(this, pkgBody, localApi);
            } catch (Exception e) {
                Log.error("Error processing received NIO data: $error", e, "socket", "nio client", "error", e.getMessage(), "channel", channel);
                throw new IOException("Data processing error", e);
            }
        }
    }

    private void writeToChannel(SelectionKey key) throws IOException {
        if (internalWriteBuffer.position() == 0) {
            // Prepare the buffer only if the previous write operation finished
            // and the buffer is empty (position == 0).

            byte[] rawData = remoteDataToArray();

            if (rawData.length > 0) {

                var sizeDataOut = new DataInOut();
                // 1. Write the size prefix
                new SerializerPackNumber().put(sizeDataOut, rawData.length);
                byte[] lengthBytes = sizeDataOut.toArray();

                int totalSize = lengthBytes.length + rawData.length;
                if (totalSize > internalWriteBuffer.capacity()) {
                    Log.error("Internal write buffer too small for packet of size: $totalSize", "socket", "nio client", "totalSize", totalSize);
                    throw new IllegalStateException("Internal write buffer too small for a single packet.");
                }

                internalWriteBuffer.clear();
                internalWriteBuffer.put(lengthBytes);
                internalWriteBuffer.put(rawData);
                internalWriteBuffer.flip();

//                Log.debug("prepared packet for send. Total size: $totalSize", "socket", "nio client", "totalSize", totalSize, "dataSize", rawData.length, "channel", channel);
            }
        }

        if (internalWriteBuffer.hasRemaining()) {
            int bytesWritten = channel.write(internalWriteBuffer);
//            Log.trace("wrote $bytesWritten bytes to channel: $channel", "socket", "nio client", "bytesWritten", bytesWritten, "remaining", internalWriteBuffer.remaining(), "channel", channel);
        }

        if (!internalWriteBuffer.hasRemaining()) {
            // All data was written, switch back to read interest only
            internalWriteBuffer.clear();
            key.interestOps(SelectionKey.OP_READ);
//            Log.trace("finished writing, switched to OP_READ for channel: $channel", "socket", "nio client", "channel", channel);
        } else {
            // Partial write occurred, compact the buffer for the next write
            internalWriteBuffer.compact();
            internalWriteBuffer.flip();
        }
    }

    @Override
    public void flush(AFuture sendFuture) {
        if (isEmpty()) {
            sendFuture.done();
            return;
        }
        if (channel.isOpen()) {
            try {
                // 1. Set the OP_WRITE bit to signal the selector that we have data
                // in the queue and the channel must be monitored for write-readiness.
                key.interestOps(key.interestOps() | SelectionKey.OP_WRITE);
            } catch (java.nio.channels.CancelledKeyException e) {
                // The channel key might have been invalidated if the channel closed
                // between checking isOpen() and setting interestOps().
                sendFuture.error(new ClosedChannelException());
                return;
            }

            // 2. Wake up the selector thread to process the updated interest set immediately.
            selector.wakeup();
//            Log.trace("context flush triggered wakeup and OP_WRITE interest on channel: $channel", "socket", "nio client", "channel", channel);
            sendFuture.done();
        } else {
            sendFuture.error(new ClosedChannelException());
        }
    }

    @Override
    public AFuture close() {
        Log.info("client initiated close on channel: $channel", "socket", "nio client", "channel", channel);
        destroyChannel();
        return AFuture.completed();
    }

    private void destroyChannel() {
        if (isClosed.compareAndSet(false, true)) {
            try {
                if (channel.isOpen()) {
                    channel.close();
                }
                // Closing the selector also cancels its keys, which is fine,
                // as this is the intended cleanup procedure.
                selector.close();
            } catch (IOException e) {
                Log.error("Error closing NIO client resources for: $channel", e, "socket", "nio client", "channel", channel);
            }
        }
    }

}