package io.aether.net.fastMeta.nio;

import io.aether.logger.LNode;
import io.aether.logger.Log;
import io.aether.net.fastMeta.FastApiContextLocal;
import io.aether.net.fastMeta.FastMetaApi;
import io.aether.net.fastMeta.FastMetaClient;
import io.aether.net.fastMeta.RemoteApi;
import io.aether.utils.RU;
import io.aether.utils.futures.AFuture;
import io.aether.utils.futures.ARFuture;
import io.aether.utils.interfaces.AFunction;
import io.aether.utils.interfaces.ARunnable;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.URI;
import java.nio.channels.SocketChannel;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;

/**
 * NIO implementation of the FastMetaClient, handling connection initiation
 * and automatic, scheduled reconnection attempts using RU.schedule.
 *
 * REWRITTEN FOR ROBUSTNESS: This implementation centralizes retry logic
 * and does not trust the connection context to always report errors.
 *
 * @param <LT> The Local API type (implemented by the client).
 * @param <RT> The Remote API type (exposed by the server).
 */
public class FastMetaClientNIO<LT, RT extends RemoteApi> implements FastMetaClient<LT, RT> {

    /** Delay in milliseconds between reconnection attempts (1 second). */
    private static final long RECONNECT_DELAY_MS = 500;
    /**
     * Connection attempt timeout. If a context doesn't connect in this time,
     * it's considered stalled and destroyed.
     */
    private static final long CONNECT_ATTEMPT_TIMEOUT_MS = 800;

    private final AtomicReference<FastApiContextClientNIO<LT, RT>> currentContextRef = new AtomicReference<>();
    private final LNode logContext = Log.createContext();
    private final AtomicReference<ScheduledFuture<?>> connectAttemptFutureRef = new AtomicReference<>();

    // Central future for the connection result
    private final ARFuture<FastApiContextClientNIO<LT, RT>> resultFuture = ARFuture.make();

    // Connection parameters
    private URI uri;
    private FastMetaApi<LT, ?> localApiMeta;
    private FastMetaApi<?, RT> remoteApiMeta;
    private AFunction<RT, LT> localApi;

    public FastMetaClientNIO() {
        // Link the public future's cancellation to our destroy mechanism
        resultFuture.onCancel(() -> {
            Log.warn("Connection cancelled via resultFuture for URI: $uri", "uri", uri);
            destroy(true); // Force destroy on external cancel
        });
    }

    @Override
    public ARFuture<FastApiContextClientNIO<LT, RT>> connect(URI uri,
                                                             FastMetaApi<LT, ?> localApiMeta,
                                                             FastMetaApi<?, RT> remoteApiMeta,
                                                             AFunction<RT, LT> localApi) {

        if (this.uri != null) {
            // Prevent re-use of the client instance
            return ARFuture.doThrow(new IllegalStateException("Client connection parameters were already set. Create a new client instance."));
        }

        // 1. Store connection parameters
        this.uri = uri;
        this.localApiMeta = localApiMeta;
        this.remoteApiMeta = remoteApiMeta;
        this.localApi = localApi;

        if (resultFuture.isFinalStatus()) {
            return resultFuture;
        }

        // 2. Initiate the first attempt immediately
        scheduleConnectAttempt(0);

        return resultFuture;
    }

    /**
     * Schedules a connection attempt after a given delay.
     * This method contains the core retry loop logic.
     *
     * @param delayMs The delay in milliseconds before executing the connection attempt.
     */
    private void scheduleConnectAttempt(long delayMs) {
        // Do not schedule if the client has already succeeded, failed permanently, or been cancelled
        if (resultFuture.isFinalStatus()) {
            return;
        }

        ARunnable connectionTask = Log.wrap(() -> {
            // Check status *again* inside the scheduled task, in case it was cancelled
            // while waiting for the schedule delay.
            if (resultFuture.isFinalStatus()) {
                Log.debug("Connection task aborted, result is final.", "uri", uri);
                return;
            }

            FastApiContextClientNIO<LT, RT> existingContext = currentContextRef.get();

            // --- Liveness Check ---
            // Check if a context already exists from a previous attempt
            if (existingContext != null) {
                // If it's already done, the listener *should* have set resultFuture.
                // If it's in error, the listener *should* have rescheduled.
                // If it's still pending, it might be stalled.

                // If the listener (in setupContextListeners) *failed* to run,
                // we manually check the state.
                if (existingContext.connectedFuture.isDone()) {
                    Log.info("Found completed context during check. Finalizing.", "uri", uri);
                    resultFuture.tryDone(existingContext);
                    return; // Success
                }

                if (existingContext.connectedFuture.isError()) {
                    Log.warn("Found stale context in error state. Clearing and retrying...", "uri", uri);
                    // Force-clear the bad context and fall through to retry
                    currentContextRef.compareAndSet(existingContext, null);
                    existingContext.close(); // Ensure it's cleaned up
                } else {
                    // Context exists and is still pending.
                    // We assume it's working and just wait.
                    // The listener set up on it is responsible for success or retry.
                    Log.debug("Connection attempt already in progress. Waiting.", "uri", uri);
                    // Reschedule this *check*
                    scheduleConnectAttempt(RECONNECT_DELAY_MS);
                    return;
                }
            }

            // --- No context, or stale context cleared. Proceed with new attempt. ---

            SocketChannel channel = null;
            FastApiContextClientNIO<LT, RT> newContext = null;

            try {
                // 1. Attempt to initiate a new connection
                InetSocketAddress address = new InetSocketAddress(uri.getHost(), uri.getPort());
                Log.info("Attempting connection to: $uri",
                        "socket", "nio client", "uri", uri, "address", address);

                channel = SocketChannel.open();
                channel.configureBlocking(false);
                channel.connect(address); // Initiate non-blocking connection

                newContext = new FastApiContextClientNIO<>(channel, localApiMeta, remoteApiMeta, localApi, logContext);

                // 2. Atomic context installation
                if (currentContextRef.compareAndSet(null, newContext)) {
                    // We successfully installed the new context.
                    // Set up listeners that will handle success or async failure.
                    setupContextListeners(newContext);
                } else {
                    // Race condition: another attempt (or manual intervention) set a context.
                    // This is unlikely but possible. Destroy our new context.
                    Log.warn("Connection race condition. Aborting redundant attempt.", "uri", uri);
                    newContext.close();
                    // Reschedule a check
                    scheduleConnectAttempt(RECONNECT_DELAY_MS);
                }

            } catch (IOException e) {
                // 3. SYNCHRONOUS error (e.g., SocketChannel.open() failed, UnresolvedAddressException)
                String errorType = e.getClass().getSimpleName();
                Log.warn("Synchronous connection setup failed with $errorType for URI: $uri. Retrying in $delayMs...", e,
                        "socket", "nio client", "uri", uri, "errorType", errorType, "delayMs", RECONNECT_DELAY_MS);

                closeChannelQuietly(channel);
                currentContextRef.compareAndSet(newContext, null); // Ensure context is nullified

                // Schedule the next attempt
                scheduleConnectAttempt(RECONNECT_DELAY_MS);

            } catch (Exception e) {
                // 4. All other setup errors (e.g., SecurityException)
                Log.warn("Critical connection setup failed for URI: $uri. Retrying in $delayMs...", e,
                        "socket", "nio client", "uri", uri, "delayMs", RECONNECT_DELAY_MS);

                closeChannelQuietly(channel);
                currentContextRef.compareAndSet(newContext, null);

                // Schedule the next attempt
                scheduleConnectAttempt(RECONNECT_DELAY_MS);
            }
        });

        // Schedule the task and store the future for cancellation
        ScheduledFuture<?> newAttempt = RU.schedule(delayMs, connectionTask);
        connectAttemptFutureRef.set(newAttempt);
    }

    /**
     * Sets up listeners on the connection context's future.
     * This handles the asynchronous success/error/cancellation.
     */
    private void setupContextListeners(FastApiContextClientNIO<LT, RT> context) {
        // This future is the *only* signal we get from FastApiContextClientNIO
        // about the status of the *asynchronous* part of the connection.
        var connectedFuture = context.connectedFuture;

        connectedFuture.addListener(Log.wrap(f -> {
            // Use try-catch block inside listener to prevent listener failure
            // from silently breaking the retry loop.
            try {
                if (resultFuture.isFinalStatus()) {
                    // We already finished (e.g., timed out, cancelled). Ignore this.
                    return;
                }

                if (f.isDone()) {
                    // SUCCESS
                    Log.info("Connection successful for URI: $uri", "socket", "nio client", "uri", uri);
                    // Try to set the final result.
                    if (!resultFuture.tryDone(context)) {
                        // Failed to set (e.g., race with cancellation).
                        // If so, we must destroy the context we just created.
                        Log.warn("Connection succeeded but resultFuture was already final. Closing context.", "uri", uri);
                        context.close();
                    }
                } else if (f.isError()) {
                    // ASYNCHRONOUS FAILURE (e.g., ConnectException, Handshake failed)
                    Log.warn("Asynchronous connection error for URI: $uri. Attempting reconnect...", f.getError(),
                            "socket", "nio client", "uri", uri);

                    // Reset the context to allow a new attempt
                    currentContextRef.compareAndSet(context, null);
                    context.close(); // Ensure failed context is destroyed

                    // Schedule the next attempt
                    scheduleConnectAttempt(RECONNECT_DELAY_MS);

                } else if (f.isCanceled()) {
                    // ASYNCHRONOUS CANCELLATION (e.g., context internal logic)
                    Log.warn("Connection context was cancelled internally for URI: $uri. Retrying...", "socket", "nio client", "uri", uri);

                    currentContextRef.compareAndSet(context, null);
                    // context.close() was likely already called

                    // Propagate cancel to main future ONLY IF it wasn't us.
                    // We treat internal cancel as a retryable error.
                    scheduleConnectAttempt(RECONNECT_DELAY_MS);
                }
            } catch (Exception e) {
                Log.warn("CRITICAL: Failure inside connection listener. Retrying.", e, "uri", uri);
                try {
                    currentContextRef.compareAndSet(context, null);
                    context.close();
                } finally {
                    // Ensure retry loop continues even if listener fails
                    if (!resultFuture.isFinalStatus()) {
                        scheduleConnectAttempt(RECONNECT_DELAY_MS);
                    }
                }
            }
        }));

        // --- Timeout for STALLED context ---
        // This handles the case where connectedFuture *never* completes
        // (the bug we observed, where it logs but doesn't fail).
        connectedFuture.timeoutMs(CONNECT_ATTEMPT_TIMEOUT_MS, () -> {
            if (resultFuture.isFinalStatus() || connectedFuture.isFinalStatus()) {
                return; // Already handled
            }

            Log.warn("Connection attempt timed out (stalled context) for URI: $uri. Retrying...", "uri", uri);
            // This will trigger the f.isError() block above with a TimeoutException
            connectedFuture.error(new RuntimeException("Connection context stalled (timeout " + CONNECT_ATTEMPT_TIMEOUT_MS + "ms)"));
        });
    }

    private void closeChannelQuietly(SocketChannel channel) {
        if (channel != null && channel.isOpen()) {
            try {
                channel.close();
            } catch (IOException ignore) {
                // Ignore error on close cleanup
            }
        }
    }

    @Override
    public AFuture destroy(boolean force) {
        Log.info("Destroying FastMetaClient (force=$force) for URI: $uri", "force", force, "uri", uri);

        // 1. Mark our main future as cancelled. This stops any pending retry loops.
        // Use tryCancel to avoid erroring if it was already done.
        resultFuture.tryCancel();

        // 2. Cancel any scheduled *future* connection attempt
        ScheduledFuture<?> scheduledFuture = connectAttemptFutureRef.getAndSet(null);
        if (scheduledFuture != null) {
            scheduledFuture.cancel(force);
        }

        // 3. Destroy the *currently active* connection context
        FastApiContextClientNIO<LT, RT> context = currentContextRef.getAndSet(null);
        if (context == null) {
            Log.debug("No active connection context to destroy.", "uri", uri);
            return AFuture.completed();
        }

        Log.info("Initiating context destruction (force=$force).", "socket", "nio client", "force", force);
        return context.close(); // Return the future that completes on cleanup
    }
}