package io.aether.net.serialization;

import io.aether.net.fastMeta.AetherException;
import io.aether.utils.dataio.DataIn;

public class DeserializerSizeStream {
    private static final int u8 = 251;
    private static final int pow8_mask = 0xFF;
    private static final int pow8_shift = 8;
    private static final int pow8 = 256;
    private static final int k8ReservedFor16 = 16;
    private static final int k16ReservedFor32 = 256;
    private static final long pow32 = 4L * 1024 * 1024 * 1024;
    private static final int u16 = 5 * 256 + u8 - k8ReservedFor16;  // 1515
    private static final long u32 = 1024 * 1024 + u16 - k16ReservedFor32;  // 1 049 835
    private static final long u64 = u32 + pow32 * k16ReservedFor32;
    // maximum supported value is (1 Tb + 1 Mb + 1515 - 1)
    private static final int pow16_mask = 0xFFFF;
    private static final int pow16_shift = 16;
    private static final int pow16 = 65536;
    private static final int pow32_shift = 32;
    private static final long pow32_mask = 0xFFFFFFFFL;
    private final DeserializerShort deserializerShort = new DeserializerShort();
    private final DeserializerInt deserializerInt = new DeserializerInt();
    private long val;
    private int state;

    public boolean put(DataIn in) {
        while (in.isReadable()) {
            switch (state) {
                case 0:
                    val = in.readUByte();
                    if (val < u8) {
                        return true;
                    }
                    state = 1;
                    break;
                case 1:
                    var v = Byte.toUnsignedInt(in.readByte());
                    val = ((val - u8) << pow8_shift) + u8 + v;
                    if (val < u16) {
                        return true;
                    }
                    state = 2;
                    break;
                case 2:
                    if (deserializerShort.put(in)) {
                        var f = Short.toUnsignedInt(deserializerShort.getValue());
                        val = ((val - u16) << pow16_shift) + u16 + f;
                        if (val < u32) {
                            return true;
                        } else {
                            state = 3;
                        }
                    } else {
                        return false;
                    }
                    break;
                case 3:
                    if (deserializerInt.put(in)) {
                        var f1 = Integer.toUnsignedLong(deserializerInt.getValue());
                        val = ((val - u32) << pow32_shift) + u32 + f1;
                        if (val < u64) {
                            return true;
                        }
                        throw new AetherException();
                    } else {
                        return false;
                    }
            }
        }
        return false;
    }

    public void reset() {
        state = 0;
        val = 0;
        deserializerInt.reset();
        deserializerShort.reset();
    }

    public long getValue() {
        return val;
    }

    private static class DeserializerShort {
        private short val;
        private int index;

        public boolean put(DataIn in) {
            if (!in.isReadable()) return false;
            switch (index) {
                case 0:
                    if (in.getSizeForRead() >= 2) {
                        val = in.readShort();
                        index = 2;
                        return true;
                    } else {
                        val = (short) in.readUByte();
                        index = 1;
                        return false;
                    }
                case 1:
                    val |= (short) (in.readUByte() << 8);
                    index = 2;
                    return true;
                default:
                    throw new RuntimeException();
            }
        }

        public void reset() {
            index = 0;
            val = 0;
        }

        public short getValue() {
            return val;
        }
    }

    private static class DeserializerInt {
        private int val;
        private int index;

        public boolean put(DataIn in) {
            if (index == 0 && in.getSizeForRead() >= 4) {
                val = in.readInt();
                index = 0;
                return true;
            }
            while (index < 4 && in.isReadable()) {
                val = val | (Byte.toUnsignedInt(in.readByte()) << (8 * index));
                index++;
            }
            return index == 4;
        }

        public void reset() {
            index = 0;
            val = 0;
        }

        public int getValue() {
            return val;
        }
    }

}

