package io.aether.utils.streams;

import io.aether.net.fastMeta.SerializerPackNumber;
import io.aether.net.serialization.DeserializerSizeStream;
import io.aether.utils.AString;
import io.aether.utils.ConcurrentHashSet;
import io.aether.utils.dataio.DataInOutStatic;
import org.jetbrains.annotations.NotNull;

import java.util.Collection;

/**
 * Accumulator node that processes Value objects.
 * <p>
 * When receiving data from the upper gate (up), it prepends a size header to the byte array and sends it down.
 * When receiving data from the lower gate (down), it reads the size header first,
 * then accumulates incoming bytes until the full data packet is received,
 * and finally sends the reconstructed Value up.
 * <p>
 * This class ensures proper framing of byte arrays with size headers for reliable transmission.
 */
public class Accumulator implements Node<byte[], byte[], byte[], byte[]> {
    private final FGate<byte[], byte[]> up = FGate.of(new AccUp(this));
    private final FGate<byte[], byte[]> down = FGate.of(new AccDown(this));
    volatile DataInOutStatic pkg;

    @Override
    public void toString(AString sb) {
        var p = pkg;
        if (p == null) {
            sb.add("Accumulator(void)");
        } else {
            sb.add("Accumulator(").add(p.total()).add(":").add(p.getSizeForWrite()).add(")");
        }
    }

    @Override
    public String toString() {
        return toString2();
    }

    @Override
    public FGate<byte[], byte[]> gDown() {
        return down;
    }

    @Override
    public FGate<byte[], byte[]> gUp() {
        return up;
    }

    public static Gate<byte[], byte[]> of(Gate<byte[], byte[]> down) {
        var ac = new Accumulator();
        ac.down().link(down);
        return ac.up.outSide();
    }

    class AccUp extends FGate.Pair<byte[], byte[], byte[]> {
        public AccUp(Object owner) {
            super(owner);
        }

        @Override
        public FGate<?, byte[]>.@NotNull InsideGate pair() {
            return down.inSide;
        }

        byte[] convert(byte[] value) {
            var b = new DataInOutStatic(value);
            var sizeHead = SerializerPackNumber.INSTANCE.calcSize(b.getSizeForRead());
            var buf = new byte[sizeHead + b.getSizeForRead()];
            var bb = new DataInOutStatic(buf);
            bb.setWritePos(0);
            SerializerPackNumber.INSTANCE.put(bb, b.getSizeForRead());
            b.read(buf, sizeHead, b.getSizeForRead());
            return buf;
        }

        @Override
        public void send(FGate<byte[], byte[]> fGate, Value<byte[]> value) {
            value.enter(Accumulator.this);
            if (value.isData()) {
//                Log.trace("acc up receive data: $data", "data", value.data());
                if (value.data().length == 0) {
                    throw new IllegalStateException();
                }
                pair().send(value.map(this::convert));
            } else {
                pair().send(value);
            }
        }
    }

    class AccDown extends FGate.Pair<byte[], byte[], byte[]> {
        final DeserializerSizeStream deserializerSizeStream = new DeserializerSizeStream();
        private volatile Collection<Value<?>> values;

        public AccDown(Object owner) {
            super(owner);
        }

        @Override
        @NotNull
        public FGate<?, byte[]>.InsideGate pair() {
            return up.inSide;
        }

        @Override
        public void send(FGate<byte[], byte[]> fGate, Value<byte[]> value) {
            value.enter(Accumulator.this);
            if (value.isOnlyRequestData()) {
                gUp().inSide.send(value);
                return;
            }
            assert !value.isRequestData();
            if (value.isData()) {
//                Log.trace("acc down receive data: $data", "data", value.data());
                var b = new DataInOutStatic(value.data());
                if (values == null) {
                    values = new ConcurrentHashSet<>();
                }
                values.add(value);
                while (b.isReadable()) {
                    if (pkg == null) {
                        if (!deserializerSizeStream.put(b)) {
                            return;
                        }
                        int size = (int) deserializerSizeStream.getValue();
                        assert size != 0;
                        pkg = new DataInOutStatic(new byte[size], 0, 0);
                        deserializerSizeStream.reset();
                    }
                    if (b.isReadable() && b.getSizeForRead() >= pkg.getSizeForWrite()) {
                        b.read(pkg.data, pkg.writePos, pkg.getSizeForWrite());
                        pkg.writePos = pkg.data.length;
                        var p = pkg;
                        var vv = values;
                        values = null;
                        pkg = null;
                        var arr = p.toArray();
//                        Log.trace("acc down send data to up: $data", "data", arr);
                        if (vv == null) {
                            pair().send(Value.of(arr));
                        } else {
                            pair().send(Value.of(arr, false, vv));
                        }
                    } else {
                        var s = b.getSizeForRead();
                        b.read(pkg.data, pkg.writePos, s);
                        pkg.writePos += s;
                    }
                }
            } else if (value.isForce() || value.isClose()) {
                gUp().inSide.send(value);
            }
        }
    }
}
