package io.aether.utils.streams;

import io.aether.logger.Log;
import io.aether.utils.AString;
import io.aether.utils.CTypeI;
import io.aether.utils.interfaces.ObjectFind;
import org.jetbrains.annotations.NotNull;

import java.lang.invoke.VarHandle;

/**
 * Represents a gate that facilitates communication between nodes in a stream.
 *
 * @param <ToNode> The type of data processed by the node on one side of the gate.
 * @param <ToOut>  The type of data output from the node through the gate.
 */
public class FGate<ToNode, ToOut> {
    /**
     * VarHandle for atomic updates of the 'link' field to ensure thread safety.
     */
    private static final VarHandle LINK_UPDATER = CTypeI.of(FGate.class).getFieldVarHandle("link");

    /**
     * AcceptorI instance that defines how to accept and process data.
     */
    public final AcceptorI<ToOut, ToNode> acceptor;

    /**
     * InsideGate instance representing the internal side of the gate.
     */
    public final InsideGate inSide = new InsideGate();

    /**
     * OutSideGate instance representing the external side of the gate.
     */
    private final OutSideGate outSide = new OutSideGate();

    /**
     * Volatile Gate reference to ensure visibility and atomicity across threads.
     */
    public volatile Gate<ToOut, ToNode> link;

    /**
     * Constructs a new FGate with the specified acceptor.
     *
     * @param acceptor The AcceptorI instance that defines how to accept and process data.
     */
    public FGate(AcceptorI<ToOut, ToNode> acceptor) {
        this.acceptor = acceptor;
        acceptor.setFGate(this);
    }

    /**
     * Returns a string representation of the gate's internal side.
     *
     * @return A string representation of the gate's internal side.
     */
    @Override
    public String toString() {
        return inSide.toString();
    }

    /**
     * Returns the external side of the gate as a Gate instance.
     *
     * @return The external side of the gate as a Gate instance.
     */
    public Gate<ToNode, ToOut> outSide() {
        return outSide;
    }

    //region static

    /**
     * Creates a new FGate with the specified acceptor.
     *
     * @param <TIn>  The type of data processed by the node on one side of the gate.
     * @param <TOut> The type of data output from the node through the gate.
     * @param acceptor The AcceptorI instance that defines how to accept and process data.
     * @return A new FGate with the specified acceptor.
     */
    public static <TIn, TOut> FGate<TIn, TOut> of(AcceptorI<TOut, TIn> acceptor) {
        return new FGate<>(acceptor);
    }

    /**
     * Creates a new FGate with the specified link and acceptor.
     *
     * @param <TIn>  The type of data processed by the node on one side of the gate.
     * @param <TOut> The type of data output from the node through the gate.
     * @param link   The Gate instance to which this FGate will be linked.
     * @param acceptor The AcceptorI instance that defines how to accept and process data.
     * @return A new FGate with the specified link and acceptor.
     */
    public static <TIn, TOut> FGate<TIn, TOut> of(Gate<TOut, TIn> link, AcceptorI<TOut, TIn> acceptor) {
        var res = new FGate<>(acceptor);
        res.outSide().link(link);
        return res;
    }

    /**
     * Abstract class representing a pair of gates with an owner.
     *
     * @param <TIn>   The type of data processed by the node on one side of the gate.
     * @param <TOut>  The type of data output from the node through the gate.
     * @param <PairOut> The type of data output from the pair of gates.
     */
    public static abstract class Pair<TIn, TOut, PairOut> implements AcceptorI<TOut, TIn>, ObjectFind {
        final Object owner;

        /**
         * Constructs a new Pair with the specified owner.
         *
         * @param owner The owner of the pair.
         */
        public Pair(Object owner) {
            this.owner = owner;
        }

        /**
         * Returns a string representation of the pair.
         *
         * @return A string representation of the pair.
         */
        @Override
        public String toString() {
            return owner + " -> " + pair();
        }

        /**
         * Appends a string representation of the pair to the provided AString instance.
         *
         * @param sb The AString instance to which the string representation will be appended.
         */
        public void toString(AString sb) {
            sb.add(owner).add(" -> ").add(pair());
        }

        /**
         * Returns the InsideGate of the paired FGate.
         *
         * @return The InsideGate of the paired FGate.
         */
        @NotNull
        public abstract FGate<?, PairOut>.InsideGate pair();

        /**
         * Finds an object of the specified type within the pair.
         *
         * @param <T>  The type of the object to find.
         * @param type The class of the object to find.
         * @return An object of the specified type, or null if not found.
         */
        @Override
        public <T> T find(Class<T> type) {
            return pair().find(type);
        }

        /**
         * Finds an object of the specified type within the pair.
         *
         * @param <T>  The type of the object to find.
         * @param type The CType of the object to find.
         * @return An object of the specified type, or null if not found.
         */
        @Override
        public <T> T find(CTypeI<T> type) {
            return pair().find(type);
        }

    }

    /**
     * Represents the external side of the gate.
     */
    public class OutSideGate implements Gate<ToNode, ToOut> {

        /**
         * Returns the FGate instance to which this OutSideGate belongs.
         *
         * @return The FGate instance to which this OutSideGate belongs.
         */
        public FGate<ToNode, ToOut> getFGate() {
            return FGate.this;
        }

        /**
         * Appends a string representation of the acceptor to the provided AString instance.
         *
         * @param sb The AString instance to which the string representation will be appended.
         */
        @Override
        public void toString(AString sb) {
            acceptor.toString(sb);
        }

        /**
         * Returns a string representation of the acceptor.
         *
         * @return A string representation of the acceptor.
         */
        @Override
        public String toString() {
            return toString2();
        }

        /**
         * Finds an object of the specified type within the acceptor.
         *
         * @param <T>  The type of the object to find.
         * @param type The class of the object to find.
         * @return An object of the specified type, or null if not found.
         */
        @Override
        public <T> T find(Class<T> type) {
            return acceptor.find(type);
        }

        /**
         * Finds an object of the specified type within the acceptor.
         *
         * @param <T>  The type of the object to find.
         * @param type The CType of the object to find.
         * @return An object of the specified type, or null if not found.
         */
        @Override
        public <T> T find(CTypeI<T> type) {
            return acceptor.find(type);
        }

        /**
         * Unlinks this OutSideGate from its current link.
         */
        @Override
        public void unlink() {
            var l = ((FGate<?, ?>.OutSideGate) link);
            if (l == null) {
                return;
            }
            LINK_UPDATER.compareAndSet(l.getFGate(), outSide, null);
            LINK_UPDATER.compareAndSet(getFGate(), l, null);
        }

        /**
         * Returns the current link of this OutSideGate.
         *
         * @return The current link of this OutSideGate.
         */
        @Override
        public Gate<ToOut, ToNode> link() {
            return link;
        }

        @Override
        public void link0(Gate<ToOut, ToNode> g) {
            link = g;
        }

        /**
         * Links this OutSideGate to the specified side.
         *
         * @param side The {@link Gate} instance to which this OutSideGate will be linked.
         */
        @Override
        public void link(Gate<ToOut, ToNode> side) {
            var old1 = (FGate<ToOut, ToNode>.OutSideGate) link;
            if (old1 == side) {
                return;
            }
//            if (link != side) {
//                unlink();
//            }
            link = side;
            side.link0(outSide);
            send(Value.ofRequest());
            side.send(Value.ofRequest());
        }

        /**
         * Sends a value through the gate.
         *
         * @param value The Value instance to be sent.
         */
        @Override
        public void send(Value<ToNode> value) {
            value.enter(acceptor);
            acceptor.send(FGate.this, value);
        }
    }

    /**
     * Represents the internal side of the gate.
     */
    public class InsideGate implements GateI<ToOut>{

        /**
         * The FGate instance to which this InsideGate belongs.
         */
        public final FGate<ToNode, ToOut> fGate = FGate.this;

        /**
         * Returns a string representation of the link or "[NO LINK]" if no link is present.
         *
         * @return A string representation of the link or "[NO LINK]" if no link is present.
         */
        @Override
        public String toString() {
            var l = link;
            if (l != null) {
                return l.toString();
            } else {
                return "[NO LINK]";
            }
        }

        /**
         * Appends a string representation of the link or "[NO LINK]" to the provided AString instance.
         *
         * @param sb The AString instance to which the string representation will be appended.
         */
        @Override
        public void toString(AString sb) {
            var l = link;
            if (l != null) {
                sb.add(l);
            } else {
                sb.add("[NO LINK]");
            }
        }

        /**
         * Finds an object of the specified type within the link, or returns null if no link is present.
         *
         * @param <T>  The type of the object to find.
         * @param type The class of the object to find.
         * @return An object of the specified type, or null if not found or no link is present.
         */
        @Override
        public <T> T find(Class<T> type) {
            var l = link;
            if (l != null) {
                return l.find(type);
            } else {
                return null;
            }
        }

        /**
         * Finds an object of the specified type within the link, or returns null if no link is present.
         *
         * @param <T>  The type of the object to find.
         * @param type The CType of the object to find.
         * @return An object of the specified type, or null if not found or no link is present.
         */
        @Override
        public <T> T find(CTypeI<T> type) {
            var l = link;
            if (l != null) {
                return l.find(type);
            } else {
                return null;
            }
        }

        /**
         * Sends a value through the gate.
         *
         * @param value The Value instance to be sent.
         */
        @Override
        public void send(Value<ToOut> value) {
            value.enter(this);
            var l = link;
            if (l != null) {
                try {
                    l.send(value);
                } catch (Exception e) {
                    Log.error(e);
                    value.reject(this);
                }
            } else {
                value.reject(this);
            }
        }
    }

    //endregion static

}