package io.aether.utils.streams;

import io.aether.logger.Log;
import io.aether.net.fastMeta.FastApiContextLocal;
import io.aether.net.fastMeta.FastFutureContext;
import io.aether.net.fastMeta.FastMetaApi;
import io.aether.net.fastMeta.FastMetaType;
import io.aether.utils.AString;
import io.aether.utils.CTypeI;
import io.aether.utils.ConcurrentHashSet;
import io.aether.utils.RU;
import io.aether.utils.dataio.DataInOut;
import io.aether.utils.dataio.DataInOutStatic;
import io.aether.utils.dataio.DataOut;
import io.aether.utils.futures.AFuture;
import io.aether.utils.futures.ARFuture;
import io.aether.utils.interfaces.ABiConsumer;
import io.aether.utils.interfaces.AConsumer;
import io.aether.utils.interfaces.AFunction;
import io.aether.utils.interfaces.APredicate;

import java.util.Arrays;
import java.util.Queue;
import java.util.Set;

/**
 * The Gate interface represents a bidirectional communication channel between nodes in a reactive stream.
 * It provides methods for data transformation, filtering, buffering, and connecting to remote APIs.
 *
 * A Gate has two directions:
 * - Upstream (write): For sending data to the connected node
 * - Downstream (read): For receiving data from the connected node
 *
 * The interface supports various operations including:
 * - Data transformation (map, mapAsync)
 * - Buffering (buffer, bufferAutoFlush)
 * - Filtering (filter)
 * - Logging (log)
 * - Connecting to APIs (toApi, toMethod)
 * - Consumer/producer patterns (toConsumer, ofFunction)
 */
public interface Gate<TWrite, TRead> extends GateI<TWrite> {

    /**
     * Maps data types between the gate's current types and new types using MetaType information.
     * This is useful for serialization/deserialization scenarios.
     *
     * @param <LT> The new local type
     * @param <RT> The new remote type
     * @param metaLt Metadata describing the local type
     * @param metaRt Metadata describing the remote type
     * @return A new Gate with transformed data types
     */
    default <LT, RT> Gate<LT, RT> map(FastMetaType<LT> metaLt, FastMetaType<RT> metaRt) {
        var res = new NodeConverter<byte[], byte[], LT, RT>() {
            @Override
            public void toString(AString sb) {
                sb.add("(" + metaLt + ":" + metaRt + ")");
            }

            @Override
            protected RT toDownConverter(byte[] value) {
                return null;
            }

            @Override
            protected byte[] toUpConverter(LT value) {
                return null;
            }

            @Override
            protected Value<RT> toDownConverterValue(Value<byte[]> value) {
                if (!value.isData()) return RU.cast(value);
                return value.map(v -> {
                    var d = new DataInOutStatic(v);
                    var dd = metaRt.deserialize(FastFutureContext.STUB, d);
                    if (d.isReadable()) {
                        throw new IllegalStateException();
                    }
                    return dd;
                });
            }

            @Override
            protected Value<byte[]> toUpConverterValue(Value<LT> value) {
                return value.map(dd -> {
                    if (dd == null) return null;
                    DataInOut d = new DataInOut();
                    metaLt.serialize(FastFutureContext.STUB, dd, d);
                    return d.toArray();
                });
            }
        };
        res.up().link(RU.cast(this));
        return res.down();
    }

    /**
     * Unlinks this gate from its connected peer.
     * After unlinking, the gate cannot send or receive data until reconnected.
     */
    void unlink();

    /**
     * Gets the linked gate on the other side of this connection.
     * @return The linked gate, or null if not connected
     */
    Gate<TRead, TWrite> link();

    /**
     * Internal method for setting the link without notification.
     * @param g The gate to link to
     */
    void link0(Gate<TRead, TWrite> g);

    /**
     * Creates a buffered version of this gate.
     * The buffer will store incoming data when the downstream is not ready to process it.
     * @return A new buffered Gate
     */
    default Gate<TWrite, TRead> buffer() {
        return BufferNode.of(this).up();
    }

    /**
     * Inserts a buffer into the current gate connection if one doesn't already exist.
     */
    default void insertBuffer() {
        var old = (FGate<TRead, TWrite>.OutSideGate) link();
        var th = ((FGate<TWrite, TRead>.OutSideGate) this).getFGate();
        if (th.acceptor instanceof BufferNode.BGate) {
            return;
        }
        if (old != null && old.getFGate().acceptor instanceof BufferNode.BGate) {
            return;
        }
        BufferNode.of(this).up().link(old);
    }

    /**
     * Links this gate to another FGate instance.
     * @param side The FGate to link to
     */
    default void linkFGate(FGate<TRead, TWrite> side) {
        link(side.outSide());
    }

    /**
     * Links this gate to another gate instance.
     * @param side The gate to link to
     */
    void link(Gate<TRead, TWrite> side);

    default <TWrite2> Gate<TWrite, TWrite2> mapWrite(String name, AFunction<TRead, TWrite2> f1) {
        return map(name, f1, AFunction.stub());
    }

    default <TRead2> Gate<TRead2, TRead> mapRead(String name, AFunction<TRead2, TWrite> f2) {
        return map(name, AFunction.stub(), f2);
    }

    default <TWrite2> Gate<TWrite, TWrite2> mapAsyncWrite(String name, AFunction<TRead, ARFuture<TWrite2>> f1) {
        return mapAsync(name, f1, ARFuture::of);
    }

    default <TRead2> Gate<TRead2, TRead> mapAsyncRead(String name, AFunction<TRead2, ARFuture<TWrite>> f2) {
        return mapAsync(name, ARFuture::of, f2);
    }

    default <TWrite2, TRead2> Gate<TRead2, TWrite2> mapAsync(String name, AFunction<TRead, ARFuture<TWrite2>> f1, AFunction<TRead2, ARFuture<TWrite>> f2) {
        var res = new NodeConverterAsync<TRead, TWrite, TRead2, TWrite2>() {
            @Override
            public void toString(AString sb) {
                sb.add("mapAsync(").add(name).add(":").add(f1).add(':').add(f2).add(")");
            }

            @Override
            public String toString() {
                return toString2();
            }

            @Override
            protected ARFuture<TWrite2> toDownConverter(TRead value) {
                return f1.apply(value);
            }

            @Override
            protected ARFuture<TWrite> toUpConverter(TRead2 value) {
                return f2.apply(value);
            }
        };
        res.up().link(this);
        return res.down();
    }

    default <TWrite2, TRead2> Gate<TRead2, TWrite2> map(String name, AFunction<TRead, TWrite2> f1, AFunction<TRead2, TWrite> f2) {
        var res = new NodeConverter<TRead2, TWrite2, TRead, TWrite>() {
            @Override
            public void toString(AString sb) {
                var c1 = f1.getClass();
                var c2 = f2.getClass();
                sb.add("map(").add(name).add(":");
                var n = c1.getName();
                if (c1.isSynthetic()) {
                    sb.add(n.substring(0, n.indexOf("$")));
                } else {
                    sb.add(c1.getSimpleName());
                }
                sb.add(':');
                n = c2.getName();
                if (c2.isSynthetic()) {
                    sb.add(n.substring(0, n.indexOf("$")));
                } else {
                    sb.add(c2.getSimpleName());
                }
                sb.add(")");
            }

            @Override
            public String toString() {
                return toString2();
            }

            @Override
            protected TWrite toDownConverter(TRead2 value) {
                return f2.apply(value);
            }

            @Override
            protected TWrite2 toUpConverter(TRead value) {
                return f1.apply(value);
            }
        };
        res.down().link(this);
        return res.up();
    }

    default Gate<TWrite, TRead> bufferAutoFlush() {
        var n = new BufferNodeAutoFlush<TRead, TWrite>();
        n.down().link(this);
        return n.up();
    }

    default GateI<TWrite> ofFunction(String name, AFunction<TRead, TWrite> acceptor) {
        return ofBiConsumer(name, (v, r) -> r.send(v.map(acceptor)));
    }

    default GateI<TWrite> ofDataOut(String name, DataOut consumer) {
        return RU.<Gate<TWrite, byte[]>>cast(this).toConsumer(name, consumer::write);
    }

    default GateI<TWrite> ofFuture(String name, ARFuture<TRead> consumer) {
        return toConsumer(name, consumer::tryDone);
    }

    default GateI<TWrite> ofSideTryFuture(String name, ARFuture<TRead> consumer) {
        return toConsumer(name, consumer::tryDone);
    }

    /**
     * Links this gate to a consumer that will process the data received through this gate. The consumer will be called
     * whenever data is available on this gate, allowing you to perform operations on it.
     *
     * @param name       A descriptive name for the consumer, used for logging and debugging purposes.
     * @param consumer   A consumer that takes a single parameter: a value received through this gate. The consumer will be
     *                   called whenever data is available on this gate, allowing you to process it as needed.
     * @return An {@link GateI} instance representing the input side of the newly created link. This can be used to send data
     *                   through the linked consumer or to perform other operations on the gate.
     */
    default GateI<TWrite> toConsumer(String name, AConsumer<TRead> consumer) {
        FGate<TRead, TWrite> res = FGate.of(new OfConsumer<>(consumer, name));
        link(res.outSide());
        return res.inSide;
    }

    default GateI<TWrite> toConsumerValue(String name, AConsumer<Value<TRead>> consumer) {
        FGate<TRead, TWrite> res = FGate.of(new OfConsumerValue<>(name, consumer));
        link(res.outSide());
        return res.inSide;
    }

    default GateI<TWrite> ofBiConsumer(String name, ABiConsumer<Value<TRead>, GateI<TWrite>> consumer) {
        FGate<TRead, TWrite> res = FGate.of(new OfBiConsumer<>(name, consumer));
        link(res.outSide());
        return res.inSide;
    }

    default GateI<TWrite> toAcceptor(String name, AcceptorI<TWrite, TRead> g) {
        var gg = FGate.of(new AcceptorI<TWrite, TRead>() {
            public void send(FGate<TRead, TWrite> fGate, Value<TRead> value) {
                g.send(fGate, value);
            }

            public void setFGate(FGate<TRead, TWrite> gg) {
                g.setFGate(gg);
            }

            public void toString(AString sb) {
                sb.add(name).add("(");
                g.toString(sb);
                sb.add(")");
            }

            public String toString2() {
                return g.toString2();
            }

            public <T> T find(Class<T> type) {
                return g.find(type);
            }

            public Object[] getOwners() {
                return g.getOwners();
            }

            public <T> T find(CTypeI<T> type) {
                return g.find(type);
            }
        });
        link(gg.outSide());
        return gg.inSide;
    }

    default GateI<TWrite> ofCollection(String name, Queue<TRead> q) {
        return toConsumer(name, q::add);
    }


    default Gate<TWrite, TRead> distinctWrite(String name) {
        Set<TWrite> oldData = new ConcurrentHashSet<>();
        return filter(name, APredicate.TRUE_STUB(), oldData::add);
    }

    default Gate<TWrite, TRead> distinctRead(String name) {
        Set<TRead> oldData = new ConcurrentHashSet<>();
        return filter(name, oldData::add, APredicate.TRUE_STUB());
    }

    default Gate<TWrite, TRead> filter(String name, APredicate<TRead> filterRead, APredicate<TWrite> filterWrite) {
        var n = new NodeFilter<TRead, TWrite>(name) {
            @Override
            public void toString(AString sb) {
                sb.add("filter(").add(filterRead).add(':').add(filterWrite).add(")");
            }

            @Override
            protected boolean toDownFilter(TRead value) {
                return filterRead.test(value);
            }

            @Override
            protected boolean toUpFilter(TWrite value) {
                return filterWrite.test(value);
            }
        };
        link(n.up());
        return n.down();
    }

    default Gate<TWrite, TRead> log(String title, String directLeft, String directRight, AConsumer<Value<TWrite>> directLeftFun, AConsumer<Value<TRead>> directRightFun) {
        var res = new NodeConverter<TWrite, TRead, TRead, TWrite>() {
            @Override
            public void toString(AString sb) {
                sb.add("loggerWithFun(").add(title).add(")");
            }

            @Override
            public String toString() {
                return toString2();
            }

            @Override
            protected TWrite toDownConverter(TWrite value) {
                return value;
            }

            @Override
            protected TRead toUpConverter(TRead value) {
                return value;
            }

            private void printLog(String direction, Object val) {
                if (val == null) return;
                if (val instanceof byte[]) {
                    var v = (byte[]) val;
                    if (v.length > 10) {
                        val = Arrays.copyOf(v, 10);
                    }
                }
                Log.debug("stream log [$streamLogTitle] $direction <- $data", "streamLogTitle", title, "direction", direction, "data", val);
            }

            @Override
            protected Value<TWrite> toDownConverterValue(Value<TWrite> value) {
                printLog(directLeft, value.data());
                directLeftFun.accept(value);
                return value;
            }

            @Override
            protected Value<TRead> toUpConverterValue(Value<TRead> value) {
                printLog(directRight, value.data());
                directRightFun.accept(value);
                return value;
            }
        };
        res.down().link(this);
        return res.up();
    }

    default Gate<TWrite, TRead> log(String title, String directLeft, String directRight) {
        var res = new NodeConverter<TWrite, TRead, TRead, TWrite>() {

            @Override
            public void toString(AString sb) {
                sb.add("logger(").add(title).add(")");
            }

            @Override
            public String toString() {
                return toString2();
            }

            private void printLog(String direction, Object val) {
                if (val == null) return;
                if (val instanceof byte[]) {
                    var v = (byte[]) val;
                    if (v.length > 10) {
                        val = Arrays.copyOf(v, 10);
                    }
                }
                Log.debug("stream log [$streamLogTitle] $direction <- $data", "streamLogTitle", title, "direction", direction, "data", val);
            }

            @Override
            protected TWrite toDownConverter(TWrite value) {
                return value;
            }

            @Override
            protected TRead toUpConverter(TRead value) {
                return value;
            }

            @Override
            protected Value<TWrite> toDownConverterValue(Value<TWrite> value) {
                printLog(directLeft, value.data());
                return value;
            }

            @Override
            protected Value<TRead> toUpConverterValue(Value<TRead> value) {
                printLog(directRight, value.data());
                return value;
            }
        };
        res.down().link(this);
        return res.up();
    }
}
