package io.aether.utils.streams;

import io.aether.utils.ConcurrentHashSet;
import io.aether.utils.RU;
import io.aether.utils.flow.Flow;
import io.aether.utils.interfaces.ABiFunction;
import io.aether.utils.interfaces.AConsumer;
import io.aether.utils.interfaces.AFunction;
import io.aether.utils.slots.AMFuture;

import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

public class MapBase<K, V> implements AConsumer<V> {
    private final MSwitcherOutput output = new MSwitcherOutput();
    private final MSwitcherInput input = new MSwitcherInput();
    final AtomicInteger futuresWithoutRequest = new AtomicInteger();
    private final ConcurrentHashMap<K, ARMultiFuture2<V>> cache = new ConcurrentHashMap<>();
    private final AFunction<V, K> keyGetter;
    private final ABiFunction<V, V, V> updater;
    boolean withLog;
    String logName;

    private interface SplitterUpGate2<KK> {
        Set<KK> getOld();
    }

    private class MSwitcherInput extends Switcher2<V, K> {

        public Gate<K, V> linkUpHard() {
            FGate<K, V> gInner = FGate.of(new SplitterUpGateHard2());
            return gInner.outSide();
        }

        public Gate<K, V> linkUp() {
            FGate<K, V> gInner = FGate.of(new SplitterUpGateSoft2());
            return gInner.outSide();
        }

        @Override
        protected void sendToDown2(Switcher2<V, K>.SplitterUpGate sup, FGate<K, V> fGate, Value<K> value) {
            var s = RU.<SplitterUpGate2<K>>cast(sup);
            if (value.isData()) {
                var key = value.data();
                var f = MapBase.this.get(key);
                if (f.isDone()) {
                    if (s.getOld().add(key)) {
                        fGate.inSide.send(Value.of(f.getNow()).onReject((o, id) -> {
                            s.getOld().remove(key);
                        }));
                    }
                }
            }
        }

        @Override
        protected void onFirstRequest(Switcher2<V, K>.SplitterUpGate sup, FGate<K, V> fGate, Value<K> value) {
            var s = RU.<SplitterUpGate2<K>>cast(sup);
            for (var e : cache.entrySet()) {
                if (e.getValue().isDone()) {
                    if (s.getOld().add(e.getKey())) {
                        fGate.inSide.send(new ValueOfData<>(e.getValue().getNow()) {
                            @Override
                            public void reject(Object owner, long blockId) {
                                s.getOld().remove(e.getKey());
                            }
                        });
                    }
                }
            }
        }

        @Override
        protected void sendUp(Value<V> value, FGate<K, V> g, AtomicInteger abortCounter, AtomicBoolean finalFlag) {
            var sup = RU.<SplitterUpGate2<K>>cast(g.acceptor);
            if (value.isData()) {
                K key = keyGetter.apply(value.data());
                if (!sup.getOld().add(key)) {
                    return;
                }
            }
            super.sendUp(value, g, abortCounter, finalFlag);
        }

        class SplitterUpGateSoft2 extends SplitterUpGateSoft implements SplitterUpGate2<K> {
            final Set<K> old = new ConcurrentHashSet<>();

            @Override
            public Set<K> getOld() {
                return old;
            }
        }

        class SplitterUpGateHard2 extends SplitterUpGateHard implements SplitterUpGate2<K> {
            final Set<K> old = new ConcurrentHashSet<>();

            @Override
            public Set<K> getOld() {
                return old;
            }
        }
    }

    private class MSwitcherOutput extends Switcher2<K, V> {
        @Override
        protected void onFirstRequest(Switcher2<K, V>.SplitterUpGate sup, FGate<V, K> fGate, Value<V> value) {
            var s = RU.<SplitterUpGate2<K>>cast(sup);
            for (var e : cache.entrySet()) {
                if (!e.getValue().isDone()) {
                    if (s.getOld().add(e.getKey())) {
                        fGate.inSide.send(new ValueOfData<>(e.getKey()) {
                            @Override
                            public void reject(Object owner, long blockId) {
                                s.getOld().remove(e.getKey());
                            }
                        });
                    }
                }
            }
        }

        public Gate<V, K> linkUpHard() {
            FGate<V, K> gInner = FGate.of(new SplitterUpGateHard2());
            return gInner.outSide();
        }

        public Gate<V, K> linkUp() {
            FGate<V, K> gInner = FGate.of(new SplitterUpGateSoft2());
            return gInner.outSide();
        }

        @Override
        protected void sendToDown2(Switcher2<K, V>.SplitterUpGate sup, FGate<V, K> fGate, Value<V> value) {
            if (value.isRequestData()) {
                for (var e : cache.entrySet()) {
                    if (output.blockMgrGlobal.isBlocked()) continue;
                    if (e.getValue().isDone()) continue;
                    if (e.getValue().tryRequest()) {
                        fGate.inSide.send(Value.of(e.getKey()).onReject((owner, id) -> {
                            if (!e.getValue().isDone()) {
                                e.getValue().request.set(false);
                            }
                        }));
                    }
                }
            }
            if (value.isData()) {
                MapBase.this.set(value.data());
            }
            value.success(MapBase.this);
        }

        @Override
        protected void sendUp(Value<K> value, FGate<V, K> g, AtomicInteger abortCounter, AtomicBoolean finalFlag) {
            var sup = RU.<SplitterUpGate2<K>>cast(g.acceptor);
            if (value.isData() && !sup.getOld().add(value.data())) {
                return;
            }
            super.sendUp(value, g, abortCounter, finalFlag);
        }

        class SplitterUpGateSoft2 extends SplitterUpGateSoft implements SplitterUpGate2<K> {
            final Set<K> old = new ConcurrentHashSet<>();

            @Override
            public Set<K> getOld() {
                return old;
            }
        }

        class SplitterUpGateHard2 extends SplitterUpGateHard implements SplitterUpGate2<K> {
            final Set<K> old = new ConcurrentHashSet<>();

            @Override
            public Set<K> getOld() {
                return old;
            }
        }
    }

    public MapBase(AFunction<V, K> keyGetter) {
        this(keyGetter, (v, v2) -> v2);
    }

    public MapBase(AFunction<V, K> keyGetter, ABiFunction<V, V, V> updater) {
        this.keyGetter = keyGetter;
        this.updater = updater;
    }

    public void refreshKey(K key) {
        cache.get(key).refresh();
    }

    public void refresh(V c) {
        K key = keyGetter.apply(c);
        cache.get(key).refresh();
    }

    public Gate<K, V> addInputHard() {
        return input.linkUpHard();
    }

    public Gate<K, V> addInput() {
        return input.linkUp();
    }

    public Gate<K, V> addInputByRequestHard() {
        return addInputHard().distinctWrite("MapBase(" + logName + ") addInputByRequestHard");
    }

    @Override
    public void accept2(V v) {
        set(v);
    }

    @Override
    public void accept(V v) {
        set(v);
    }

    public void set(V val) {
        set0(keyGetter.apply(val), val);
    }

    public void setAll(Iterable<V> val) {
        for (var e : val) {
            set(e);
        }
    }

    public void set(K key, V val) {
        if (!keyGetter.apply(val).equals(key)) {
            throw new IllegalStateException();
        }
        set0(key, val);
    }

    protected void set0(K key, V val) {
        var f = cache.computeIfAbsent(key, this::makeFuture);
        if (f.getNow() == null) {
            f.set(val);
        } else {
            f.set(updater.apply(f.getNow(), val));
            f.refresh();
        }
    }

    public MapBase<K, V> withLog() {
        this.withLog = true;
        return this;
    }

    public <K2, V2> MapBase<K2, V2> map(
            String name, AFunction<V, V2> valConverter1,
            AFunction<K2, K> keyConverter2,
            AFunction<V2, K2> keyConverter3
    ) {
        var res = new MapBase<>(keyConverter3).withLog(name);
        res.addSourceHard(addInputHard().map("MapBase (" + logName + "->" + name + ") remap", valConverter1, keyConverter2));
        return res;
    }

    public AMFuture<V> get(K key, AFunction<K, V> orCreate) {
        var res = cache.computeIfAbsent(RU.cast(key), k -> {
            var res2 = makeFuture(k);
            res2.set(orCreate.apply(k));
            return res2;
        });
        if (!res.isDone()) {
            if (res.tryRequest()) {
                output.sendUp(Value.of(key));
            }
        }
        return res;
    }

    private ARMultiFuture2<V> makeFuture(K k) {
        return new MyARMultiFuture2();
    }

    public AMFuture<V> get(K key) {
        var res = cache.computeIfAbsent(RU.cast(key), this::makeFuture);
        if (!res.isDone()) {
            if (res.tryRequest()) {
                output.sendUp(new ValueOfData<>(key) {
                    @Override
                    public void reject(Object owner, long blockId) {
                        res.request.set(false);
                    }
                });
            }
        }
        return res;
    }

    public void flush() {
        for (var e : cache.entrySet()) {
            if (e.getValue().isDone()) continue;
            output.sendUp(new ValueOfData<>(e.getKey()) {
                @Override
                public void reject(Object owner, long blockId) {
                }
            });
        }
        output.flushUp();
        input.flushUp();
    }

    public Flow<V> values() {
        return Flow.flow(cache.values()).map(AMFuture::getNow).filterNotNull();
    }

    public void addSource(Gate<K, V> src) {
        output.linkUp(src.buffer());
    }

    public void addSourceHard(Gate<K, V> src) {
        output.linkUpHard(src.buffer());
    }

    public Gate<V, K> addSourceHard() {
        return output.linkUpHard().buffer();
    }

    public MapBase<K, V> withLog(String logName) {
        this.logName = logName;
        return withLog();
    }

    public static class ARMultiFuture2<T> extends AMFuture<T> {
        private final AtomicBoolean request = new AtomicBoolean(false);
        private final AtomicInteger futuresWithoutRequest;

        public ARMultiFuture2(AtomicInteger futuresWithoutRequest) {
            this.futuresWithoutRequest = futuresWithoutRequest;
        }

        public boolean tryRequest() {
            var res = request.compareAndSet(false, true);
            futuresWithoutRequest.decrementAndGet();
            return res;
        }
    }


    private class MyARMultiFuture2 extends ARMultiFuture2<V> {

        public MyARMultiFuture2() {
            super(futuresWithoutRequest);
        }

        @Override
        public boolean set(V value) {
            var vv = this.value;
            if (vv == null) {
                if (UPDATER.compareAndSet(this, null, value)) {
                    eventConsumer.fire(value);
                    BlockMgr blockMgr = new BlockMgr();
                    input.sendUp(new ValueOfData<>(value) {
                        @Override
                        public void reject(Object owner, long blockId) {
                            if (!blockMgr.block(blockId)) {
                                input.sendUp(this);
                            }
                        }
                    });
                    return true;
                } else {
                    return set(value);
                }
            } else if (vv == value) {
                return false;
            }
            value = updater.apply(vv, value);
            this.value = value;
            eventConsumer.fire(value);
            input.sendUp(new ValueOfData<>(value) {
                @Override
                public void reject(Object owner, long blockId) {
//                    super.abort(owner);
                }
            });
            return true;
        }

        @Override
        public void refresh() {
            super.refresh();
            var v = getNow();
            if (v != null) {
                input.sendUp(new ValueOfData<>(v) {
                    @Override
                    public void reject(Object owner, long blockId) {

                    }
                });
            }
        }
    }
}
