package io.aether.utils.streams;

import io.aether.utils.RU;
import org.jetbrains.annotations.NotNull;

public abstract class NodeConverter<TDown2, TUp2, TUp, TDown>
        implements Node<TDown2, TUp2, TUp, TDown> {
    public NodeConverter() {
    }

    @Override
    public FGate<TDown2, TUp2> gUp() {
        return up;
    }

    @Override
    public FGate<TUp, TDown> gDown() {
        return down;
    }

    protected abstract TDown toDownConverter(TDown2 value);

    protected abstract TUp2 toUpConverter(TUp value);

    protected Value<TDown> toDownConverterValue(Value<TDown2> value) {
        return value.map(this::toDownConverter);
    }

    protected Value<TUp2> toUpConverterValue(Value<TUp> value) {
        return value.map(this::toUpConverter);
    }

    @Override
    public String toString() {
        return "Node converter";
    }

    private final FGate<TDown2, TUp2> up =
            FGate.of(new FGate.Pair<TDown2, TUp2, TDown>(this) {
                @Override
                public FGate<?, TDown>.@NotNull InsideGate pair() {
                    return down.inSide;
                }

                @Override
                public void send(FGate<TDown2, TUp2> fGate, Value<TDown2> value) {
                    if (value.isData()) {
                        if (pair().fGate.outSide().link() == null) {
                            value.reject(pair());
                            return;
                        }
                        var v = toDownConverterValue(value);
                        if (v != null) {
                            pair().send(v);
                        }
                    } else {
                        pair().send(RU.cast(value));
                    }
                }

            });


    private final FGate<TUp, TDown> down = FGate.of(new FGate.Pair<TUp, TDown, TUp2>(this) {

        @Override
        public FGate<?, TUp2>.@NotNull InsideGate pair() {
            return up.inSide;
        }

        @Override
        public void send(FGate<TUp, TDown> fGate, Value<TUp> value) {
            if (pair().fGate.outSide().link() == null) {
                value.reject(pair());
                return;
            }
            if (value.isOnlyRequestData()) {
                pair().send(RU.cast(value));
                return;
            }
            if (value.isData()) {
                var v = toUpConverterValue(value);
                if (v != null) {
                    pair().send(v);
                }
            } else if (value.isForce()) {
                pair().send(RU.cast(value));
            }
        }
    });


}
