package io.aether.utils.streams;

import io.aether.logger.Log;
import io.aether.utils.RU;
import io.aether.utils.ToString;
import io.aether.utils.futures.AFuture;
import io.aether.utils.interfaces.AConsumer;
import io.aether.utils.interfaces.AFunction;
import org.jetbrains.annotations.NotNull;

import java.util.Collection;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicLong;

/**
 * Interface representing a value that can be processed in a stream-like manner in reactive programming.
 * Values act as data containers that flow through processing pipelines, with lifecycle management capabilities.
 *
 * <p>Key features:
 * <ul>
 *   <li>Lifecycle management (enter/reject/success)</li>
 *   <li>Priority-based processing</li>
 *   <li>Special value types (force, request, close)</li>
 *   <li>Timeout handling</li>
 *   <li>Data transformation capabilities</li>
 * </ul>
 *
 * @param <T> The type of data contained in this value
 */
public interface Value<T> extends ToString {
    AtomicLong BLOCK_COUNTER = new AtomicLong();
    // Static instances for special values
    Value<?> FORCE_INSTANCE = new ValueForce();
    Value<?> CLOSE_INSTANCE = new ValueClose();
    long TIMEOUT = 800000L; // Default timeout for value processing (8 seconds)

    /**
     * Retrieves the data payload contained in this value.
     * The actual data type is determined by the generic parameter T.
     *
     * @return The contained data object, which may be null for special values
     */
    T data();

    /**
     * Returns the processing priority of this value.
     * Higher priority values should be processed before lower priority ones.
     * Default implementation returns 0 (normal priority).
     *
     * @return Priority level as an integer (higher = more important)
     */
    default int priority() {
        return 0;
    }

    /**
     * Checks if this value is marked as "force".
     * Force values typically bypass normal processing constraints and
     * must be handled immediately regardless of system state.
     *
     * @return true if this is a force value, false otherwise
     */
    default boolean isForce() {
        return false;
    }

    /**
     * Determines if this value represents a data request.
     * Request values typically ask for data to be generated/sent rather than
     * containing data themselves.
     *
     * @return true if this is a request value, false otherwise
     */
    default boolean isRequestData() {
        return false;
    }

    default long getRequestDataId() {
        return 0;
    }

    /**
     * Checks if this value signals stream/processing closure.
     * Close values typically indicate the end of a data stream or processing sequence.
     *
     * @return true if this is a close signal, false otherwise
     */
    default boolean isClose() {
        return false;
    }

    /**
     * Releases resources associated with this value when it's no longer needed.
     * This is the normal end-of-life operation for a value.
     *
     * @param owner The object initiating the success operation (for tracking/logging)
     */
    void success(Object owner);

    /**
     * Marks the beginning of value processing by a handler.
     * Called when a processor starts working with this value.
     *
     * @param owner The object that begins processing this value
     */
    void enter(Object owner);

    /**
     * Rejects the value sending process due to a stream being blocked.
     *
     * @param owner  An object that provides information about the current state of the block.
     *                <p>
     *                When the creator of the Value object receives an reject call, it understands that the stream is currently blocked and data cannot be sent at this moment.
     *                <p>
     *                If, at the same time, a Value object with the isRequestData flag arrives from the stream, a race condition can occur. To resolve this issue, the checkWritable method should be called to determine the actual state of the remote node.
     * @param blockId
     */
    void reject(Object owner, long blockId);

    default void reject(Object owner) {
        reject(owner, BLOCK_COUNTER.incrementAndGet());
    }

    /**
     * Sets a processing timeout for this value.
     * If the value isn't fully processed (either succeed or rejected) within the specified time,
     * a warning will be logged including information about which processors entered the value.
     *
     * @param time Timeout duration in milliseconds
     * @return A new Value instance with timeout tracking behavior
     */
    default Value<T> timeout(long time) {
        return new ValueProxyTimeout<>(this,time);
    }

    /**
     * Enhanced timeout with custom action when timeout occurs.
     * Similar to timeout(long) but executes the provided task when timeout occurs.
     *
     * @param time Timeout duration in milliseconds
     * @param task Action to execute when timeout occurs, receives enter tracking queue
     * @return A new Value instance with timeout tracking behavior
     */
    default Value<T> timeout(long time, AConsumer<ConcurrentLinkedQueue<Object>> task) {
        var lastEnter = new ConcurrentLinkedQueue<>();
        var self = this;
        var sch = RU.schedule(time, () -> {
            Log.warn("timeout value completed [$enter]", "enter", lastEnter, "value", self);
            task.accept(lastEnter);
        });
        return onReject((o, id) -> sch.cancel(false))
                .onEnter(lastEnter::add)
                .onSuccess(o -> sch.cancel(false));
    }

    /**
     * Attaches a handler to be called when this value enters processing.
     * The original value remains unchanged - returns a new wrapper value.
     *
     * @param f Handler to call on enter
     * @return New Value instance with enter handler
     */
    default Value<T> onEnter(AConsumer<Object> f) {
        var self = this;
        return new ValueOnEnter<>(self, f);
    }

    /**
     * Transforms this value into a new value with different data.
     * Useful for creating derivative values while maintaining the original's lifecycle.
     *
     * @param val The new data to wrap
     * @return New Value instance with different data type
     */
    default <T2> Value<T2> map2(T2 val) {
        var self = this;
        return new ValueMap2<>(self, val);
    }

    /**
     * Transforms the contained data using a mapping function.
     * The mapping occurs lazily when data() is first called on the returned value.
     *
     * @param mapper Function to transform the contained data
     * @return New Value instance with transformed data
     */
    default <T2> Value<T2> map(AFunction<T, T2> mapper) {
        var self = this;
        return new ValueMap<>(self, mapper);
    }

    /**
     * Attaches an exclusive reject handler that replaces default reject behavior.
     * Only this handler will be called when the value is rejected.
     *
     * @param task Handler to call exclusively on reject
     * @return New Value instance with exclusive reject handler
     */
    default Value<T> linkOnRejectExclusive(ValueExclusiveOnReject.Listener task) {
        var self = this;
        return new ValueExclusiveOnReject<>(self, task);
    }

    /**
     * Adds an reject handler to be called when this value is rejected.
     * The handler runs in addition to the default reject behavior.
     *
     * @param task Handler to call on reject
     * @return New Value instance with additional reject handler
     */
    default Value<T> onReject(ValueOnReject.Listener task) {
        var self = this;
        var l = Log.get();
        return new ValueOnReject<>(self, (o, id) -> {
            Log.push(l);
            try {
                task.accept(o, id);
            } finally {
                Log.pop(l);
            }
        });
    }

    /**
     * Adds a success handler to be called when this value is succeed.
     * The handler runs in addition to the default success behavior.
     *
     * @param task Handler to call on success
     * @return New Value instance with additional success handler
     */
    default Value<T> onSuccess(AConsumer<Object> task) {
        var self = this;
        return new ValueOnDrop<>(self, Log.wrap(task));
    }

    /**
     * Checks if this value contains processable data.
     * Returns false for control values (force, request, close).
     *
     * @return true if this value contains data, false otherwise
     */
    boolean isData();

    /**
     * Creates a new value with the same data but marked as "force".
     * If this value is already a force value, returns itself.
     *
     * @return Force-marked Value instance
     */
    default Value<T> withForce() {
        if (isForce()) return this;
        var self = this;
        return new ValueWithForce<>(self);
    }

    /**
     * Creates a new value with the same data but marked as "close".
     * If this value is already a close value, returns itself.
     *
     * @return Close-marked Value instance
     */
    default Value<T> withClose() {
        if (isClose()) return this;
        var self = this;
        return new ValueWithClose<>(self);
    }

    /**
     * Creates a new value with the same data but not marked as "close".
     * If this value is already not a close value, returns itself.
     *
     * @return Not-close-marked Value instance
     */
    default Value<T> notClose() {
        if (!isClose()) return this;
        var self = this;
        return new ValueNotClose<>(self);
    }

    /**
     * Creates a new value with request semantics added.
     * Useful for creating values that both contain data and request more data.
     *
     * @return New Value instance with request flag set
     */
    @NotNull
    default Value<T> addRequest() {
        var self = this;
        return new ValueAddRequest<>(self);
    }

    default boolean isOnlyRequestData() {
        return isRequestData() && !isData() && !isClose() && !isForce();
    }

    default Value<T> linkFuture(AFuture res){
        return new ValueProxy<>(this){
            @Override
            public void success(Object owner) {
                res.done();
                super.success(owner);
            }

            @Override
            public void reject(Object owner) {
                res.cancel();
                super.reject(owner);
            }
        };
    }

    /**
     * Creates a new value with data, force flag, and sub-values.
     * Sub-values allow creating hierarchical value structures.
     *
     * @param data The primary data payload
     * @param force Whether this value should be force-processed
     * @param subValues Collection of child values
     * @return New Value instance with timeout applied
     */
    static <T> Value<T> of(T data, boolean force, Collection<Value<?>> subValues) {
        return new ValueWithSubValues<>(data, force, subValues).timeout(TIMEOUT);
    }

    /**
     * Creates a simple data-carrying value.
     *
     * @param data The data payload
     * @return New Value instance with timeout applied
     */
    static <T> Value<T> of(T data) {
        return new ValueOfData<>(data).timeout(TIMEOUT);
    }

    /**
     * Creates a force-marked data-carrying value.
     *
     * @param data The data payload
     * @return New force-marked Value instance with timeout applied
     */
    static <T> Value<T> ofForce(T data) {
        return new ValueOfForce<>(data).timeout(TIMEOUT);
    }

    /**
     * Creates a data-carrying value with success handler.
     *
     * @param data The data payload
     * @param onDrop Handler to call when value is succeed
     * @return New Value instance with success handler and timeout
     */
    static <T> Value<T> of(T data, AConsumer<Object> onDrop) {
        return new ValueOfOnDrop<>(data, onDrop).timeout(TIMEOUT);
    }

    /**
     * Creates a value with data, force flag, and success handler.
     *
     * @param data The data payload
     * @param force Whether this value should be force-processed
     * @param onDrop Handler to call when value is succeed
     * @return New Value instance with specified properties and timeout
     */
    static <T> Value<T> of(T data, boolean force, AConsumer<Object> onDrop) {
        return new ValueOfDataForceOnDrop<>(data, onDrop, force).timeout(TIMEOUT);
    }

    /**
     * Creates a force-marked value with data and success handler.
     *
     * @param data The data payload
     * @param onDrop Handler to call when value is succeed
     * @return New force-marked Value instance with success handler and timeout
     */
    static <T> Value<T> ofForce(T data, AConsumer<Object> onDrop) {
        return new ValueOfForceOnDrop<>(data, onDrop).timeout(TIMEOUT);
    }

    /**
     * Creates a generic force-marked value without data.
     *
     * @return Singleton force value instance
     */
    static <K> Value<K> ofForce() {
        return RU.cast(FORCE_INSTANCE);
    }

    /**
     * Creates a data request value.
     *
     * @return Singleton request value instance
     */
    @NotNull
    static <T> Value<T> ofRequest() {
        return ofRequest(BLOCK_COUNTER.incrementAndGet());
    }

    @NotNull
    static <T> Value<T> ofRequest(long id) {
        return new ValueOfRequest<>(id);
    }

    /**
     * Creates a stream close value.
     *
     * @return Singleton close value instance
     */
    static <T> Value<T> ofClose() {
        return RU.cast(CLOSE_INSTANCE);
    }

}