package io.aether.api.common;

import io.aether.crypto.*;
import io.aether.utils.RU;

public interface KeyUtil {
    byte[] getData();

    static <T extends AKey> T of(Key key) {
        if (key instanceof SodiumCurvePrivate) {
            return CryptoProviderFactory.getProvider("SODIUM").createKey(KeyType.ASYMMETRIC_PRIVATE, key.getData());
        } else if (key instanceof SodiumSignPublic) {
            return CryptoProviderFactory.getProvider("SODIUM").createKey(KeyType.SIGN_PUBLIC, key.getData());
        } else if (key instanceof SodiumSignPrivate) {
            return CryptoProviderFactory.getProvider("SODIUM").createKey(KeyType.SIGN_PRIVATE, key.getData());
        } else if (key instanceof SodiumCurvePublic) {
            return CryptoProviderFactory.getProvider("SODIUM").createKey(KeyType.ASYMMETRIC_PUBLIC, key.getData());
        } else if (key instanceof HydrogenCurvePrivate) {
            return CryptoProviderFactory.getProvider("HYDROGEN").createKey(KeyType.ASYMMETRIC_PRIVATE, key.getData());
        } else if (key instanceof HydrogenCurvePublic) {
            return CryptoProviderFactory.getProvider("HYDROGEN").createKey(KeyType.ASYMMETRIC_PUBLIC, key.getData());
        } else if (key instanceof HydrogenSignPublic) {
            return CryptoProviderFactory.getProvider("HYDROGEN").createKey(KeyType.SIGN_PUBLIC, key.getData());
        } else if (key instanceof HydrogenSignPrivate) {
            return CryptoProviderFactory.getProvider("HYDROGEN").createKey(KeyType.SIGN_PRIVATE, key.getData());
        } else if (key instanceof SodiumChacha20Poly1305) {
            return CryptoProviderFactory.getProvider("SODIUM").createKey(KeyType.SYMMETRIC, key.getData());
        } else if (key instanceof HydrogenSecretBox) {
            return CryptoProviderFactory.getProvider("HYDROGEN").createKey(KeyType.SYMMETRIC, key.getData());
        } else {
            throw new UnsupportedOperationException();
        }
    }

    static CryptoEngine makeProvider(Key key) {
        var k = of(key);
        switch (k.getKeyType()) {
            case ASYMMETRIC_PUBLIC:
                return ((AKey.AsymmetricPublic) k).toCryptoEngine();
            case SYMMETRIC:
                return ((AKey.Symmetric) k).toCryptoEngine();
            default:
                throw new UnsupportedOperationException();
        }
    }

    static <T extends Key> T of(AKey key) {
        switch (key.getProviderName().toLowerCase()) {
            case "sodium":
                switch (key.getKeyType()) {
                    case SYMMETRIC:
                        return RU.cast(new SodiumChacha20Poly1305(key.getData()));
                    case ASYMMETRIC_PRIVATE:
                        return RU.cast(new SodiumCurvePrivate(key.getData()));
                    case ASYMMETRIC_PUBLIC:
                        return RU.cast(new SodiumCurvePublic(key.getData()));
                    case SIGN_PRIVATE:
                        return RU.cast(new SodiumSignPrivate(key.getData()));
                    case SIGN_PUBLIC:
                        return RU.cast(new SodiumSignPublic(key.getData()));
                    default:
                        throw new UnsupportedOperationException();
                }
            case "hydrogen":
                switch (key.getKeyType()) {
                    case SYMMETRIC:
                        return RU.cast(new HydrogenSecretBox(key.getData()));
                    case ASYMMETRIC_PRIVATE:
                        return RU.cast(new HydrogenCurvePrivate(key.getData()));
                    case ASYMMETRIC_PUBLIC:
                        return RU.cast(new HydrogenCurvePublic(key.getData()));
                    case SIGN_PRIVATE:
                        return RU.cast(new HydrogenSignPrivate(key.getData()));
                    case SIGN_PUBLIC:
                        return RU.cast(new HydrogenSignPublic(key.getData()));
                    default:
                        throw new UnsupportedOperationException();
                }
        }
        throw new UnsupportedOperationException();
    }

}
