package io.aether.crypto;



/**
 * A class representing a cryptographic key with its corresponding digital signature.
 * The functionality of this class is independent of the underlying crypto library.
 */
public class SignedKey implements CryptoProviderUnit{
    public final AKey key;
    public final Sign sign;

    public SignedKey(AKey key, Signer signer) {
        this.key=key;
        sign = signer.sign(key.getData());
    }

    @Override
    public String getProviderName() {
        return key.getProviderName();
    }

    @Override
    public CryptoProvider getCryptoProvider() {
        return key.getCryptoProvider();
    }

    public SignedKey(AKey key, Sign sign) {
        assert key != null;
        assert sign != null;
        this.key = key;
        this.sign = sign;
    }

    public static SignedKey of(AKey key, Sign sign) {
        return new SignedKey(key, sign);
    }

    public AKey key() {
        return key;
    }

    public Sign sign() {
        return sign;
    }

    /**
     * Checks if the key's signature is valid using a Signer.
     * The Signer object is responsible for the actual verification process.
     *
     * @param signer The Signer object to perform the verification.
     * @return true if the signature is valid, false otherwise.
     */
    public boolean check(AKey.SignPublic signer) {
        return signer.toSignChecker().checkSign(key.getData(), sign);
    }
    public boolean check(SignChecker signChecker) {
        return signChecker.checkSign(key.getData(), sign);
    }

    @Override
    public String toString() {
        return key + ":" + sign;
    }
}