package io.aether.utils.streams;

import io.aether.logger.Log;
import io.aether.utils.AString;
import io.aether.utils.CTypeI;
import io.aether.utils.ConcurrentPriorityQueue;
import io.aether.utils.RU;
import io.aether.utils.flow.Flow;
import io.aether.utils.interfaces.ObjectFind;
import org.jetbrains.annotations.NotNull;

import java.util.Queue;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * Represents a node in a stream that buffers incoming data.
 *
 * @param <TWrite> The type of data written to the buffer.
 * @param <TRead>  The type of data read from the buffer.
 */
public class BufferNode<TRead, TWrite> implements Node<TWrite, TRead, TRead, TWrite>, ObjectFind {
    private final FGate<TWrite, TRead> up;
    private final FGate<TRead, TWrite> down;

    /**
     * Constructs a new {@link BufferNode} with initialized up and down gates.
     */
    public BufferNode() {
        up = new FGate<>(initUp());
        down = new FGate<>(initDown());
    }

    /**
     * Checks if both the up and down buffers are empty.
     *
     * @return true if both buffers are empty, false otherwise
     */
    public boolean isEmpty() {
        return isEmptyToUp() && isEmptyToDown();
    }

    /**
     * Checks if the buffer for sending data upwards is empty.
     *
     * @return true if the up buffer is empty, false otherwise
     */
    public boolean isEmptyToUp() {
        return ((BGate<?, ?>) down.acceptor).getQueue().isEmpty();
    }

    /**
     * Checks if the buffer for sending data downwards is empty.
     *
     * @return true if the down buffer is empty, false otherwise
     */
    public boolean isEmptyToDown() {
        return ((BGate<?, ?>) up.acceptor).getQueue().isEmpty();
    }

    /**
     * Appends a string representation of this node to the provided {@link AString}.
     *
     * @param sb the {@link AString} to append to
     */
    @Override
    public void toString(AString sb) {
        int s1 = -1;
        int s2 = -1;
        try {
            s1 = ((BGate<?, ?>) up.acceptor).getQueue().size();
        } catch (Exception e) {
        }
        try {
            s2 = ((BGate<?, ?>) down.acceptor).getQueue().size();
        } catch (Exception e) {
        }
        sb.add("buffer(").add(s1).add(":").add(s2).add(")");
    }

    /**
     * Returns a string representation of this node.
     *
     * @return the string representation
     */
    @Override
    public String toString() {
        return toString2();
    }

    /**
     * Returns the gate for sending data upwards.
     *
     * @return the up {@link FGate}
     */
    @Override
    public FGate<TWrite, TRead> gUp() {
        return up;
    }

    /**
     * Returns the gate for sending data downwards.
     *
     * @return the down {@link FGate}
     */
    @Override
    public FGate<TRead, TWrite> gDown() {
        return down;
    }

    /**
     * Resumes sending data upwards.
     */
    public void resumeUp() {
        ((BGate<TWrite, TRead>) up.acceptor).resume();
    }

    /**
     * Resumes sending data downwards.
     */
    public void resumeDown() {
        ((BGate<TRead, TWrite>) down.acceptor).resume();
    }

    /**
     * Resumes both upwards and downwards data sending.
     */
    public void resume() {
        resumeUp();
        resumeDown();
    }

    /**
     * Pauses sending data upwards.
     */
    public void pauseUp() {
        ((BGate<?, ?>) up.acceptor).pause();
    }

    /**
     * Pauses sending data downwards.
     */
    public void pauseDown() {
        ((BGate<?, ?>) down.acceptor).pause();
    }

    /**
     * Pauses both upwards and downwards data sending.
     */
    public void pause() {
        pauseUp();
        pauseDown();
    }

    /**
     * Initializes the up {@link BGate}.
     *
     * @return a new instance of {@link BGateUp}
     */
    protected BGateUp initUp() {
        return new BGateUp();
    }

    /**
     * Initializes the down {@link BGate}.
     *
     * @return a new instance of {@link BGateDown}
     */
    protected BGateDown initDown() {
        return new BGateDown();
    }

    /**
     * Creates a new {@link BufferNode} with default settings.
     *
     * @param <TUp>   the type of data written to the buffer
     * @param <TDown> the type of data read from the buffer
     * @return a new instance of {@link BufferNode}
     */
    public static <TUp, TDown> BufferNode<TUp, TDown> of() {
        return new BufferNode<>();
    }

    /**
     * Creates a new {@link BufferNode} and links it to the provided gate.
     *
     * @param g       the gate to link
     * @param <TUp>   the type of data written to the buffer
     * @param <TDown> the type of data read from the buffer
     * @return a new instance of {@link BufferNode}
     */
    public static <TUp, TDown> BufferNode<TUp, TDown> of(Gate<TDown, TUp> g) {
        var res = new BufferNode<TUp, TDown>();
        res.down().link(g);
        return res;
    }

    /**
     * Represents a buffer gate that can send and receive data.
     *
     * @param <TIn>  the type of incoming data
     * @param <TOut> the type of outgoing data
     */
    public abstract class BGate<TIn, TOut> extends FGate.Pair<TIn, TOut, TIn> {
        public int countDataValues() {
            return Flow.flow(queue).filter(Value::isData).count();
        }

        protected final Queue<Value<TIn>> queue = new ConcurrentPriorityQueue<>((v1, v2) -> {
            int c = v1.priority() - v2.priority();
            if (c == 0) {
                c = v1.hashCode() - v2.hashCode();
            }
            return c;
        });
        volatile boolean pauseWrite;

        public boolean isProcessing() {
            return !isPause() && !getQueue().isEmpty();
        }

        /**
         * Constructs a new {@link BGate} with the specified owner.
         *
         * @param owner the owner of this gate
         */
        public BGate(Object owner) {
            super(owner);
        }

        /**
         * Finds an object of the specified type.
         *
         * @param type the type to find
         * @param <T1> the type parameter
         * @return the found object or null if not found
         */
        @Override
        public <T1> T1 find(CTypeI<T1> type) {
            if (type.isInstance(owner)) return RU.cast(owner);
            return super.find(type);
        }

        /**
         * Finds an object of the specified class.
         *
         * @param type the class to find
         * @param <T1> the type parameter
         * @return the found object or null if not found
         */
        @Override
        public <T1> T1 find(Class<T1> type) {
            if (type.isInstance(owner)) return RU.cast(owner);
            return super.find(type);
        }

        FGate<TIn, TOut> fGate;

        @Override
        public void setFGate(FGate<TIn, TOut> g) {
            fGate = g;
        }

        final BlockMgr blockMgr = new BlockMgr();

        BGate<TOut, TIn> pairAcceptor() {
            return RU.cast(pair().fGate.acceptor);
        }

        /**
         * Flushes all tasks in the queue.
         */
        void flushTasks() {
            try {
                flushTasks0();
            } catch (Exception e) {
                Log.error(e);
            }
        }

        private void flushTasks0() {
            try {
                var g = pair();
                if (g.fGate.link == null) return;
                AtomicBoolean stopFlag = new AtomicBoolean();
                while (!isPause() && !stopFlag.get() && !blockMgr.isBlocked()) {
                    var v = queue.poll();
                    if (v == null) break;
                    if (v.isRequestData()) {
                        assert !v.isData();
                        var pa = pairAcceptor();
                        if (!pa.blockMgr.unblock(v.getRequestDataId())) {
                            break;
                        }
                    }
                    g.send(v.linkOnRejectExclusive((o, id) -> {
                        Log.trace("buffer: value is aborted $value (bid: $bid, owner: $owner)",
                                "owner", o,
                                "bid", id,
                                "value", v);
                        v.enter(BufferNode.this);
                        stopFlag.set(true);
                        queue.add(v);
                        if (!blockMgr.block(id)) {
                            flushTasks();
                        }
                    }));
                }
            } catch (Exception e) {
                Log.error(e);
            }
        }

        /**
         * Pauses writing to the buffer.
         */
        public void pause() {
            pauseWrite = true;
        }

        /**
         * Checks if writing is paused.
         *
         * @return true if writing is paused, false otherwise
         */
        public final boolean isPause() {
            return pauseWrite;
        }

        /**
         * Sends data through the gate and adds it to the queue.
         *
         * @param fGate the gate to send data through
         * @param value the data to send
         */
        @Override
        public void send(FGate<TIn, TOut> fGate, Value<TIn> value) {
            if (value.isOnlyRequestData()) {
                requestDataPair(value);
                return;
            }
            if (value.isRequestData()) {
                requestDataPair(value.getRequestDataId());
            }
            queue.add(value);
            flushTasks();
        }

        /**
         * Requests a pair of data.
         */
        public void requestDataPair(Value<TIn> r) {
            var p = pairAcceptor();
            p.fGate.inSide.send(r);
            if (p.blockMgr.unblock(r.getRequestDataId())) {
                p.flushTasks();
            }
        }

        public void requestDataPair(long id) {
            requestDataPair(Value.ofRequest(id));
        }

        /**
         * Resumes sending data.
         */
        public abstract void resume();

        /**
         * Resumes sending data through the specified gate.
         *
         * @param g the gate to send data through
         */
        public void resume(FGate<TIn, TOut> g) {
            if (pauseWrite) {
                pauseWrite = false;
                flushTasks();
            }
            g.inSide.send(Value.ofRequest());
        }

        /**
         * Returns the queue of values.
         *
         * @return the queue of values
         */
        public Queue<Value<TIn>> getQueue() {
            return queue;
        }
    }

    /**
     * Represents a buffer gate for sending data downwards.
     */
    public class BGateDown extends BGate<TRead, TWrite> {
        /**
         * Constructs a new {@link BGateDown}.
         */
        public BGateDown() {
            super(BufferNode.this);
        }

        /**
         * Resumes sending data through the down gate.
         */
        @Override
        public void resume() {
            resume(down);
        }

        /**
         * Returns the inside gate for the up gate.
         *
         * @return the inside gate for the up gate
         */
        @Override
        public FGate<?, TRead>.@NotNull InsideGate pair() {
            return gUp().inSide;
        }
    }

    /**
     * Represents a buffer gate for sending data upwards.
     */
    public class BGateUp extends BGate<TWrite, TRead> {
        /**
         * Constructs a new {@link BGateUp}.
         */
        public BGateUp() {
            super(BufferNode.this);
        }

        /**
         * Resumes sending data through the up gate.
         */
        @Override
        public void resume() {
            resume(up);
        }

        /**
         * Returns the inside gate for the down gate.
         *
         * @return the inside gate for the down gate
         */
        @Override
        public FGate<?, TWrite>.@NotNull InsideGate pair() {
            return gDown().inSide;
        }
    }
}