package io.aether.utils.streams.safe;

import io.aether.net.fastMeta.FastApiContext;
import io.aether.net.fastMeta.Command;
import io.aether.utils.AString;
import io.aether.utils.AutoRun;
import io.aether.utils.futures.AFuture;
import io.aether.utils.streams.FGate;
import io.aether.utils.streams.Value;
import org.jetbrains.annotations.NotNull;

import java.util.*;
import java.util.concurrent.ConcurrentLinkedDeque;

public class SafeNodeImpl implements SafeNode<Object> {
    public final InStreamApi inStreamApi = new InStreamApi();
    private final int circleSize;
    private final int windowSize;
    private final UpGate upWorker = new UpGate();
    private final FastApiContext con;
    private final FGate<byte[], byte[]> down;
    private final AutoRun.Multi autoWork;
    private final AutoRun.Multi.Task autoFlush;
    private final AutoRun.Multi.Task autoStatus;
    private boolean needSendStatus;
    final FGate<byte[], byte[]> up = FGate.of(new FGate.Pair<byte[], byte[], byte[]>(this) {

        @Override
        public FGate<?, byte[]>.@NotNull InsideGate pair() {
            return down.inSide;
        }

        @Override
        public void send(FGate<byte[], byte[]> fGate, Value<byte[]> value) {
            upWorker.send(value);
        }
    });
    private int repeatCounter = 0;

    public SafeNodeImpl() {
        this(0xFFFF);
    }

    public SafeNodeImpl(int circleSize) {
        this.circleSize = circleSize;
        windowSize = circleSize / 2;
        con = new FastApiContext(){
            @Override
            public void flush(AFuture sendFuture) {
                throw new UnsupportedOperationException();
            }
        };
        down = null;// FGate.of(GateI.proxy(this));
//        down.acceptor.setTarget(con.down());
//        con.linkLocalApi(inStreamApi);
        autoWork = new AutoRun.Multi(Runnable::run);
        autoFlush = new AutoRun.Multi.Task(autoWork) {
            @Override
            public void work() {
                downFlush();
            }
        };
        autoStatus = new AutoRun.Multi.Task(autoWork) {
            {
                setTimeout(10);
            }

            @Override
            public void work() {
                needSendStatus = true;
                autoFlush.needWork();
            }
        };
    }

    @Override
    public void toString(AString sb) {
        sb.add("SafeStream");
    }

    private void sendStatus() {
//        Log.debug("send status: " + upWorker.beginSend + " " + getLocalApi().begin);
//        getLocalApi().remoteApi.run(a -> a.status(convertShort(upWorker.beginSend), convertShort(getLocalApi().begin)));
        needSendStatus = false;
        autoStatus.refreshNeedWork();
    }

    public void confirmToRemote(int remoteEnd) {
        var l = getLocalApi();
        if (l.begin != l.end) {
            var off = l.begin;
            for (var e : l.bufferIn) {
                if (e.offset > off) {
                    int finalOff = off;
//                    l.remoteApi.run(a -> a.requestRepeat(convertShort(finalOff), (short) (e.offset - finalOff)));
                }
                off = e.end();
            }
            if (l.end != remoteEnd) {
//                l.remoteApi.run(a -> a.requestRepeatAfter(convertShort(l.end)));
            }
            autoFlush.needWork();
        } else {
//            l.remoteApi.run(a -> a.confirmReceive(convertShort(l.begin)));
        }
        autoFlush.needWork();
    }

    @Override
    public FGate<byte[], byte[]> gUp() {
        return up;
    }

    @Override
    public FGate<byte[], byte[]> gDown() {
        return down;
    }

    private int convert(int value) {
        return value % circleSize;
    }

    private short convertShort(int value) {
        return (short) convert(value);
    }

    private InStreamApi getLocalApi() {
        return inStreamApi;
    }

    private void downFlush() {
        upWorker.flush();
        if (needSendStatus) {
            sendStatus();
        }
//        con.flush();
    }

    private interface SafeStreamApi {
        @Command(3)
        void close();

        @Command(6)
        void confirmReceive(short offset);

        @Command(7)
        void confirmSend(short offset);

        @Command(8)
        void requestRepeat(short offset, short length);

        @Command(8)
        void send(short offset, byte[] data);

        @Command(9)
        void repeat(short repeat, short offset, byte[] data);

        @Command(11)
        void requestRepeatAfter(short offset);

        @Command(12)
        void repeatInt(int repeat, int offset, byte[] data);

        @Command(13)
        void requestRepeatAfterInt(int offset);

        @Command(14)
        void sendInt(int offset, byte[] data);

        @Command(15)
        void requestRepeatInt(int offset, int length);


        @Command(17)
        void confirmReceiveInt(int offset);

        @Command(18)
        void confirmSendInt(int offset);

        @Command(25)
        void status(short send, short receive);

        @Command(26)
        void statusInt(int send, int receive);
    }

    //<editor-fold desc="RecIn">
    private static final class RecIn implements Comparable<RecIn> {
        private final Value<byte[]> data;
        private int offset;

        private RecIn(int offset, Value<byte[]> data) {
            this.offset = offset;
            this.data = data;
        }

        public int len() {
            return data.data().length;
        }

        @Override
        public int compareTo(@NotNull SafeNodeImpl.RecIn o) {
            return offset - o.offset;
        }

        public int end() {
            return offset + data.data().length;
        }

        @Override
        public boolean equals(Object obj) {
            if (obj == this) return true;
            if (obj == null || obj.getClass() != this.getClass()) return false;
            var that = (RecIn) obj;
            return this.offset == that.offset && Arrays.equals(this.data.data(), that.data.data());
        }

        @Override
        public int hashCode() {
            return Objects.hash(offset, data);
        }

        @Override
        public String toString() {
            return "RecIn[" + "offset=" + offset + ", " + "data=" + Arrays.toString(data.data()) + ']';
        }

    }

    //</editor-fold>
    public class InStreamApi implements SafeStreamApi {
        final SortedSet<RecIn> bufferIn = new TreeSet<>();
        int begin;
        int end;
//        Remote<SafeStreamApi> remoteApi;
//        ApiBack apiConnection;

        @Override
        public void requestRepeatInt(int offset, int length) {
            upWorker.repeat(offset, length);
        }

        @Override
        public void status(short send, short receive) {
            statusInt(Short.toUnsignedInt(send), Short.toUnsignedInt(receive));
        }

        @Override
        public void statusInt(int send, int receive) {
            send = convertOffsetToLocal(send);
            if (send != end) {
                var off = begin;
                for (var e : bufferIn) {
                    if (e.offset > off) {
                        int finalOff = off;
//                        remoteApi.run(a -> a.requestRepeat(convertShort(finalOff), (short) (e.offset - finalOff)));
                    }
                    off = e.end();
                }
//                remoteApi.run(a -> a.requestRepeatAfter(convertShort(end)));
            }
            upWorker.confirmRemoteReceive(receive);
            needSendStatus = true;
            autoFlush.needWork();
        }

        @Override
        public void close() {
//            apiConnection.close();
        }

        @Override
        public void requestRepeat(short offset, short length) {
            requestRepeatInt(Short.toUnsignedInt(offset), Short.toUnsignedInt(length));
        }

        @Override
        public void send(short offset, byte[] data) {
            sendInt(Short.toUnsignedInt(offset), data);
        }

        @Override
        public void requestRepeatAfter(short offset) {
            requestRepeatAfterInt(Short.toUnsignedInt(offset));
        }

        @Override
        public void requestRepeatAfterInt(int offset) {
            upWorker.repeatAfter(offset);
        }

        private int convertOffsetToLocal(int offset) {
            if (end % circleSize < begin % circleSize) {
                if (offset < begin) offset += circleSize * (end / circleSize);
            } else if (begin >= circleSize) {
                begin %= circleSize;
                end %= circleSize;
                for (var e : bufferIn) {
                    e.offset %= circleSize;
                }
            }
            return offset;
        }

        private void correctRanges() {
            if (begin >= circleSize) {
                begin %= circleSize;
                end %= circleSize;
            }
        }

        private void receiveDataFromRemote(int offset, byte[] data) {
            end = Math.max(end, offset + data.length);
            var val = Value.of(data);
            if (offset == begin) {
                addBegin(data.length);
                up.inSide.send(val);
            } else {
                bufferIn.add(new RecIn(offset, val));
            }
            removeReadArrays();
        }

        @Override
        public void confirmReceive(short offset) {
            confirmReceiveInt(Short.toUnsignedInt(offset));
        }

        @Override
        public void confirmSend(short offset) {
            confirmSendInt(Short.toUnsignedInt(offset));
        }

        @Override
        public void confirmSendInt(int offset) {
            if (offset != end) {
                confirmToRemote(offset);
            }
        }

        @Override
        public void confirmReceiveInt(int offset) {
            upWorker.confirmRemoteReceive(offset);
        }

        private void addBegin(int val) {
            begin += val;
            correctRanges();
            needSendStatus = true;
            autoFlush.needWork();
        }

        private void removeReadArrays() {
            while (!bufferIn.isEmpty()) {
                var d = bufferIn.first();
                if (d.offset == begin) {
                    bufferIn.remove(d);
                    up.inSide.send(d.data);
                    addBegin(d.data.data().length);
                } else if (d.offset < begin) {
                    bufferIn.remove(d);
                    if (d.offset + d.len() > begin) {
                        var s = begin - d.offset;
                        var l = d.len() - s;
                        up.inSide.send(Value.of(Arrays.copyOfRange(d.data.data(), s, l)));
                        addBegin(l);
                    }
                } else {
                    break;
                }
            }
            needSendStatus = true;
            autoFlush.needWork();
        }


        private short convertShort(int value) {
            return (short) convert(value);
        }

        public void sendInt(int offset, byte[] data) {
            offset = convertOffsetToLocal(offset);
            receiveDataFromRemote(offset, data);
        }

        @Override
        public void repeat(short repeat, short offset, byte[] data) {
            repeatInt(Short.toUnsignedInt(repeat), Short.toUnsignedInt(offset), data);
        }

        public void repeatInt(int repeat, int offset, byte[] data) {
            offset = convertOffsetToLocal(offset);
            receiveDataFromRemote(offset, data);
        }
    }

    protected class UpGate {
        final Deque<Value<byte[]>> buffer = new ConcurrentLinkedDeque<>();
        final Deque<Value<byte[]>> buffer2 = new ConcurrentLinkedDeque<>();
        int begin;
        int beginSend;
        int end;

        public boolean isWritable() {
            return end - begin > windowSize;
        }

        public boolean send(Value<byte[]> value) {
            assert value.data().length <= windowSize : value.data().length;
            if ((end - begin) + value.data().length > windowSize) {
                needSendStatus = true;
                downFlush();
                return false;
            }
            buffer.add(value);
            buffer2.add(value);
            end += value.data().length;
            flushToDown();
            autoFlush.refreshNeedWork();
            return true;
        }

        public void flush() {
            flushToDown();
            correctRanges();
            autoFlush.needWork();
        }

        private void flushToDown() {
            correctRanges();
            while (true) {
                var v = buffer2.poll();
                if (v == null) break;
//                con.getRemoteApi(a -> a.send(convertShort(beginSend), v.data()));
                beginSend = (beginSend + v.data().length) % circleSize;
            }
            if (end - begin == windowSize) {
                correctRanges();
            } else {
                autoFlush.needWork();
            }
        }

        private void correctRanges() {
            if (begin >= circleSize) {
                begin %= circleSize;
                end %= circleSize;
            }
        }

        private int convertOffsetToLocal(int offset) {
            if (end % circleSize < begin % circleSize) {
                if (offset < begin) offset += circleSize * (end / circleSize);
            } else correctRanges();
            return offset;
        }

        public void confirmRemoteReceive(int offset) {
            offset = convertOffsetToLocal(offset);
            if (offset <= begin || offset > end) {
                return;
            }
            var skipOffset = (offset - begin);
            while (skipOffset > 0) {
                var ar = buffer.removeFirst();
                if (ar.data().length < skipOffset) {
                    skipOffset -= ar.data().length;
                } else if (skipOffset == ar.data().length) {
                    break;
                } else {
                    var ar2 = Arrays.copyOfRange(ar.data(), skipOffset, ar.data().length);
                    buffer.addFirst(ar.map2(ar2));
                    break;
                }
            }
            begin = offset;
            correctRanges();
            up.inSide.send(Value.ofRequest());
        }

        public void repeatAfter(int offset) {
            offset = convertOffsetToLocal(offset);
            if (offset < begin || offset >= end) return;
            var offset2 = begin;
            var repId = (short) (repeatCounter++);
            for (var ar : buffer) {
                var step = offset2 + ar.data().length;
                if (step > offset) {
                    var arBegin = offset - offset2;
                    int finalOffset = offset;
//                    con.getRemoteApi(a -> a.repeat(repId, convertShort(finalOffset), Arrays.copyOfRange(ar.data(), arBegin, ar.data().length)));
                    offset = step;
                }
                offset2 = step;
            }
        }

        public void repeat(int offset, int length) {
            autoFlush.refreshNeedWork();
            offset = convertOffsetToLocal(offset);
            if (offset < begin || offset > end) return;
            var repId = (short) repeatCounter++;
            var offset2 = begin;
            for (var ar : buffer) {
                var step = offset2 + ar.data().length;
                if (step > offset) {
                    int finalOffset = offset;
                    if (ar.data().length == length) {
//                        con.getRemoteApi(a -> a.repeat(repId, convertShort(finalOffset), ar.data()));
                        break;
                    } else if (ar.data().length < length) {
                        var arBegin = offset - offset2;
//                        con.getRemoteApi(a -> a.repeat(repId, convertShort(finalOffset), Arrays.copyOfRange(ar.data(), arBegin, ar.data().length)));
                        offset = step;
                        length -= (ar.data().length - arBegin);
                    } else {
                        var arBegin = offset - offset2;
                        int finalLength = length;
//                        con.getRemoteApi(a -> a.repeat(repId, convertShort(finalOffset), Arrays.copyOfRange(ar.data(), arBegin, arBegin + finalLength)));
                        break;
                    }
                }
                offset2 = step;
            }
        }

        public int getSizeForWrite() {
            return windowSize - (end - begin);
        }

    }
}
