/*
 * Decompiled with CFR 0.152.
 */
package io.aether.utils.streams;

import io.aether.logger.LNode;
import io.aether.logger.Log;
import io.aether.utils.AString;
import io.aether.utils.RU;
import io.aether.utils.dataio.DataInOutStatic;
import io.aether.utils.streams.FGate;
import io.aether.utils.streams.Node;
import io.aether.utils.streams.Value;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.Base64;

public class WebSocketNode
implements Node<byte[], byte[], byte[], byte[]> {
    private final FGate<byte[], byte[]> up = FGate.of(new WebSocketUpHandler(this));
    private final FGate<byte[], byte[]> down = FGate.of(new WebSocketDownHandler(this));
    private final LNode log = Log.createContext();
    private volatile State state = State.HANDSHAKE;
    private final String websocketKey;
    private ByteBuffer partialFrame;
    private int expectedLength = -1;
    private boolean isFinalFragment;
    private int opcode;

    public WebSocketNode() {
        byte[] random = new byte[16];
        RU.SECURE_RANDOM.nextBytes(random);
        this.websocketKey = Base64.getEncoder().encodeToString(random);
    }

    @Override
    public FGate<byte[], byte[]> gUp() {
        return this.up;
    }

    @Override
    public FGate<byte[], byte[]> gDown() {
        return this.down;
    }

    public void toString(AString sb) {
        sb.add("WebSocketNode(state=").add(this.state.name()).add(")");
    }

    private class WebSocketDownHandler
    extends FGate.Pair<byte[], byte[], byte[]> {
        public WebSocketDownHandler(Object owner) {
            super(owner);
        }

        @Override
        public FGate.InsideGate pair() {
            return WebSocketNode.this.up.inSide;
        }

        @Override
        public void send(FGate<byte[], byte[]> fGate, Value<byte[]> value) {
            try {
                if (!value.isData()) {
                    this.pair().send(value);
                    return;
                }
                byte[] data = value.data();
                if (WebSocketNode.this.partialFrame != null) {
                    data = this.mergePartialData(data);
                }
                DataInOutStatic in = new DataInOutStatic(data);
                while (in.isReadable()) {
                    if (WebSocketNode.this.state == State.HANDSHAKE) {
                        if (this.processHandshakeResponse(in)) continue;
                        return;
                    }
                    if (WebSocketNode.this.expectedLength == -1 && !this.parseFrameHeader(in)) {
                        this.savePartialData(in);
                        return;
                    }
                    if (in.getSizeForRead() >= WebSocketNode.this.expectedLength) {
                        this.processCompleteFrame(in);
                        continue;
                    }
                    this.savePartialData(in);
                    return;
                }
            }
            catch (IOException e) {
                Log.error((String)"WebSocket protocol error", (Throwable)e, (Object[])new Object[0]);
                value.reject(this);
            }
        }

        private boolean processHandshakeResponse(DataInOutStatic in) throws IOException {
            String response = in.readString1();
            if (!(response.contains("HTTP/1.1 101") && response.contains("Upgrade: websocket") && response.contains("Connection: Upgrade"))) {
                throw new IOException("Invalid WebSocket handshake response");
            }
            String acceptKey = WebSocketNode.this.websocketKey + "258EAFA5-E914-47DA-95CA-C5AB0DC85B11";
            String expectedAccept = Base64.getEncoder().encodeToString(RU.sha1((byte[])acceptKey.getBytes(StandardCharsets.UTF_8)));
            if (!response.contains("Sec-WebSocket-Accept: " + expectedAccept)) {
                throw new IOException("WebSocket accept key mismatch");
            }
            WebSocketNode.this.state = State.CONNECTED;
            this.pair().send(Value.ofRequest());
            return true;
        }

        private boolean parseFrameHeader(DataInOutStatic in) throws IOException {
            if (in.getSizeForRead() < 2) {
                return false;
            }
            byte b1 = in.readByte();
            byte b2 = in.readByte();
            WebSocketNode.this.isFinalFragment = (b1 & 0x80) != 0;
            WebSocketNode.this.opcode = b1 & 0xF;
            boolean masked = (b2 & 0x80) != 0;
            int payloadLength = b2 & 0x7F;
            if (payloadLength == 126) {
                if (in.getSizeForRead() < 2) {
                    return false;
                }
                payloadLength = in.readUShort();
            } else if (payloadLength == 127) {
                if (in.getSizeForRead() < 8) {
                    return false;
                }
                payloadLength = (int)in.readLong();
            }
            if (masked) {
                if (in.getSizeForRead() < 4) {
                    return false;
                }
                in.skipBytes(4);
            }
            WebSocketNode.this.expectedLength = payloadLength;
            return true;
        }

        private void processCompleteFrame(DataInOutStatic in) throws IOException {
            byte[] payload = in.readBytes(WebSocketNode.this.expectedLength);
            WebSocketNode.this.expectedLength = -1;
            switch (WebSocketNode.this.opcode) {
                case 2: {
                    if (!WebSocketNode.this.isFinalFragment) break;
                    this.pair().send(Value.of(payload));
                    break;
                }
                case 8: {
                    this.sendCloseFrame();
                    WebSocketNode.this.state = State.CLOSED;
                    this.pair().send(Value.ofClose());
                    break;
                }
                case 9: {
                    this.sendPongFrame(payload);
                    break;
                }
                default: {
                    Log.warn((String)("Unsupported WebSocket opcode: " + WebSocketNode.this.opcode), (LNode[])new LNode[0]);
                }
            }
        }

        private void sendCloseFrame() throws IOException {
            byte[] closeFrame = new byte[]{-120, 0};
            WebSocketNode.this.down.inSide.send(Value.of(closeFrame));
        }

        private void sendPongFrame(byte[] payload) throws IOException {
            DataInOutStatic out = new DataInOutStatic(2 + payload.length);
            out.writeByte(-118);
            out.writeByte((int)((byte)payload.length));
            out.write(payload);
            WebSocketNode.this.down.inSide.send(Value.of(out.toArray()));
        }

        private void savePartialData(DataInOutStatic in) {
            WebSocketNode.this.partialFrame = ByteBuffer.wrap(in.readBytes(in.getSizeForRead()));
        }

        private byte[] mergePartialData(byte[] newData) {
            ByteBuffer merged = ByteBuffer.allocate(WebSocketNode.this.partialFrame.remaining() + newData.length);
            merged.put(WebSocketNode.this.partialFrame);
            merged.put(newData);
            WebSocketNode.this.partialFrame = null;
            return merged.array();
        }
    }

    private class WebSocketUpHandler
    extends FGate.Pair<byte[], byte[], byte[]> {
        public WebSocketUpHandler(Object owner) {
            super(owner);
        }

        @Override
        public FGate.InsideGate pair() {
            return WebSocketNode.this.down.inSide;
        }

        @Override
        public void send(FGate<byte[], byte[]> fGate, Value<byte[]> value) {
            try {
                switch (WebSocketNode.this.state) {
                    case HANDSHAKE: {
                        this.sendHandshakeRequest();
                        break;
                    }
                    case CONNECTED: {
                        if (value.isData()) {
                            this.sendDataFrame(value.data());
                            break;
                        }
                        if (!value.isClose()) break;
                        this.sendCloseFrame();
                        WebSocketNode.this.state = State.CLOSED;
                        break;
                    }
                    case CLOSED: {
                        Log.warn((String)"Attempt to send data on closed WebSocket", (LNode[])new LNode[0]);
                    }
                }
            }
            catch (IOException e) {
                Log.error((String)"WebSocket send error", (Throwable)e, (Object[])new Object[0]);
                value.reject(this);
            }
        }

        private void sendHandshakeRequest() throws IOException {
            String request = "GET / HTTP/1.1\r\nHost: example.com\r\nUpgrade: websocket\r\nConnection: Upgrade\r\nSec-WebSocket-Key: " + WebSocketNode.this.websocketKey + "\r\nSec-WebSocket-Version: 13\r\n\r\n";
            this.pair().send(Value.of(request.getBytes(StandardCharsets.UTF_8)));
        }

        private void sendDataFrame(byte[] data) throws IOException {
            DataInOutStatic out = new DataInOutStatic(2 + (data.length < 126 ? 0 : (data.length < 65536 ? 2 : 8)) + data.length);
            out.writeByte(-126);
            if (data.length < 126) {
                out.writeByte((int)((byte)data.length));
            } else if (data.length < 65536) {
                out.writeByte(126);
                out.writeShort((short)data.length);
            } else {
                out.writeByte(127);
                out.writeLong((long)data.length);
            }
            out.write(data);
            this.pair().send(Value.of(out.toArray()));
        }

        private void sendCloseFrame() throws IOException {
            byte[] closeFrame = new byte[]{-120, 0};
            this.pair().send(Value.of(closeFrame));
        }
    }

    private static enum State {
        HANDSHAKE,
        CONNECTED,
        CLOSED;

    }
}

