package io.aether.crypto.hydrogen;

import io.aether.crypto.AKey;
import io.aether.crypto.CryptoEngine;
import io.aether.crypto.CryptoProvider;
import io.aether.crypto.DecryptException;
import io.aether.crypto.EncryptException;
import io.aether.nativeLib.HydrogenLib;

import java.util.Arrays;

public class HydrogenAsymmetricEngine implements CryptoEngine {
    private final HydrogenLib.hydro_kx_session_keypair session_kp;
    private final HydrogenLib.hydro_kx_keypair keypair;
    private final AKey.AsymmetricPublic publicKey;
    private final AKey.AsymmetricPrivate privateKey;

    public HydrogenAsymmetricEngine(AKey.AsymmetricPublic publicKey) {
        this.publicKey = publicKey;
        this.privateKey = null;
        session_kp = new HydrogenLib.hydro_kx_session_keypair();
        keypair = new HydrogenLib.hydro_kx_keypair(publicKey.getData(), null);
    }

    public HydrogenAsymmetricEngine(AKey.AsymmetricPrivate privateKey, AKey.AsymmetricPublic publicKey) {
        this.privateKey = privateKey;
        this.publicKey = publicKey;
        session_kp = new HydrogenLib.hydro_kx_session_keypair();
        keypair = new HydrogenLib.hydro_kx_keypair(publicKey.getData(), privateKey.getData());
    }

    @Override
    public byte[] encrypt(byte[] data) {
        var ephemeral_pk = new byte[HydrogenLib.hydro_kx_N_PACKET1BYTES];
        HydrogenLib.hydro_kx_n_1(session_kp, ephemeral_pk, null, publicKey.getData());

        // Создаем симметричный движок с выведенным сессионным ключом
        var symmetricKey = new HydrogenKey.Symmetric(session_kp.tx);
        var symmetricEngine = new HydrogenSymmetricEngine(symmetricKey);

        var encryptedData = symmetricEngine.encrypt(data);
        return addArray(ephemeral_pk, encryptedData);
    }

    @Override
    public byte[] decrypt(byte[] data) {
        if (privateKey == null) {
            throw new UnsupportedOperationException("This engine is not configured for decryption.");
        }
        var ephemeral_pk = Arrays.copyOfRange(data, 0, HydrogenLib.hydro_kx_N_PACKET1BYTES);
        var in = Arrays.copyOfRange(data, HydrogenLib.hydro_kx_N_PACKET1BYTES, data.length);

        if (0 != HydrogenLib.hydro_kx_n_2(session_kp, ephemeral_pk, null, keypair)) {
            throw new DecryptException("Failed to derive session keys for decryption.");
        }

        // Создаем симметричный движок с выведенным сессионным ключом
        var symmetricKey = new HydrogenKey.Symmetric(session_kp.rx);
        var symmetricEngine = new HydrogenSymmetricEngine(symmetricKey);

        return symmetricEngine.decrypt(in);
    }

    @Override
    public CryptoProvider getCryptoProvider() {
        return HydrogenCryptoProvider.INSTANCE;
    }

    @Override
    public String getProviderName() {
        return "HYDROGEN";
    }

    private static byte[] addArray(byte[] a1, byte[] a2) {
        var res = new byte[a1.length + a2.length];
        System.arraycopy(a1, 0, res, 0, a1.length);
        System.arraycopy(a2, 0, res, a1.length, a2.length);
        return res;
    }
}