package io.aether.crypto.hydrogen;

import io.aether.crypto.*;
import io.aether.nativeLib.HydrogenLib;
import io.aether.utils.HexUtils;
import io.aether.utils.RU;

import java.nio.charset.StandardCharsets;
import java.util.Arrays;

public class HydrogenCryptoProvider implements CryptoProvider {

    public static final HydrogenCryptoProvider INSTANCE = new HydrogenCryptoProvider();
    private static final byte[] KDF_CONTEXT = "_aether_".getBytes(StandardCharsets.UTF_8);

    private HydrogenCryptoProvider() {
    }

    @Override
    public AKey.SignPublic createSignPublicKey(byte[] data) {
        return new HydrogenKey.SignPublic(data);
    }

    @Override
    public AKey.SignPrivate createSignPrivateKey(byte[] data) {
        return new HydrogenKey.SignPrivate(data);
    }

    @Override
    public String getCryptoLibName() {
        return "HYDROGEN";
    }

    @Override
    public AKey.Symmetric createSymmetricKey(byte[] bytes) {
        return new HydrogenKey.Symmetric(bytes);
    }

    @Override
    public PairAsymKeys createAsymmetricKeys() {
        var keys = new HydrogenLib.hydro_kx_keypair();
        HydrogenLib.hydro_kx_keygen(keys);
        return new PairAsymKeys(new HydrogenKey.AsymmetricPublic(keys.pk), new HydrogenKey.AsymmetricPrivate(keys.sk));
    }

    @Override
    public AKey.Symmetric createSymmetricKey() {
        var key = new byte[(int) HydrogenLib.hydro_secretbox_KEYBYTES];
        HydrogenLib.hydro_secretbox_keygen(key);
        return new HydrogenKey.Symmetric(key);
    }

    @Override
    public PairSignKeys createSignKeys() {
        var keys = new HydrogenLib.hydro_sign_keypair();
        HydrogenLib.hydro_sign_keygen(keys);
        return new PairSignKeys(new HydrogenKey.SignPublic(keys.pk), new HydrogenKey.SignPrivate(keys.sk));
    }

    @Override
    public Signer createSigner(PairSignKeys keys) {
        return createSigner(keys.publicKey, keys.privateKey);
    }

    @Override
    public Signer createSigner(AKey.SignPublic publicKey, AKey.SignPrivate privateKey) {
        if (!(publicKey instanceof HydrogenKey.SignPublic) || !(privateKey instanceof HydrogenKey.SignPrivate)) {
            throw new IllegalArgumentException("Keys must be instances of HydrogenKey.SignPublic and HydrogenKey.SignPrivate");
        }
        return new HydrogenSigner(publicKey, privateKey);
    }

    @Override
    public Signer createSigner(AKey.SignPublic publicKey) {
        if (!(publicKey instanceof HydrogenKey.SignPublic)) {
            throw new IllegalArgumentException("Public key must be an instance of HydrogenKey.SignPublic");
        }
        return new HydrogenSigner(publicKey, null);
    }

    @Override
    public CryptoEngine createSymmetricEngine(AKey.Symmetric key) {
        if (!(key instanceof HydrogenKey.Symmetric)) {
            throw new IllegalArgumentException("Key must be a HydrogenKey.Symmetric instance");
        }
        return new HydrogenSymmetricEngine(key);
    }

    @Override
    public CryptoEngine createAsymmetricEngine(AKey.AsymmetricPublic key) {
        if (!(key instanceof HydrogenKey.AsymmetricPublic)) {
            throw new IllegalArgumentException("Key must be a HydrogenKey.AsymmetricPublic instance");
        }
        return new HydrogenAsymmetricEngine(key);
    }

    @Override
    public CryptoEngine createAsymmetricEngine(AKey.AsymmetricPrivate privateKey, AKey.AsymmetricPublic publicKey) {
        if (!(privateKey instanceof HydrogenKey.AsymmetricPrivate) || !(publicKey instanceof HydrogenKey.AsymmetricPublic)) {
            throw new IllegalArgumentException("Keys must be instances of HydrogenKey.AsymmetricPrivate and HydrogenKey.AsymmetricPublic");
        }
        return new HydrogenAsymmetricEngine(privateKey, publicKey);
    }

    @Override
    public CryptoEngine createAsymmetricEngine(PairAsymKeys keys) {
        return createAsymmetricEngine(keys.getPrivateKey(), keys.getPublicKey());
    }

    @Override
    public <T extends AKey> T createKey(KeyType keyType, byte[] data) {
        var res = switch (keyType) {
            case SYMMETRIC -> new HydrogenKey.Symmetric(data);
            case ASYMMETRIC_PUBLIC -> new HydrogenKey.AsymmetricPublic(data);
            case ASYMMETRIC_PRIVATE -> new HydrogenKey.AsymmetricPrivate(data);
            case SIGN_PUBLIC -> new HydrogenKey.SignPublic(data);
            case SIGN_PRIVATE -> new HydrogenKey.SignPrivate(data);
            default -> throw new UnsupportedOperationException();
        };
        return RU.cast(res);
    }

    @Override
    public <T extends AKey> T createKey(String data) {
        var parts = data.split(":");
        if (parts.length != 3 || !parts[0].equals(getCryptoLibName())) {
            throw new IllegalArgumentException("Invalid key string for this provider.");
        }
        KeyType keyType = KeyType.valueOf(parts[1]);
        byte[] bytes = HexUtils.hexToBytes(parts[2]);
        return RU.cast(createKey(keyType, bytes));
    }

    @Override
    public Sign createSign(String data) {
        var parts = data.split(":");
        if (parts.length != 2 || !parts[0].equals(getCryptoLibName())) {
            throw new IllegalArgumentException("Invalid sign string for this provider.");
        }
        return new HydrogenSign(HexUtils.hexToBytes(parts[1]));
    }

    /**
     * Derives a pair of symmetric keys (for client-to-server and server-to-client communication)
     * using Key Derivation Function (KDF) from a master key and session/key identifiers.
     *
     * @param masterKey The master symmetric key.
     * @param serverId  The server identifier (32-bit).
     * @param keyNumber The key number/index (32-bit).
     * @return A PairSymmetricKeys object containing client and server keys.
     */
    @Override
    public PairSymKeys deriveSymmetricKeys(AKey.Symmetric masterKey, int serverId, int keyNumber) {
        if (!(masterKey instanceof HydrogenKey.Symmetric)) {
            throw new IllegalArgumentException("Key must be a HydrogenKey.Symmetric instance");
        }

        // 32-byte key size for hydro_secretbox
        int keySize = HydrogenLib.hydro_secretbox_KEYBYTES;
        byte[] derivedKey = new byte[keySize * 2];

        // C++: std::uint64_t const subkey_id = (static_cast<std::uint64_t>(server_id) << 32) | key_number;
        long subkeyId = (((long) serverId) << 32) | (keyNumber & 0xFFFFFFFFL);

        // Context is from HydrogenKey.java: HYDROGEN_LIB_CTX
        // int hydro_kdf_derive_from_key(byte[] subkey, long subkey_len, long subkey_id, byte[] ctx, byte[] key)
        int result = HydrogenLib.hydro_kdf_derive_from_key(
                derivedKey, subkeyId, KDF_CONTEXT, masterKey.getData()
        );

        if (result != 0) {
            throw new RuntimeException("Hydro KDF derivation failed with error code: " + result);
        }

        // Split 64-byte key into two 32-byte keys
        byte[] clientToServerKeyBytes = Arrays.copyOfRange(derivedKey, 0, keySize);
        byte[] serverToClientKeyBytes = Arrays.copyOfRange(derivedKey, keySize, derivedKey.length);

        AKey.Symmetric clientToServerKey = new HydrogenKey.Symmetric(clientToServerKeyBytes);
        AKey.Symmetric serverToClientKey = new HydrogenKey.Symmetric(serverToClientKeyBytes);

        return new PairSymKeys(clientToServerKey, serverToClientKey);
    }

    @Override
    public Sign createSign(byte[] data) {
        return new HydrogenSign(data);
    }
}

