package io.aether.crypto.hydrogen;

import io.aether.crypto.*;
import io.aether.utils.HexUtils;

import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Objects;

public abstract class HydrogenKey implements AKey {
    protected final byte[] data;
    protected final KeyType keyType;
    protected final String providerName = "HYDROGEN";
    public final static byte[] HYDROGEN_LIB_CTX = "aetherio".getBytes(StandardCharsets.UTF_8);

    protected HydrogenKey(byte[] data, KeyType keyType) {
        this.data = data;
        this.keyType = keyType;
    }

    @Override
    public byte[] getData() {
        return data;
    }

    @Override
    public KeyType getKeyType() {
        return keyType;
    }

    @Override
    public String getProviderName() {
        return providerName;
    }

    @Override
    public CryptoProvider getCryptoProvider() {
        return HydrogenCryptoProvider.INSTANCE;
    }

    @Override
    public int size() {
        return data.length;
    }

    @Override
    public String keyToString() {
        return getProviderName() + ":" + getKeyType() + ":" + HexUtils.toHexString(getData());
    }

    @Override
    public String toString() {
        return keyToString();
    }

    @Override
    public boolean equals(Object obj) {
        if (obj instanceof AKey k) {
            return getKeyType() == k.getKeyType() && getProviderName().equals(k.getProviderName()) && Arrays.equals(getData(), k.getData());
        }
        return false;
    }

    @Override
    public int hashCode() {
        return Objects.hash(getKeyType(), getProviderName(), Arrays.hashCode(getData()));
    }

    public static class Symmetric extends HydrogenKey implements AKey.Symmetric {
        public Symmetric(byte[] data) {
            super(data, KeyType.SYMMETRIC);
        }

        @Override
        public CryptoEngine toCryptoEngine() {
            return CryptoProviderFactory.getProvider(this).createSymmetricEngine(this);
        }
    }

    public static class AsymmetricPrivate extends HydrogenKey implements AKey.AsymmetricPrivate {
        public AsymmetricPrivate(byte[] data) {
            super(data, KeyType.ASYMMETRIC_PRIVATE);
        }
    }

    public static class AsymmetricPublic extends HydrogenKey implements AKey.AsymmetricPublic {
        public AsymmetricPublic(byte[] data) {
            super(data, KeyType.ASYMMETRIC_PUBLIC);
        }

        @Override
        public CryptoEngine toCryptoEngine() {
            return CryptoProviderFactory.getProvider(this).createAsymmetricEngine(this);
        }
    }

    public static class SignPrivate extends HydrogenKey implements AKey.SignPrivate {
        public SignPrivate(byte[] data) {
            super(data, KeyType.SIGN_PRIVATE);
        }
    }

    public static class SignPublic extends HydrogenKey implements AKey.SignPublic {
        public SignPublic(byte[] data) {
            super(data, KeyType.SIGN_PUBLIC);
        }

        @Override
        public SignChecker toSignChecker() {
            return CryptoProviderFactory.getProvider(this).createSigner(this);
        }
    }
}

