package io.aether.crypto.hydrogen;

import io.aether.api.common.KeySignPublic;
import io.aether.crypto.AKey;
import io.aether.crypto.Sign;
import io.aether.crypto.Signer;
import io.aether.crypto.CryptoProvider;
import io.aether.nativeLib.HydrogenLib;

public class HydrogenSigner implements Signer {
    private final AKey.SignPublic publicKey;
    private final AKey.SignPrivate privateKey;

    public HydrogenSigner(AKey.SignPublic publicKey, AKey.SignPrivate privateKey) {
        this.publicKey = publicKey;
        this.privateKey = privateKey;
    }

    @Override
    public Sign sign(byte[] data) {
        if (privateKey == null) {
            throw new IllegalStateException("This Signer instance is not configured with a private key for signing.");
        }
        byte[] sign = new byte[HydrogenLib.hydro_sign_BYTES];
        HydrogenLib.hydro_sign_create(sign, data, HydrogenKey.HYDROGEN_LIB_CTX, privateKey.getData());
        return new HydrogenSign(sign);
    }

    @Override
    public boolean checkSign(byte[] data, Sign sign) {
        if (!(sign instanceof HydrogenSign)) {
            throw new IllegalArgumentException("Sign must be a HydrogenSign instance");
        }
        if (publicKey == null) {
            throw new IllegalStateException("This Signer instance is not configured with a public key for checking.");
        }
        return HydrogenLib.hydro_sign_verify(sign.getSignData(), data, HydrogenKey.HYDROGEN_LIB_CTX, publicKey.getData()) == 0;
    }

    @Override
    public AKey.SignPublic getPublicKey() {
        return publicKey;
    }

    @Override
    public CryptoProvider getCryptoProvider() {
        return HydrogenCryptoProvider.INSTANCE;
    }
}

