package io.aether.crypto.hydrogen;

import io.aether.crypto.AKey;
import io.aether.crypto.CryptoEngine;
import io.aether.crypto.CryptoProvider;
import io.aether.crypto.DecryptException;
import io.aether.crypto.EncryptException;
import io.aether.nativeLib.HydrogenLib;
import io.aether.utils.DataUtils;

import java.util.Arrays;
import java.util.concurrent.atomic.AtomicLong;

public class HydrogenSymmetricEngine implements CryptoEngine {
    private final AKey.Symmetric key;
    private final AtomicLong msgId = new AtomicLong(0);

    public HydrogenSymmetricEngine(AKey.Symmetric key) {
        this.key = key;
    }

    @Override
    public byte[] encrypt(byte[] data) {
        long currentMsgId = msgId.incrementAndGet();

        var result = new byte[Long.BYTES + HydrogenLib.hydro_secretbox_HEADERBYTES + data.length];

        DataUtils.writeLongLE(result, 0, currentMsgId);

        var encryptedOutputBuffer = new byte[HydrogenLib.hydro_secretbox_HEADERBYTES + data.length];

        if (0 != HydrogenLib.hydro_secretbox_encrypt(encryptedOutputBuffer, data, currentMsgId, HydrogenKey.HYDROGEN_LIB_CTX, key.getData())) {
            throw new EncryptException();
        }

        System.arraycopy(encryptedOutputBuffer, 0, result, Long.BYTES, encryptedOutputBuffer.length);

        return result;
    }

    @Override
    public byte[] decrypt(byte[] data) {
        if (data.length < Long.BYTES + HydrogenLib.hydro_secretbox_HEADERBYTES) {
            throw new DecryptException("Invalid data length. Missing msgId or header.");
        }

        long currentMsgId = DataUtils.readLongLE(data, 0);

        var encryptedData = Arrays.copyOfRange(data, Long.BYTES, data.length);

        var result = new byte[encryptedData.length - HydrogenLib.hydro_secretbox_HEADERBYTES];

        if (0 != HydrogenLib.hydro_secretbox_decrypt(result, encryptedData, currentMsgId, HydrogenKey.HYDROGEN_LIB_CTX, key.getData())) {
            throw new DecryptException();
        }
        return result;
    }

    @Override
    public String getProviderName() {
        return "HYDROGEN";
    }

    @Override
    public CryptoProvider getCryptoProvider() {
        return HydrogenCryptoProvider.INSTANCE;
    }

}