package io.aether.crypto.sodium;

import io.aether.crypto.*;
import io.aether.nativeLib.SodiumLib;
import io.aether.utils.HexUtils;
import io.aether.utils.RU;

public class SodiumCryptoProvider implements CryptoProvider {

    public static final SodiumCryptoProvider INSTANCE = new SodiumCryptoProvider();

    // Константы из libsodium
    private static final int CRYPTO_SECRETBOX_KEYBYTES = 32;
    private static final int CRYPTO_BOX_PUBLICKEYBYTES = 32;
    private static final int CRYPTO_BOX_SECRETKEYBYTES = 32;
    private static final int CRYPTO_SIGN_PUBLICKEYBYTES = 32;
    private static final int CRYPTO_SIGN_SECRETKEYBYTES = 64;

    private SodiumCryptoProvider() {}

    @Override
    public AKey.SignPublic createSignPublicKey(byte[] data) {
        return new SodiumKey.SignPublic(data);
    }

    @Override
    public AKey.SignPrivate createSignPrivateKey(byte[] data) {
        return new SodiumKey.SignPrivate(data);
    }

    @Override
    public AKey.Symmetric createSymmetricKey(byte[] data) {
        return new SodiumKey.Symmetric(data);
    }

    @Override
    public String getCryptoLibName() {
        return "SODIUM";
    }

    @Override
    public PairAsymKeys createAsymmetricKeys() {
        var keys = new SodiumLib.crypto_box_keypair();
        SodiumLib.crypto_box_keypair(keys);
        return new PairAsymKeys(new SodiumKey.AsymmetricPublic(keys.pk), new SodiumKey.AsymmetricPrivate(keys.sk));
    }

    @Override
    public AKey.Symmetric createSymmetricKey() {
        var key = new byte[CRYPTO_SECRETBOX_KEYBYTES];
        SodiumLib.crypto_secretbox_keygen(key);
        return new SodiumKey.Symmetric(key);
    }

    @Override
    public PairSignKeys createSignKeys() {
        var keys = new SodiumLib.crypto_sign_keypair();
        SodiumLib.crypto_sign_keypair(keys);
        return new PairSignKeys(new SodiumKey.SignPublic(keys.pk), new SodiumKey.SignPrivate(keys.sk));
    }

    @Override
    public Signer createSigner(PairSignKeys keys) {
        return createSigner(keys.publicKey, keys.privateKey);
    }

    @Override
    public Signer createSigner(AKey.SignPublic publicKey, AKey.SignPrivate privateKey) {
        if (!(publicKey instanceof SodiumKey.SignPublic) || !(privateKey instanceof SodiumKey.SignPrivate)) {
            throw new IllegalArgumentException("Keys must be instances of SodiumKey.SignPublic and SodiumKey.SignPrivate");
        }
        return new SodiumSigner(publicKey, privateKey);
    }

    @Override
    public Signer createSigner(AKey.SignPublic publicKey) {
        if (!(publicKey instanceof SodiumKey.SignPublic)) {
            throw new IllegalArgumentException("Public key must be an instance of SodiumKey.SignPublic");
        }
        return new SodiumSigner(publicKey, null);
    }

    @Override
    public CryptoEngine createSymmetricEngine(AKey.Symmetric key) {
        if (!(key instanceof SodiumKey.Symmetric)) {
            throw new IllegalArgumentException("Key must be a SodiumKey.Symmetric instance");
        }
        return new SodiumSymmetricEngine(key);
    }

    @Override
    public CryptoEngine createAsymmetricEngine(AKey.AsymmetricPublic key) {
        if (!(key instanceof SodiumKey.AsymmetricPublic)) {
            throw new IllegalArgumentException("Key must be a SodiumKey.AsymmetricPublic instance");
        }
        return new SodiumAsymmetricEngine(key);
    }

    @Override
    public CryptoEngine createAsymmetricEngine(AKey.AsymmetricPrivate privateKey, AKey.AsymmetricPublic publicKey) {
        if (!(privateKey instanceof SodiumKey.AsymmetricPrivate) || !(publicKey instanceof SodiumKey.AsymmetricPublic)) {
            throw new IllegalArgumentException("Keys must be instances of SodiumKey.AsymmetricPrivate and SodiumKey.AsymmetricPublic");
        }
        return new SodiumAsymmetricEngine(privateKey, publicKey);
    }

    @Override
    public CryptoEngine createAsymmetricEngine(PairAsymKeys keys) {
        return createAsymmetricEngine(keys.getPrivateKey(), keys.getPublicKey());
    }

    @Override
    public <T extends AKey> T createKey(KeyType keyType, byte[] data) {
        return switch (keyType) {
            case SYMMETRIC -> RU.cast(new SodiumKey.Symmetric(data));
            case ASYMMETRIC_PUBLIC -> RU.cast(new SodiumKey.AsymmetricPublic(data));
            case ASYMMETRIC_PRIVATE -> RU.cast(new SodiumKey.AsymmetricPrivate(data));
            case SIGN_PUBLIC -> RU.cast(new SodiumKey.SignPublic(data));
            case SIGN_PRIVATE -> RU.cast(new SodiumKey.SignPrivate(data));
        };
    }

    @Override
    public <T extends AKey> T createKey(String data) {
        var parts = data.split(":");
        if (parts.length != 3 || !parts[0].equals(getCryptoLibName())) {
            throw new IllegalArgumentException("Invalid key string for this provider.");
        }
        KeyType keyType = KeyType.valueOf(parts[1]);
        byte[] bytes = HexUtils.hexToBytes(parts[2]);
        return createKey(keyType, bytes);
    }

    @Override
    public Sign createSign(String data) {
        var parts = data.split(":");
        if (parts.length != 2 || !parts[0].equals(getCryptoLibName())) {
            throw new IllegalArgumentException("Invalid sign string for this provider.");
        }
        return new SodiumSign(HexUtils.hexToBytes(parts[1]));
    }

    @Override
    public Sign createSign(byte[] data) {
        return new SodiumSign(data);
    }

    @Override
    public PairSymKeys deriveSymmetricKeys(AKey.Symmetric masterKey, int serverId, int keyNumber) {
        throw new UnsupportedOperationException();
    }
}