package io.aether.crypto.sodium;

import io.aether.crypto.*;
import io.aether.nativeLib.SodiumLib;
import java.util.Arrays;

public class SodiumAsymmetricEngine implements CryptoEngine {
    private final AKey.AsymmetricPublic publicKey;
    private final AKey.AsymmetricPrivate privateKey;

    // Константы из libsodium для crypto_box
    private static final int CRYPTO_BOX_NONCEBYTES = 24;
    private static final int CRYPTO_BOX_MACBYTES = 16;

    public SodiumAsymmetricEngine(AKey.AsymmetricPublic publicKey) {
        this.publicKey = publicKey;
        this.privateKey = null;
    }

    public SodiumAsymmetricEngine(AKey.AsymmetricPrivate privateKey, AKey.AsymmetricPublic publicKey) {
        this.privateKey = privateKey;
        this.publicKey = publicKey;
    }

    @Override
    public byte[] encrypt(byte[] data) {
        var nonce = new byte[CRYPTO_BOX_NONCEBYTES];
        var result = new byte[data.length + CRYPTO_BOX_MACBYTES];
        if (privateKey == null) {
            throw new UnsupportedOperationException("This engine is not configured for encryption.");
        }
        if (0 != SodiumLib.crypto_box_easy(result, data, nonce, publicKey.getData(), privateKey.getData())) {
            throw new EncryptException();
        }
        return addArray(nonce, result);
    }

    @Override
    public byte[] decrypt(byte[] data) {
        var nonce = new byte[CRYPTO_BOX_NONCEBYTES];
        System.arraycopy(data, 0, nonce, 0, CRYPTO_BOX_NONCEBYTES);

        var encryptedData = new byte[data.length - CRYPTO_BOX_NONCEBYTES];
        System.arraycopy(data, CRYPTO_BOX_NONCEBYTES, encryptedData, 0, encryptedData.length);

        var result = new byte[encryptedData.length - CRYPTO_BOX_MACBYTES];
        if (privateKey == null) {
            throw new UnsupportedOperationException("This engine is not configured for decryption.");
        }
        if (0 != SodiumLib.crypto_box_open_easy(result, encryptedData, nonce, publicKey.getData(), privateKey.getData())) {
            throw new DecryptException();
        }
        return result;
    }

    private static byte[] addArray(byte[] a1, byte[] a2) {
        var res = new byte[a1.length + a2.length];
        System.arraycopy(a1, 0, res, 0, a1.length);
        System.arraycopy(a2, 0, res, a1.length, a2.length);
        return res;
    }

    @Override
    public CryptoProvider getCryptoProvider() {
        return SodiumCryptoProvider.INSTANCE;
    }

    @Override
    public String getProviderName() {
        return "SODIUM";
    }
}