package io.aether.crypto.sodium;

import io.aether.crypto.AKey;
import io.aether.crypto.Sign;
import io.aether.crypto.Signer;
import io.aether.crypto.CryptoProvider;
import io.aether.nativeLib.SodiumLib;

public class SodiumSigner implements Signer {
    private final AKey.SignPublic publicKey;
    private final AKey.SignPrivate privateKey;

    // Константы из libsodium для подписи
    private static final int CRYPTO_SIGN_BYTES = 64;

    public SodiumSigner(AKey.SignPublic publicKey, AKey.SignPrivate privateKey) {
        this.publicKey = publicKey;
        this.privateKey = privateKey;
    }

    @Override
    public Sign sign(byte[] data) {
        if (privateKey == null) {
            throw new IllegalStateException("This Signer instance is not configured with a private key for signing.");
        }
        byte[] sign = new byte[CRYPTO_SIGN_BYTES];
        SodiumLib.crypto_sign_detached(sign, data, privateKey.getData());
        return new SodiumSign(sign);
    }

    @Override
    public boolean checkSign(byte[] data, Sign sign) {
        if (!(sign instanceof SodiumSign)) {
            throw new IllegalArgumentException("Sign must be a SodiumSign instance");
        }
        if (publicKey == null) {
            throw new IllegalStateException("This Signer instance is not configured with a public key for checking.");
        }
        return SodiumLib.crypto_sign_verify_detached(sign.getSignData(), data, publicKey.getData()) == 0;
    }

    @Override
    public AKey.SignPublic getPublicKey() {
        return publicKey;
    }

    @Override
    public String getProviderName() {
        return "SODIUM";
    }

    @Override
    public CryptoProvider getCryptoProvider() {
        return SodiumCryptoProvider.INSTANCE;
    }
}