package io.aether.crypto.sodium;

import io.aether.crypto.AKey;
import io.aether.crypto.CryptoEngine;
import io.aether.crypto.CryptoProvider;
import io.aether.crypto.DecryptException;
import io.aether.crypto.EncryptException;
import io.aether.nativeLib.SodiumLib;

public class SodiumSymmetricEngine implements CryptoEngine {
    private final AKey.Symmetric key;

    // Константы из libsodium
    private static final int CRYPTO_SECRETBOX_NONCEBYTES = 24;
    private static final int CRYPTO_SECRETBOX_MACBYTES = 16;

    public SodiumSymmetricEngine(AKey.Symmetric key) {
        this.key = key;
    }

    @Override
    public byte[] encrypt(byte[] data) {
        var nonce = new byte[CRYPTO_SECRETBOX_NONCEBYTES];
        var result = new byte[data.length + CRYPTO_SECRETBOX_MACBYTES];
        if (0 != SodiumLib.crypto_secretbox_easy(result, data, nonce, key.getData())) {
            throw new EncryptException();
        }
        return addArray(nonce, result);
    }

    @Override
    public byte[] decrypt(byte[] data) {
        var nonce = new byte[CRYPTO_SECRETBOX_NONCEBYTES];
        System.arraycopy(data, 0, nonce, 0, CRYPTO_SECRETBOX_NONCEBYTES);

        var encryptedData = new byte[data.length - CRYPTO_SECRETBOX_NONCEBYTES];
        System.arraycopy(data, CRYPTO_SECRETBOX_NONCEBYTES, encryptedData, 0, encryptedData.length);

        var result = new byte[encryptedData.length - CRYPTO_SECRETBOX_MACBYTES];
        if (0 != SodiumLib.crypto_secretbox_open_easy(result, encryptedData, nonce, key.getData())) {
            throw new DecryptException();
        }
        return result;
    }

    private static byte[] addArray(byte[] a1, byte[] a2) {
        var res = new byte[a1.length + a2.length];
        System.arraycopy(a1, 0, res, 0, a1.length);
        System.arraycopy(a2, 0, res, a1.length, a2.length);
        return res;
    }

    @Override
    public String getProviderName() {
        return "SODIUM";
    }

    @Override
    public CryptoProvider getCryptoProvider() {
        return SodiumCryptoProvider.INSTANCE;
    }
}