package io.aether.nativeLib;

import java.lang.foreign.FunctionDescriptor;
import java.lang.foreign.Linker;
import java.lang.foreign.MemorySegment;
import java.lang.foreign.SymbolLookup;
import java.lang.foreign.ValueLayout;
import java.lang.invoke.MethodHandle;
import java.lang.foreign.Arena;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.StandardCharsets;
import java.util.UUID;
import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;

import io.aether.logger.Log;
import io.aether.logger.LNode;

public final class SodiumLib {

    // Constants
    public static final int crypto_box_PUBLICKEYBYTES = 32; 
    public static final int crypto_box_SECRETKEYBYTES = 32; 
    public static final int crypto_box_NONCEBYTES = 24; 
    public static final int crypto_box_MACBYTES = 16; 
    public static final int crypto_secretbox_KEYBYTES = 32; 
    public static final int crypto_secretbox_NONCEBYTES = 24; 
    public static final int crypto_secretbox_MACBYTES = 16; 
    public static final int crypto_auth_KEYBYTES = 32; 
    public static final int crypto_auth_BYTES = 32; 
    public static final int crypto_sign_PUBLICKEYBYTES = 32; 
    public static final int crypto_sign_SECRETKEYBYTES = 64; 
    public static final int crypto_sign_BYTES = 64; 
    public static final int crypto_sign_SEEDBYTES = 32; 
    public static final int crypto_generichash_BYTES = 32; 
    public static final int crypto_generichash_BYTES_MIN = 16; 
    public static final int crypto_generichash_BYTES_MAX = 64; 
    public static final int crypto_generichash_KEYBYTES = 32; 
    public static final int crypto_generichash_KEYBYTES_MIN = 16; 
    public static final int crypto_generichash_KEYBYTES_MAX = 64; 
    public static final int sodium_bin2hex_BUF_LEN = 32; 
    public static final int sodium_bin2hex_LEN_MULT = 2; 
    public static final int sodium_hex2bin_BUF_LEN = 16; 

public static class crypto_box_keypair {
    public byte[] pk;
    public byte[] sk;

    public crypto_box_keypair() {
        this.pk = new byte[crypto_box_PUBLICKEYBYTES];
        this.sk = new byte[crypto_box_SECRETKEYBYTES];
    }

    public crypto_box_keypair(byte[] pk, byte[] sk) {
        this.pk = pk;
        this.sk = sk;
    }

    public byte[] getPk() {
        return this.pk;
    }

    public byte[] getSk() {
        return this.sk;
    }

    public long serialize(MemorySegment segment, long offset, Arena arena) {
        long currentOffset = offset;
        MemorySegment.copy(MemorySegment.ofArray(this.pk), 0, segment, currentOffset, this.pk.length);
        currentOffset += 1 * this.pk.length;
        MemorySegment.copy(MemorySegment.ofArray(this.sk), 0, segment, currentOffset, this.sk.length);
        currentOffset += 1 * this.sk.length;
        return currentOffset;
    }

    public void deserialize(MemorySegment segment, long offset) {
        long currentOffset = offset;
        MemorySegment.copy(segment, currentOffset, MemorySegment.ofArray(this.pk), 0, this.pk.length);
        currentOffset += 1 * this.pk.length;
        MemorySegment.copy(segment, currentOffset, MemorySegment.ofArray(this.sk), 0, this.sk.length);
        currentOffset += 1 * this.sk.length;
    }

    public int getSerializedSize() {
        return 64;
    }
    public static int getStaticSize() {
        return 64;
    }

}

public static class crypto_sign_keypair {
    public byte[] pk;
    public byte[] sk;

    public crypto_sign_keypair() {
        this.pk = new byte[crypto_sign_PUBLICKEYBYTES];
        this.sk = new byte[crypto_sign_SECRETKEYBYTES];
    }

    public crypto_sign_keypair(byte[] pk, byte[] sk) {
        this.pk = pk;
        this.sk = sk;
    }

    public byte[] getPk() {
        return this.pk;
    }

    public byte[] getSk() {
        return this.sk;
    }

    public long serialize(MemorySegment segment, long offset, Arena arena) {
        long currentOffset = offset;
        MemorySegment.copy(MemorySegment.ofArray(this.pk), 0, segment, currentOffset, this.pk.length);
        currentOffset += 1 * this.pk.length;
        MemorySegment.copy(MemorySegment.ofArray(this.sk), 0, segment, currentOffset, this.sk.length);
        currentOffset += 1 * this.sk.length;
        return currentOffset;
    }

    public void deserialize(MemorySegment segment, long offset) {
        long currentOffset = offset;
        MemorySegment.copy(segment, currentOffset, MemorySegment.ofArray(this.pk), 0, this.pk.length);
        currentOffset += 1 * this.pk.length;
        MemorySegment.copy(segment, currentOffset, MemorySegment.ofArray(this.sk), 0, this.sk.length);
        currentOffset += 1 * this.sk.length;
    }

    public int getSerializedSize() {
        return 96;
    }
    public static int getStaticSize() {
        return 96;
    }

}

public static class crypto_generichash_state {
    public long[] h;
    public byte[] buf;
    public long buflen;
    public long buflen_max;
    public long[] pad;

    public crypto_generichash_state() {
        this.h = new long[8];
        this.buf = new byte[64];
        this.pad = new long[2];
    }

    public crypto_generichash_state(long[] h, byte[] buf, long buflen, long buflen_max, long[] pad) {
        this.h = h;
        this.buf = buf;
        this.buflen = buflen;
        this.buflen_max = buflen_max;
        this.pad = pad;
    }

    public long[] getH() {
        return this.h;
    }

    public byte[] getBuf() {
        return this.buf;
    }

    public long getBuflen() {
        return this.buflen;
    }

    public long getBuflen_max() {
        return this.buflen_max;
    }

    public long[] getPad() {
        return this.pad;
    }

    public long serialize(MemorySegment segment, long offset, Arena arena) {
        long currentOffset = offset;
        for (int i = 0; i < this.h.length; i++) {    segment.set(ValueLayout.JAVA_LONG, currentOffset + i * 8, this.h[i]);}
        currentOffset += 8 * this.h.length;
        MemorySegment.copy(MemorySegment.ofArray(this.buf), 0, segment, currentOffset, this.buf.length);
        currentOffset += 1 * this.buf.length;
        segment.set(ValueLayout.JAVA_LONG, currentOffset, this.buflen);
        currentOffset += 8;
        segment.set(ValueLayout.JAVA_LONG, currentOffset, this.buflen_max);
        currentOffset += 8;
        for (int i = 0; i < this.pad.length; i++) {    segment.set(ValueLayout.JAVA_LONG, currentOffset + i * 8, this.pad[i]);}
        currentOffset += 8 * this.pad.length;
        return currentOffset;
    }

    public void deserialize(MemorySegment segment, long offset) {
        long currentOffset = offset;
        for (int i = 0; i < this.h.length; i++) {    this.h[i] = segment.get(ValueLayout.JAVA_LONG, currentOffset + i * 8);}
        currentOffset += 8 * this.h.length;
        MemorySegment.copy(segment, currentOffset, MemorySegment.ofArray(this.buf), 0, this.buf.length);
        currentOffset += 1 * this.buf.length;
        this.buflen = segment.get(ValueLayout.JAVA_LONG, currentOffset);
        currentOffset += 8;
        this.buflen_max = segment.get(ValueLayout.JAVA_LONG, currentOffset);
        currentOffset += 8;
        for (int i = 0; i < this.pad.length; i++) {    this.pad[i] = segment.get(ValueLayout.JAVA_LONG, currentOffset + i * 8);}
        currentOffset += 8 * this.pad.length;
    }

    public int getSerializedSize() {
        return 160;
    }
    public static int getStaticSize() {
        return 160;
    }

}

public static class crypto_auth_state {
    public long[] h;
    public byte[] buf;
    public long buflen;
    public long buflen_max;
    public long[] pad;
    public byte[] k;

    public crypto_auth_state() {
        this.h = new long[8];
        this.buf = new byte[64];
        this.pad = new long[2];
        this.k = new byte[32];
    }

    public crypto_auth_state(long[] h, byte[] buf, long buflen, long buflen_max, long[] pad, byte[] k) {
        this.h = h;
        this.buf = buf;
        this.buflen = buflen;
        this.buflen_max = buflen_max;
        this.pad = pad;
        this.k = k;
    }

    public long[] getH() {
        return this.h;
    }

    public byte[] getBuf() {
        return this.buf;
    }

    public long getBuflen() {
        return this.buflen;
    }

    public long getBuflen_max() {
        return this.buflen_max;
    }

    public long[] getPad() {
        return this.pad;
    }

    public byte[] getK() {
        return this.k;
    }

    public long serialize(MemorySegment segment, long offset, Arena arena) {
        long currentOffset = offset;
        for (int i = 0; i < this.h.length; i++) {    segment.set(ValueLayout.JAVA_LONG, currentOffset + i * 8, this.h[i]);}
        currentOffset += 8 * this.h.length;
        MemorySegment.copy(MemorySegment.ofArray(this.buf), 0, segment, currentOffset, this.buf.length);
        currentOffset += 1 * this.buf.length;
        segment.set(ValueLayout.JAVA_LONG, currentOffset, this.buflen);
        currentOffset += 8;
        segment.set(ValueLayout.JAVA_LONG, currentOffset, this.buflen_max);
        currentOffset += 8;
        for (int i = 0; i < this.pad.length; i++) {    segment.set(ValueLayout.JAVA_LONG, currentOffset + i * 8, this.pad[i]);}
        currentOffset += 8 * this.pad.length;
        MemorySegment.copy(MemorySegment.ofArray(this.k), 0, segment, currentOffset, this.k.length);
        currentOffset += 1 * this.k.length;
        return currentOffset;
    }

    public void deserialize(MemorySegment segment, long offset) {
        long currentOffset = offset;
        for (int i = 0; i < this.h.length; i++) {    this.h[i] = segment.get(ValueLayout.JAVA_LONG, currentOffset + i * 8);}
        currentOffset += 8 * this.h.length;
        MemorySegment.copy(segment, currentOffset, MemorySegment.ofArray(this.buf), 0, this.buf.length);
        currentOffset += 1 * this.buf.length;
        this.buflen = segment.get(ValueLayout.JAVA_LONG, currentOffset);
        currentOffset += 8;
        this.buflen_max = segment.get(ValueLayout.JAVA_LONG, currentOffset);
        currentOffset += 8;
        for (int i = 0; i < this.pad.length; i++) {    this.pad[i] = segment.get(ValueLayout.JAVA_LONG, currentOffset + i * 8);}
        currentOffset += 8 * this.pad.length;
        MemorySegment.copy(segment, currentOffset, MemorySegment.ofArray(this.k), 0, this.k.length);
        currentOffset += 1 * this.k.length;
    }

    public int getSerializedSize() {
        return 192;
    }
    public static int getStaticSize() {
        return 192;
    }

}

    private static final MethodHandle CRYPTO_BOX_KEYPAIR_HANDLE;
    private static final MethodHandle CRYPTO_BOX_EASY_HANDLE;
    private static final MethodHandle CRYPTO_BOX_OPEN_EASY_HANDLE;
    private static final MethodHandle CRYPTO_SECRETBOX_KEYGEN_HANDLE;
    private static final MethodHandle CRYPTO_SECRETBOX_EASY_HANDLE;
    private static final MethodHandle CRYPTO_SECRETBOX_OPEN_EASY_HANDLE;
    private static final MethodHandle CRYPTO_SIGN_KEYPAIR_HANDLE;
    private static final MethodHandle CRYPTO_SIGN_HANDLE;
    private static final MethodHandle CRYPTO_SIGN_OPEN_HANDLE;
    private static final MethodHandle CRYPTO_SIGN_DETACHED_HANDLE;
    private static final MethodHandle CRYPTO_SIGN_VERIFY_DETACHED_HANDLE;
    private static final MethodHandle CRYPTO_GENERICHASH_HANDLE;
    private static final MethodHandle CRYPTO_GENERICHASH_INIT_HANDLE;
    private static final MethodHandle CRYPTO_GENERICHASH_UPDATE_HANDLE;
    private static final MethodHandle CRYPTO_GENERICHASH_FINAL_HANDLE;
    private static final MethodHandle CRYPTO_AUTH_KEYGEN_HANDLE;
    private static final MethodHandle CRYPTO_AUTH_HANDLE;
    private static final MethodHandle CRYPTO_AUTH_VERIFY_HANDLE;
    private static final MethodHandle SODIUM_BIN2HEX_HANDLE;
    private static final MethodHandle SODIUM_HEX2BIN_HANDLE;

    private static final Linker LINKER = Linker.nativeLinker();
    private static final SymbolLookup SYMBOL_LOOKUP;

    static {
        io.aether.utils.RU.loadLibrary("sodium");
        SYMBOL_LOOKUP = SymbolLookup.loaderLookup();

        CRYPTO_BOX_KEYPAIR_HANDLE = LINKER.downcallHandle(
            SYMBOL_LOOKUP.find("crypto_box_keypair").orElseThrow(() -> new UnsatisfiedLinkError("Метод crypto_box_keypair не найден")), 
            FunctionDescriptor.of(ValueLayout.JAVA_INT, ValueLayout.ADDRESS)
        );
        CRYPTO_BOX_EASY_HANDLE = LINKER.downcallHandle(
            SYMBOL_LOOKUP.find("crypto_box_easy").orElseThrow(() -> new UnsatisfiedLinkError("Метод crypto_box_easy не найден")), 
            FunctionDescriptor.of(ValueLayout.JAVA_INT, ValueLayout.ADDRESS, ValueLayout.ADDRESS, ValueLayout.ADDRESS, ValueLayout.ADDRESS, ValueLayout.ADDRESS)
        );
        CRYPTO_BOX_OPEN_EASY_HANDLE = LINKER.downcallHandle(
            SYMBOL_LOOKUP.find("crypto_box_open_easy").orElseThrow(() -> new UnsatisfiedLinkError("Метод crypto_box_open_easy не найден")), 
            FunctionDescriptor.of(ValueLayout.JAVA_INT, ValueLayout.ADDRESS, ValueLayout.ADDRESS, ValueLayout.ADDRESS, ValueLayout.ADDRESS, ValueLayout.ADDRESS)
        );
        CRYPTO_SECRETBOX_KEYGEN_HANDLE = LINKER.downcallHandle(
            SYMBOL_LOOKUP.find("crypto_secretbox_keygen").orElseThrow(() -> new UnsatisfiedLinkError("Метод crypto_secretbox_keygen не найден")), 
            FunctionDescriptor.of(ValueLayout.JAVA_INT, ValueLayout.ADDRESS)
        );
        CRYPTO_SECRETBOX_EASY_HANDLE = LINKER.downcallHandle(
            SYMBOL_LOOKUP.find("crypto_secretbox_easy").orElseThrow(() -> new UnsatisfiedLinkError("Метод crypto_secretbox_easy не найден")), 
            FunctionDescriptor.of(ValueLayout.JAVA_INT, ValueLayout.ADDRESS, ValueLayout.ADDRESS, ValueLayout.ADDRESS, ValueLayout.ADDRESS)
        );
        CRYPTO_SECRETBOX_OPEN_EASY_HANDLE = LINKER.downcallHandle(
            SYMBOL_LOOKUP.find("crypto_secretbox_open_easy").orElseThrow(() -> new UnsatisfiedLinkError("Метод crypto_secretbox_open_easy не найден")), 
            FunctionDescriptor.of(ValueLayout.JAVA_INT, ValueLayout.ADDRESS, ValueLayout.ADDRESS, ValueLayout.ADDRESS, ValueLayout.ADDRESS)
        );
        CRYPTO_SIGN_KEYPAIR_HANDLE = LINKER.downcallHandle(
            SYMBOL_LOOKUP.find("crypto_sign_keypair").orElseThrow(() -> new UnsatisfiedLinkError("Метод crypto_sign_keypair не найден")), 
            FunctionDescriptor.of(ValueLayout.JAVA_INT, ValueLayout.ADDRESS)
        );
        CRYPTO_SIGN_HANDLE = LINKER.downcallHandle(
            SYMBOL_LOOKUP.find("crypto_sign").orElseThrow(() -> new UnsatisfiedLinkError("Метод crypto_sign не найден")), 
            FunctionDescriptor.of(ValueLayout.JAVA_INT, ValueLayout.ADDRESS, ValueLayout.ADDRESS, ValueLayout.ADDRESS)
        );
        CRYPTO_SIGN_OPEN_HANDLE = LINKER.downcallHandle(
            SYMBOL_LOOKUP.find("crypto_sign_open").orElseThrow(() -> new UnsatisfiedLinkError("Метод crypto_sign_open не найден")), 
            FunctionDescriptor.of(ValueLayout.JAVA_INT, ValueLayout.ADDRESS, ValueLayout.ADDRESS, ValueLayout.ADDRESS)
        );
        CRYPTO_SIGN_DETACHED_HANDLE = LINKER.downcallHandle(
            SYMBOL_LOOKUP.find("crypto_sign_detached").orElseThrow(() -> new UnsatisfiedLinkError("Метод crypto_sign_detached не найден")), 
            FunctionDescriptor.of(ValueLayout.JAVA_INT, ValueLayout.ADDRESS, ValueLayout.ADDRESS, ValueLayout.ADDRESS)
        );
        CRYPTO_SIGN_VERIFY_DETACHED_HANDLE = LINKER.downcallHandle(
            SYMBOL_LOOKUP.find("crypto_sign_verify_detached").orElseThrow(() -> new UnsatisfiedLinkError("Метод crypto_sign_verify_detached не найден")), 
            FunctionDescriptor.of(ValueLayout.JAVA_INT, ValueLayout.ADDRESS, ValueLayout.ADDRESS, ValueLayout.ADDRESS)
        );
        CRYPTO_GENERICHASH_HANDLE = LINKER.downcallHandle(
            SYMBOL_LOOKUP.find("crypto_generichash").orElseThrow(() -> new UnsatisfiedLinkError("Метод crypto_generichash не найден")), 
            FunctionDescriptor.of(ValueLayout.JAVA_INT, ValueLayout.ADDRESS, ValueLayout.ADDRESS, ValueLayout.ADDRESS)
        );
        CRYPTO_GENERICHASH_INIT_HANDLE = LINKER.downcallHandle(
            SYMBOL_LOOKUP.find("crypto_generichash_init").orElseThrow(() -> new UnsatisfiedLinkError("Метод crypto_generichash_init не найден")), 
            FunctionDescriptor.of(ValueLayout.JAVA_INT, ValueLayout.ADDRESS, ValueLayout.ADDRESS, ValueLayout.JAVA_LONG)
        );
        CRYPTO_GENERICHASH_UPDATE_HANDLE = LINKER.downcallHandle(
            SYMBOL_LOOKUP.find("crypto_generichash_update").orElseThrow(() -> new UnsatisfiedLinkError("Метод crypto_generichash_update не найден")), 
            FunctionDescriptor.of(ValueLayout.JAVA_INT, ValueLayout.ADDRESS, ValueLayout.ADDRESS, ValueLayout.JAVA_LONG)
        );
        CRYPTO_GENERICHASH_FINAL_HANDLE = LINKER.downcallHandle(
            SYMBOL_LOOKUP.find("crypto_generichash_final").orElseThrow(() -> new UnsatisfiedLinkError("Метод crypto_generichash_final не найден")), 
            FunctionDescriptor.of(ValueLayout.JAVA_INT, ValueLayout.ADDRESS, ValueLayout.ADDRESS, ValueLayout.JAVA_LONG)
        );
        CRYPTO_AUTH_KEYGEN_HANDLE = LINKER.downcallHandle(
            SYMBOL_LOOKUP.find("crypto_auth_keygen").orElseThrow(() -> new UnsatisfiedLinkError("Метод crypto_auth_keygen не найден")), 
            FunctionDescriptor.of(ValueLayout.JAVA_INT, ValueLayout.ADDRESS)
        );
        CRYPTO_AUTH_HANDLE = LINKER.downcallHandle(
            SYMBOL_LOOKUP.find("crypto_auth").orElseThrow(() -> new UnsatisfiedLinkError("Метод crypto_auth не найден")), 
            FunctionDescriptor.of(ValueLayout.JAVA_INT, ValueLayout.ADDRESS, ValueLayout.ADDRESS, ValueLayout.ADDRESS)
        );
        CRYPTO_AUTH_VERIFY_HANDLE = LINKER.downcallHandle(
            SYMBOL_LOOKUP.find("crypto_auth_verify").orElseThrow(() -> new UnsatisfiedLinkError("Метод crypto_auth_verify не найден")), 
            FunctionDescriptor.of(ValueLayout.JAVA_INT, ValueLayout.ADDRESS, ValueLayout.ADDRESS, ValueLayout.ADDRESS)
        );
        SODIUM_BIN2HEX_HANDLE = LINKER.downcallHandle(
            SYMBOL_LOOKUP.find("sodium_bin2hex").orElseThrow(() -> new UnsatisfiedLinkError("Метод sodium_bin2hex не найден")), 
            FunctionDescriptor.of(ValueLayout.ADDRESS, ValueLayout.ADDRESS, ValueLayout.ADDRESS)
        );
        SODIUM_HEX2BIN_HANDLE = LINKER.downcallHandle(
            SYMBOL_LOOKUP.find("sodium_hex2bin").orElseThrow(() -> new UnsatisfiedLinkError("Метод sodium_hex2bin не найден")), 
            FunctionDescriptor.of(ValueLayout.JAVA_INT, ValueLayout.ADDRESS, ValueLayout.ADDRESS, ValueLayout.JAVA_LONG, ValueLayout.ADDRESS, ValueLayout.ADDRESS, ValueLayout.ADDRESS)
        );
    }

    public static int crypto_box_keypair(crypto_box_keypair keypair) {
        if (keypair == null) throw new IllegalArgumentException("Параметр 'keypair' не может быть null.");
        try (Arena arena = Arena.ofConfined()) {
            MemorySegment keypairSegment = keypair == null ? MemorySegment.NULL : arena.allocate(crypto_box_keypair.getStaticSize());
            if (keypair != null) keypair.serialize(keypairSegment, 0, arena);

            int result = (int) CRYPTO_BOX_KEYPAIR_HANDLE.invokeExact(keypairSegment);
            if (keypair != null) keypair.deserialize(keypairSegment, 0);
            return result;
        } catch (Throwable e) {
            throw new RuntimeException("Ошибка вызова нативного метода crypto_box_keypair", e);
        }
    }

    public static int crypto_box_easy(byte[] c, byte[] m, byte[] n, byte[] pk, byte[] sk) {
        if (c == null) throw new IllegalArgumentException("Параметр 'c' не может быть null.");
        if (m == null) throw new IllegalArgumentException("Параметр 'm' не может быть null.");
        if (n == null) throw new IllegalArgumentException("Параметр 'n' не может быть null.");
        if (n != null && n.length != crypto_box_NONCEBYTES) throw new IllegalArgumentException("Параметр 'n' должен иметь размер crypto_box_NONCEBYTES.");
        if (pk == null) throw new IllegalArgumentException("Параметр 'pk' не может быть null.");
        if (pk != null && pk.length != crypto_box_PUBLICKEYBYTES) throw new IllegalArgumentException("Параметр 'pk' должен иметь размер crypto_box_PUBLICKEYBYTES.");
        if (sk == null) throw new IllegalArgumentException("Параметр 'sk' не может быть null.");
        if (sk != null && sk.length != crypto_box_SECRETKEYBYTES) throw new IllegalArgumentException("Параметр 'sk' должен иметь размер crypto_box_SECRETKEYBYTES.");
        try (Arena arena = Arena.ofConfined()) {
            MemorySegment cSegment = c == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, c);
            MemorySegment mSegment = m == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, m);
            MemorySegment nSegment = n == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, n);
            MemorySegment pkSegment = pk == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, pk);
            MemorySegment skSegment = sk == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, sk);

            int result = (int) CRYPTO_BOX_EASY_HANDLE.invokeExact(cSegment, mSegment, nSegment, pkSegment, skSegment);
            if (c != null) MemorySegment.copy(cSegment, 0, MemorySegment.ofArray(c), 0, c.length);
            if (m != null) MemorySegment.copy(mSegment, 0, MemorySegment.ofArray(m), 0, m.length);
            if (n != null) MemorySegment.copy(nSegment, 0, MemorySegment.ofArray(n), 0, n.length);
            if (pk != null) MemorySegment.copy(pkSegment, 0, MemorySegment.ofArray(pk), 0, pk.length);
            if (sk != null) MemorySegment.copy(skSegment, 0, MemorySegment.ofArray(sk), 0, sk.length);
            return result;
        } catch (Throwable e) {
            throw new RuntimeException("Ошибка вызова нативного метода crypto_box_easy", e);
        }
    }

    public static int crypto_box_open_easy(byte[] m, byte[] c, byte[] n, byte[] pk, byte[] sk) {
        if (m == null) throw new IllegalArgumentException("Параметр 'm' не может быть null.");
        if (c == null) throw new IllegalArgumentException("Параметр 'c' не может быть null.");
        if (n == null) throw new IllegalArgumentException("Параметр 'n' не может быть null.");
        if (n != null && n.length != crypto_box_NONCEBYTES) throw new IllegalArgumentException("Параметр 'n' должен иметь размер crypto_box_NONCEBYTES.");
        if (pk == null) throw new IllegalArgumentException("Параметр 'pk' не может быть null.");
        if (pk != null && pk.length != crypto_box_PUBLICKEYBYTES) throw new IllegalArgumentException("Параметр 'pk' должен иметь размер crypto_box_PUBLICKEYBYTES.");
        if (sk == null) throw new IllegalArgumentException("Параметр 'sk' не может быть null.");
        if (sk != null && sk.length != crypto_box_SECRETKEYBYTES) throw new IllegalArgumentException("Параметр 'sk' должен иметь размер crypto_box_SECRETKEYBYTES.");
        try (Arena arena = Arena.ofConfined()) {
            MemorySegment mSegment = m == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, m);
            MemorySegment cSegment = c == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, c);
            MemorySegment nSegment = n == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, n);
            MemorySegment pkSegment = pk == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, pk);
            MemorySegment skSegment = sk == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, sk);

            int result = (int) CRYPTO_BOX_OPEN_EASY_HANDLE.invokeExact(mSegment, cSegment, nSegment, pkSegment, skSegment);
            if (m != null) MemorySegment.copy(mSegment, 0, MemorySegment.ofArray(m), 0, m.length);
            if (c != null) MemorySegment.copy(cSegment, 0, MemorySegment.ofArray(c), 0, c.length);
            if (n != null) MemorySegment.copy(nSegment, 0, MemorySegment.ofArray(n), 0, n.length);
            if (pk != null) MemorySegment.copy(pkSegment, 0, MemorySegment.ofArray(pk), 0, pk.length);
            if (sk != null) MemorySegment.copy(skSegment, 0, MemorySegment.ofArray(sk), 0, sk.length);
            return result;
        } catch (Throwable e) {
            throw new RuntimeException("Ошибка вызова нативного метода crypto_box_open_easy", e);
        }
    }

    public static int crypto_secretbox_keygen(byte[] key) {
        if (key == null) throw new IllegalArgumentException("Параметр 'key' не может быть null.");
        if (key != null && key.length != crypto_secretbox_KEYBYTES) throw new IllegalArgumentException("Параметр 'key' должен иметь размер crypto_secretbox_KEYBYTES.");
        try (Arena arena = Arena.ofConfined()) {
            MemorySegment keySegment = key == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, key);

            int result = (int) CRYPTO_SECRETBOX_KEYGEN_HANDLE.invokeExact(keySegment);
            if (key != null) MemorySegment.copy(keySegment, 0, MemorySegment.ofArray(key), 0, key.length);
            return result;
        } catch (Throwable e) {
            throw new RuntimeException("Ошибка вызова нативного метода crypto_secretbox_keygen", e);
        }
    }

    public static int crypto_secretbox_easy(byte[] c, byte[] m, byte[] n, byte[] k) {
        if (c == null) throw new IllegalArgumentException("Параметр 'c' не может быть null.");
        if (m == null) throw new IllegalArgumentException("Параметр 'm' не может быть null.");
        if (n == null) throw new IllegalArgumentException("Параметр 'n' не может быть null.");
        if (n != null && n.length != crypto_secretbox_NONCEBYTES) throw new IllegalArgumentException("Параметр 'n' должен иметь размер crypto_secretbox_NONCEBYTES.");
        if (k == null) throw new IllegalArgumentException("Параметр 'k' не может быть null.");
        if (k != null && k.length != crypto_secretbox_KEYBYTES) throw new IllegalArgumentException("Параметр 'k' должен иметь размер crypto_secretbox_KEYBYTES.");
        try (Arena arena = Arena.ofConfined()) {
            MemorySegment cSegment = c == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, c);
            MemorySegment mSegment = m == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, m);
            MemorySegment nSegment = n == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, n);
            MemorySegment kSegment = k == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, k);

            int result = (int) CRYPTO_SECRETBOX_EASY_HANDLE.invokeExact(cSegment, mSegment, nSegment, kSegment);
            if (c != null) MemorySegment.copy(cSegment, 0, MemorySegment.ofArray(c), 0, c.length);
            if (m != null) MemorySegment.copy(mSegment, 0, MemorySegment.ofArray(m), 0, m.length);
            if (n != null) MemorySegment.copy(nSegment, 0, MemorySegment.ofArray(n), 0, n.length);
            if (k != null) MemorySegment.copy(kSegment, 0, MemorySegment.ofArray(k), 0, k.length);
            return result;
        } catch (Throwable e) {
            throw new RuntimeException("Ошибка вызова нативного метода crypto_secretbox_easy", e);
        }
    }

    public static int crypto_secretbox_open_easy(byte[] m, byte[] c, byte[] n, byte[] k) {
        if (m == null) throw new IllegalArgumentException("Параметр 'm' не может быть null.");
        if (c == null) throw new IllegalArgumentException("Параметр 'c' не может быть null.");
        if (n == null) throw new IllegalArgumentException("Параметр 'n' не может быть null.");
        if (n != null && n.length != crypto_secretbox_NONCEBYTES) throw new IllegalArgumentException("Параметр 'n' должен иметь размер crypto_secretbox_NONCEBYTES.");
        if (k == null) throw new IllegalArgumentException("Параметр 'k' не может быть null.");
        if (k != null && k.length != crypto_secretbox_KEYBYTES) throw new IllegalArgumentException("Параметр 'k' должен иметь размер crypto_secretbox_KEYBYTES.");
        try (Arena arena = Arena.ofConfined()) {
            MemorySegment mSegment = m == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, m);
            MemorySegment cSegment = c == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, c);
            MemorySegment nSegment = n == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, n);
            MemorySegment kSegment = k == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, k);

            int result = (int) CRYPTO_SECRETBOX_OPEN_EASY_HANDLE.invokeExact(mSegment, cSegment, nSegment, kSegment);
            if (m != null) MemorySegment.copy(mSegment, 0, MemorySegment.ofArray(m), 0, m.length);
            if (c != null) MemorySegment.copy(cSegment, 0, MemorySegment.ofArray(c), 0, c.length);
            if (n != null) MemorySegment.copy(nSegment, 0, MemorySegment.ofArray(n), 0, n.length);
            if (k != null) MemorySegment.copy(kSegment, 0, MemorySegment.ofArray(k), 0, k.length);
            return result;
        } catch (Throwable e) {
            throw new RuntimeException("Ошибка вызова нативного метода crypto_secretbox_open_easy", e);
        }
    }

    public static int crypto_sign_keypair(crypto_sign_keypair keypair) {
        if (keypair == null) throw new IllegalArgumentException("Параметр 'keypair' не может быть null.");
        try (Arena arena = Arena.ofConfined()) {
            MemorySegment keypairSegment = keypair == null ? MemorySegment.NULL : arena.allocate(crypto_sign_keypair.getStaticSize());
            if (keypair != null) keypair.serialize(keypairSegment, 0, arena);

            int result = (int) CRYPTO_SIGN_KEYPAIR_HANDLE.invokeExact(keypairSegment);
            if (keypair != null) keypair.deserialize(keypairSegment, 0);
            return result;
        } catch (Throwable e) {
            throw new RuntimeException("Ошибка вызова нативного метода crypto_sign_keypair", e);
        }
    }

    public static int crypto_sign(byte[] sm, byte[] m, byte[] sk) {
        if (sm == null) throw new IllegalArgumentException("Параметр 'sm' не может быть null.");
        if (m == null) throw new IllegalArgumentException("Параметр 'm' не может быть null.");
        if (sk == null) throw new IllegalArgumentException("Параметр 'sk' не может быть null.");
        if (sk != null && sk.length != crypto_sign_SECRETKEYBYTES) throw new IllegalArgumentException("Параметр 'sk' должен иметь размер crypto_sign_SECRETKEYBYTES.");
        try (Arena arena = Arena.ofConfined()) {
            MemorySegment smSegment = sm == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, sm);
            MemorySegment mSegment = m == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, m);
            MemorySegment skSegment = sk == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, sk);

            int result = (int) CRYPTO_SIGN_HANDLE.invokeExact(smSegment, mSegment, skSegment);
            if (sm != null) MemorySegment.copy(smSegment, 0, MemorySegment.ofArray(sm), 0, sm.length);
            if (m != null) MemorySegment.copy(mSegment, 0, MemorySegment.ofArray(m), 0, m.length);
            if (sk != null) MemorySegment.copy(skSegment, 0, MemorySegment.ofArray(sk), 0, sk.length);
            return result;
        } catch (Throwable e) {
            throw new RuntimeException("Ошибка вызова нативного метода crypto_sign", e);
        }
    }

    public static int crypto_sign_open(byte[] m, byte[] sm, byte[] pk) {
        if (m == null) throw new IllegalArgumentException("Параметр 'm' не может быть null.");
        if (sm == null) throw new IllegalArgumentException("Параметр 'sm' не может быть null.");
        if (pk == null) throw new IllegalArgumentException("Параметр 'pk' не может быть null.");
        if (pk != null && pk.length != crypto_sign_PUBLICKEYBYTES) throw new IllegalArgumentException("Параметр 'pk' должен иметь размер crypto_sign_PUBLICKEYBYTES.");
        try (Arena arena = Arena.ofConfined()) {
            MemorySegment mSegment = m == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, m);
            MemorySegment smSegment = sm == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, sm);
            MemorySegment pkSegment = pk == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, pk);

            int result = (int) CRYPTO_SIGN_OPEN_HANDLE.invokeExact(mSegment, smSegment, pkSegment);
            if (m != null) MemorySegment.copy(mSegment, 0, MemorySegment.ofArray(m), 0, m.length);
            if (sm != null) MemorySegment.copy(smSegment, 0, MemorySegment.ofArray(sm), 0, sm.length);
            if (pk != null) MemorySegment.copy(pkSegment, 0, MemorySegment.ofArray(pk), 0, pk.length);
            return result;
        } catch (Throwable e) {
            throw new RuntimeException("Ошибка вызова нативного метода crypto_sign_open", e);
        }
    }

    public static int crypto_sign_detached(byte[] sig, byte[] m, byte[] sk) {
        if (sig == null) throw new IllegalArgumentException("Параметр 'sig' не может быть null.");
        if (sig != null && sig.length != crypto_sign_BYTES) throw new IllegalArgumentException("Параметр 'sig' должен иметь размер crypto_sign_BYTES.");
        if (m == null) throw new IllegalArgumentException("Параметр 'm' не может быть null.");
        if (sk == null) throw new IllegalArgumentException("Параметр 'sk' не может быть null.");
        if (sk != null && sk.length != crypto_sign_SECRETKEYBYTES) throw new IllegalArgumentException("Параметр 'sk' должен иметь размер crypto_sign_SECRETKEYBYTES.");
        try (Arena arena = Arena.ofConfined()) {
            MemorySegment sigSegment = sig == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, sig);
            MemorySegment mSegment = m == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, m);
            MemorySegment skSegment = sk == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, sk);

            int result = (int) CRYPTO_SIGN_DETACHED_HANDLE.invokeExact(sigSegment, mSegment, skSegment);
            if (sig != null) MemorySegment.copy(sigSegment, 0, MemorySegment.ofArray(sig), 0, sig.length);
            if (m != null) MemorySegment.copy(mSegment, 0, MemorySegment.ofArray(m), 0, m.length);
            if (sk != null) MemorySegment.copy(skSegment, 0, MemorySegment.ofArray(sk), 0, sk.length);
            return result;
        } catch (Throwable e) {
            throw new RuntimeException("Ошибка вызова нативного метода crypto_sign_detached", e);
        }
    }

    public static int crypto_sign_verify_detached(byte[] sig, byte[] m, byte[] pk) {
        if (sig == null) throw new IllegalArgumentException("Параметр 'sig' не может быть null.");
        if (sig != null && sig.length != crypto_sign_BYTES) throw new IllegalArgumentException("Параметр 'sig' должен иметь размер crypto_sign_BYTES.");
        if (m == null) throw new IllegalArgumentException("Параметр 'm' не может быть null.");
        if (pk == null) throw new IllegalArgumentException("Параметр 'pk' не может быть null.");
        if (pk != null && pk.length != crypto_sign_PUBLICKEYBYTES) throw new IllegalArgumentException("Параметр 'pk' должен иметь размер crypto_sign_PUBLICKEYBYTES.");
        try (Arena arena = Arena.ofConfined()) {
            MemorySegment sigSegment = sig == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, sig);
            MemorySegment mSegment = m == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, m);
            MemorySegment pkSegment = pk == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, pk);

            int result = (int) CRYPTO_SIGN_VERIFY_DETACHED_HANDLE.invokeExact(sigSegment, mSegment, pkSegment);
            if (sig != null) MemorySegment.copy(sigSegment, 0, MemorySegment.ofArray(sig), 0, sig.length);
            if (m != null) MemorySegment.copy(mSegment, 0, MemorySegment.ofArray(m), 0, m.length);
            if (pk != null) MemorySegment.copy(pkSegment, 0, MemorySegment.ofArray(pk), 0, pk.length);
            return result;
        } catch (Throwable e) {
            throw new RuntimeException("Ошибка вызова нативного метода crypto_sign_verify_detached", e);
        }
    }

    public static int crypto_generichash(byte[] out, byte[] in, byte[] key) {
        if (out == null) throw new IllegalArgumentException("Параметр 'out' не может быть null.");
        if (in == null) throw new IllegalArgumentException("Параметр 'in' не может быть null.");
        if (key == null) throw new IllegalArgumentException("Параметр 'key' не может быть null.");
        try (Arena arena = Arena.ofConfined()) {
            MemorySegment outSegment = out == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, out);
            MemorySegment inSegment = in == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, in);
            MemorySegment keySegment = key == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, key);

            int result = (int) CRYPTO_GENERICHASH_HANDLE.invokeExact(outSegment, inSegment, keySegment);
            if (out != null) MemorySegment.copy(outSegment, 0, MemorySegment.ofArray(out), 0, out.length);
            if (in != null) MemorySegment.copy(inSegment, 0, MemorySegment.ofArray(in), 0, in.length);
            if (key != null) MemorySegment.copy(keySegment, 0, MemorySegment.ofArray(key), 0, key.length);
            return result;
        } catch (Throwable e) {
            throw new RuntimeException("Ошибка вызова нативного метода crypto_generichash", e);
        }
    }

    public static int crypto_generichash_init(crypto_generichash_state state, byte[] key, long keylen) {
        if (state == null) throw new IllegalArgumentException("Параметр 'state' не может быть null.");
        if (key == null) throw new IllegalArgumentException("Параметр 'key' не может быть null.");
        try (Arena arena = Arena.ofConfined()) {
            MemorySegment stateSegment = state == null ? MemorySegment.NULL : arena.allocate(crypto_generichash_state.getStaticSize());
            if (state != null) state.serialize(stateSegment, 0, arena);
            MemorySegment keySegment = key == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, key);

            int result = (int) CRYPTO_GENERICHASH_INIT_HANDLE.invokeExact(stateSegment, keySegment, keylen);
            if (state != null) state.deserialize(stateSegment, 0);
            if (key != null) MemorySegment.copy(keySegment, 0, MemorySegment.ofArray(key), 0, key.length);
            return result;
        } catch (Throwable e) {
            throw new RuntimeException("Ошибка вызова нативного метода crypto_generichash_init", e);
        }
    }

    public static int crypto_generichash_update(crypto_generichash_state state, byte[] in, long inlen) {
        if (state == null) throw new IllegalArgumentException("Параметр 'state' не может быть null.");
        if (in == null) throw new IllegalArgumentException("Параметр 'in' не может быть null.");
        try (Arena arena = Arena.ofConfined()) {
            MemorySegment stateSegment = state == null ? MemorySegment.NULL : arena.allocate(crypto_generichash_state.getStaticSize());
            if (state != null) state.serialize(stateSegment, 0, arena);
            MemorySegment inSegment = in == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, in);

            int result = (int) CRYPTO_GENERICHASH_UPDATE_HANDLE.invokeExact(stateSegment, inSegment, inlen);
            if (state != null) state.deserialize(stateSegment, 0);
            if (in != null) MemorySegment.copy(inSegment, 0, MemorySegment.ofArray(in), 0, in.length);
            return result;
        } catch (Throwable e) {
            throw new RuntimeException("Ошибка вызова нативного метода crypto_generichash_update", e);
        }
    }

    public static int crypto_generichash_final(crypto_generichash_state state, byte[] out, long outlen) {
        if (state == null) throw new IllegalArgumentException("Параметр 'state' не может быть null.");
        if (out == null) throw new IllegalArgumentException("Параметр 'out' не может быть null.");
        try (Arena arena = Arena.ofConfined()) {
            MemorySegment stateSegment = state == null ? MemorySegment.NULL : arena.allocate(crypto_generichash_state.getStaticSize());
            if (state != null) state.serialize(stateSegment, 0, arena);
            MemorySegment outSegment = out == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, out);

            int result = (int) CRYPTO_GENERICHASH_FINAL_HANDLE.invokeExact(stateSegment, outSegment, outlen);
            if (state != null) state.deserialize(stateSegment, 0);
            if (out != null) MemorySegment.copy(outSegment, 0, MemorySegment.ofArray(out), 0, out.length);
            return result;
        } catch (Throwable e) {
            throw new RuntimeException("Ошибка вызова нативного метода crypto_generichash_final", e);
        }
    }

    public static int crypto_auth_keygen(byte[] key) {
        if (key == null) throw new IllegalArgumentException("Параметр 'key' не может быть null.");
        if (key != null && key.length != crypto_auth_KEYBYTES) throw new IllegalArgumentException("Параметр 'key' должен иметь размер crypto_auth_KEYBYTES.");
        try (Arena arena = Arena.ofConfined()) {
            MemorySegment keySegment = key == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, key);

            int result = (int) CRYPTO_AUTH_KEYGEN_HANDLE.invokeExact(keySegment);
            if (key != null) MemorySegment.copy(keySegment, 0, MemorySegment.ofArray(key), 0, key.length);
            return result;
        } catch (Throwable e) {
            throw new RuntimeException("Ошибка вызова нативного метода crypto_auth_keygen", e);
        }
    }

    public static int crypto_auth(byte[] out, byte[] in, byte[] k) {
        if (out == null) throw new IllegalArgumentException("Параметр 'out' не может быть null.");
        if (out != null && out.length != crypto_auth_BYTES) throw new IllegalArgumentException("Параметр 'out' должен иметь размер crypto_auth_BYTES.");
        if (in == null) throw new IllegalArgumentException("Параметр 'in' не может быть null.");
        if (k == null) throw new IllegalArgumentException("Параметр 'k' не может быть null.");
        if (k != null && k.length != crypto_auth_KEYBYTES) throw new IllegalArgumentException("Параметр 'k' должен иметь размер crypto_auth_KEYBYTES.");
        try (Arena arena = Arena.ofConfined()) {
            MemorySegment outSegment = out == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, out);
            MemorySegment inSegment = in == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, in);
            MemorySegment kSegment = k == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, k);

            int result = (int) CRYPTO_AUTH_HANDLE.invokeExact(outSegment, inSegment, kSegment);
            if (out != null) MemorySegment.copy(outSegment, 0, MemorySegment.ofArray(out), 0, out.length);
            if (in != null) MemorySegment.copy(inSegment, 0, MemorySegment.ofArray(in), 0, in.length);
            if (k != null) MemorySegment.copy(kSegment, 0, MemorySegment.ofArray(k), 0, k.length);
            return result;
        } catch (Throwable e) {
            throw new RuntimeException("Ошибка вызова нативного метода crypto_auth", e);
        }
    }

    public static int crypto_auth_verify(byte[] h, byte[] in, byte[] k) {
        if (h == null) throw new IllegalArgumentException("Параметр 'h' не может быть null.");
        if (h != null && h.length != crypto_auth_BYTES) throw new IllegalArgumentException("Параметр 'h' должен иметь размер crypto_auth_BYTES.");
        if (in == null) throw new IllegalArgumentException("Параметр 'in' не может быть null.");
        if (k == null) throw new IllegalArgumentException("Параметр 'k' не может быть null.");
        if (k != null && k.length != crypto_auth_KEYBYTES) throw new IllegalArgumentException("Параметр 'k' должен иметь размер crypto_auth_KEYBYTES.");
        try (Arena arena = Arena.ofConfined()) {
            MemorySegment hSegment = h == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, h);
            MemorySegment inSegment = in == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, in);
            MemorySegment kSegment = k == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, k);

            int result = (int) CRYPTO_AUTH_VERIFY_HANDLE.invokeExact(hSegment, inSegment, kSegment);
            if (h != null) MemorySegment.copy(hSegment, 0, MemorySegment.ofArray(h), 0, h.length);
            if (in != null) MemorySegment.copy(inSegment, 0, MemorySegment.ofArray(in), 0, in.length);
            if (k != null) MemorySegment.copy(kSegment, 0, MemorySegment.ofArray(k), 0, k.length);
            return result;
        } catch (Throwable e) {
            throw new RuntimeException("Ошибка вызова нативного метода crypto_auth_verify", e);
        }
    }

    public static byte[] sodium_bin2hex(byte[] hex, byte[] bin) {
        if (hex == null) throw new IllegalArgumentException("Параметр 'hex' не может быть null.");
        if (bin == null) throw new IllegalArgumentException("Параметр 'bin' не может быть null.");
        try (Arena arena = Arena.ofConfined()) {
            MemorySegment hexSegment = hex == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, hex);
            MemorySegment binSegment = bin == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, bin);

            byte[] result = (byte[]) SODIUM_BIN2HEX_HANDLE.invokeExact(hexSegment, binSegment);
            if (hex != null) MemorySegment.copy(hexSegment, 0, MemorySegment.ofArray(hex), 0, hex.length);
            if (bin != null) MemorySegment.copy(binSegment, 0, MemorySegment.ofArray(bin), 0, bin.length);
            return result;
        } catch (Throwable e) {
            throw new RuntimeException("Ошибка вызова нативного метода sodium_bin2hex", e);
        }
    }

    public static int sodium_hex2bin(byte[] bin, byte[] hex, long hex_len, String ignore, long[] bin_len, String ignore2) {
        if (bin == null) throw new IllegalArgumentException("Параметр 'bin' не может быть null.");
        if (hex == null) throw new IllegalArgumentException("Параметр 'hex' не может быть null.");
        if (ignore == null) throw new IllegalArgumentException("Параметр 'ignore' не может быть null.");
        if (bin_len == null) throw new IllegalArgumentException("Параметр 'bin_len' не может быть null.");
        if (ignore2 == null) throw new IllegalArgumentException("Параметр 'ignore2' не может быть null.");
        try (Arena arena = Arena.ofConfined()) {
            MemorySegment binSegment = bin == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, bin);
            MemorySegment hexSegment = hex == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_BYTE, hex);
            MemorySegment ignoreSegment;
            if (ignore == null) {
                ignoreSegment = MemorySegment.NULL;
            } else {
                byte[] ignoreBytes = ignore.getBytes(StandardCharsets.UTF_8);
                ignoreSegment = arena.allocate(ignoreBytes.length + 1);
                ignoreSegment.copyFrom(MemorySegment.ofArray(ignoreBytes));
                ignoreSegment.set(ValueLayout.JAVA_BYTE, ignoreBytes.length, (byte)0);
            }
            MemorySegment binlenSegment = bin_len == null ? MemorySegment.NULL : arena.allocateFrom(ValueLayout.JAVA_LONG, bin_len);
            MemorySegment ignore2Segment;
            if (ignore2 == null) {
                ignore2Segment = MemorySegment.NULL;
            } else {
                byte[] ignore2Bytes = ignore2.getBytes(StandardCharsets.UTF_8);
                ignore2Segment = arena.allocate(ignore2Bytes.length + 1);
                ignore2Segment.copyFrom(MemorySegment.ofArray(ignore2Bytes));
                ignore2Segment.set(ValueLayout.JAVA_BYTE, ignore2Bytes.length, (byte)0);
            }

            int result = (int) SODIUM_HEX2BIN_HANDLE.invokeExact(binSegment, hexSegment, hex_len, ignoreSegment, binlenSegment, ignore2Segment);
            if (bin != null) MemorySegment.copy(binSegment, 0, MemorySegment.ofArray(bin), 0, bin.length);
            if (hex != null) MemorySegment.copy(hexSegment, 0, MemorySegment.ofArray(hex), 0, hex.length);
            if (bin_len != null) MemorySegment.copy(binlenSegment, 0, MemorySegment.ofArray(bin_len), 0, bin_len.length);
            return result;
        } catch (Throwable e) {
            throw new RuntimeException("Ошибка вызова нативного метода sodium_hex2bin", e);
        }
    }

}
