package io.aether.crypto.sodium;

import io.aether.crypto.AKey;
import io.aether.crypto.KeyType;
import io.aether.crypto.CryptoEngine;
import io.aether.crypto.SignChecker;
import io.aether.crypto.CryptoProviderFactory;
import io.aether.crypto.CryptoProvider;
import io.aether.utils.HexUtils;

import java.util.Arrays;
import java.util.Objects;

public abstract class SodiumKey implements AKey {
    protected final byte[] data;
    protected final KeyType keyType;
    protected final String providerName = "SODIUM";

    protected SodiumKey(byte[] data, KeyType keyType) {
        this.data = data;
        this.keyType = keyType;
    }

    @Override
    public byte[] getData() {
        return data;
    }

    @Override
    public KeyType getKeyType() {
        return keyType;
    }

    @Override
    public String getProviderName() {
        return providerName;
    }

    @Override
    public CryptoProvider getCryptoProvider() {
        return SodiumCryptoProvider.INSTANCE;
    }

    @Override
    public int size() {
        return data.length;
    }

    @Override
    public String keyToString() {
        return getProviderName() + ":" + getKeyType() + ":" + HexUtils.toHexString(getData());
    }

    @Override
    public String toString() {
        return keyToString();
    }

    @Override
    public boolean equals(Object obj) {
        if (obj instanceof AKey) {
            var k = (AKey) obj;
            return getKeyType() == k.getKeyType() && getProviderName().equals(k.getProviderName()) && Arrays.equals(getData(), k.getData());
        }
        return false;
    }

    @Override
    public int hashCode() {
        return Objects.hash(getKeyType(), getProviderName(), Arrays.hashCode(getData()));
    }

    public static class Symmetric extends SodiumKey implements AKey.Symmetric {
        public Symmetric(byte[] data) {
            super(data, KeyType.SYMMETRIC);
        }

        @Override
        public CryptoEngine toCryptoEngine() {
            return CryptoProviderFactory.getProvider(this).createSymmetricEngine(this);
        }
    }

    public static class AsymmetricPrivate extends SodiumKey implements AKey.AsymmetricPrivate {
        public AsymmetricPrivate(byte[] data) {
            super(data, KeyType.ASYMMETRIC_PRIVATE);
        }
    }

    public static class AsymmetricPublic extends SodiumKey implements AKey.AsymmetricPublic {
        public AsymmetricPublic(byte[] data) {
            super(data, KeyType.ASYMMETRIC_PUBLIC);
        }

        @Override
        public CryptoEngine toCryptoEngine() {
            return CryptoProviderFactory.getProvider(this).createAsymmetricEngine(this);
        }
    }

    public static class SignPrivate extends SodiumKey implements AKey.SignPrivate {
        public SignPrivate(byte[] data) {
            super(data, KeyType.SIGN_PRIVATE);
        }
    }

    public static class SignPublic extends SodiumKey implements AKey.SignPublic {
        public SignPublic(byte[] data) {
            super(data, KeyType.SIGN_PUBLIC);
        }

        @Override
        public SignChecker toSignChecker() {
            return CryptoProviderFactory.getProvider(this).createSigner(this);
        }
    }
}

