package io.aether.crypto.sodium;

import com.goterl.lazysodium.LazySodiumJava;
import com.goterl.lazysodium.SodiumJava;
import com.goterl.lazysodium.exceptions.SodiumException;
import io.aether.crypto.*;
import io.aether.utils.HexUtils;
import io.aether.utils.RU;

public class SodiumCryptoProvider implements CryptoProvider {

    public static final SodiumCryptoProvider INSTANCE = new SodiumCryptoProvider();

    private final LazySodiumJava lazySodium;
    private final SodiumJava sodium;

    @Override
    public AKey.SignPublic createSignPublicKey(byte[] data) {
        return new SodiumKey.SignPublic(data);
    }

    @Override
    public AKey.SignPrivate createSignPrivateKey(byte[] data) {
        return new SodiumKey.SignPrivate(data);
    }

    @Override
    public AKey.Symmetric createSymmetricKey(byte[] bytes) {
        return new SodiumKey.Symmetric(bytes);
    }

    private SodiumCryptoProvider() {
        this.sodium = new SodiumJava();
        this.lazySodium = new LazySodiumJava(this.sodium);
    }

    @Override
    public String getCryptoLibName() {
        return "SODIUM";
    }

    @Override
    public PairAsymKeys createAsymmetricKeys() {
        byte[] privateKey = new byte[KeySize.SODIUM_CURVE_PRIVATE];
        byte[] publicKey = new byte[KeySize.SODIUM_CURVE25519_PUBLIC];
        sodium.crypto_box_keypair(publicKey, privateKey);
        return new PairAsymKeys(new SodiumKey.AsymmetricPublic(publicKey), new SodiumKey.AsymmetricPrivate(privateKey));
    }

    @Override
    public AKey.Symmetric createSymmetricKey() {
        var key = new byte[KeySize.SODIUM_CHACHA20POLY1305];
        sodium.crypto_aead_chacha20poly1305_keygen(key);
        return new SodiumKey.Symmetric(key);
    }

    @Override
    public PairSignKeys createSignKeys() {
        try {
            var keys = lazySodium.cryptoSignKeypair();
            var publicKey = keys.getPublicKey().getAsBytes();
            var privateKey = keys.getSecretKey().getAsBytes();
            return new PairSignKeys(new SodiumKey.SignPublic(publicKey), new SodiumKey.SignPrivate(privateKey));
        } catch (SodiumException e) {
            throw new EncryptException("Failed to generate signing keys", e);
        }
    }

    @Override
    public Signer createSigner(PairSignKeys keys) {
        return createSigner(keys.publicKey, keys.privateKey);
    }

    @Override
    public Signer createSigner(AKey.SignPublic publicKey, AKey.SignPrivate privateKey) {
        if (!(publicKey instanceof SodiumKey.SignPublic) || !(privateKey instanceof SodiumKey.SignPrivate)) {
            throw new IllegalArgumentException("Keys must be instances of SodiumKey.SignPublic and SodiumKey.SignPrivate");
        }
        return new SodiumSigner(publicKey, privateKey, lazySodium);
    }

    @Override
    public Signer createSigner(AKey.SignPublic publicKey) {
        if (!(publicKey instanceof SodiumKey.SignPublic)) {
            throw new IllegalArgumentException("Public key must be an instance of SodiumKey.SignPublic");
        }
        return new SodiumSigner(publicKey, null, lazySodium);
    }

    @Override
    public CryptoEngine createSymmetricEngine(AKey.Symmetric key) {
        if (!(key instanceof SodiumKey.Symmetric)) {
            throw new IllegalArgumentException("Key must be a SodiumKey.Symmetric instance");
        }
        return new SodiumSymmetricEngine(key);
    }

    @Override
    public CryptoEngine createAsymmetricEngine(AKey.AsymmetricPublic key) {
        if (!(key instanceof SodiumKey.AsymmetricPublic)) {
            throw new IllegalArgumentException("Key must be a SodiumKey.AsymmetricPublic instance");
        }
        return new SodiumAsymmetricEngine(key);
    }

    @Override
    public CryptoEngine createAsymmetricEngine(AKey.AsymmetricPrivate privateKey, AKey.AsymmetricPublic publicKey) {
        if (!(privateKey instanceof SodiumKey.AsymmetricPrivate) || !(publicKey instanceof SodiumKey.AsymmetricPublic)) {
            throw new IllegalArgumentException("Keys must be instances of SodiumKey.AsymmetricPrivate and SodiumKey.AsymmetricPublic");
        }
        return new SodiumAsymmetricEngine(privateKey, publicKey);
    }

    @Override
    public CryptoEngine createAsymmetricEngine(PairAsymKeys keys) {
        return createAsymmetricEngine(keys.getPrivateKey(), keys.getPublicKey());
    }

    @Override
    public <T extends AKey> T createKey(KeyType keyType, byte[] data) {
        AKey result;
        switch (keyType) {
            case SYMMETRIC:
                result= new SodiumKey.Symmetric(data);
                break;
            case ASYMMETRIC_PUBLIC:
                result= new SodiumKey.AsymmetricPublic(data);
                break;
            case ASYMMETRIC_PRIVATE:
                result= new SodiumKey.AsymmetricPrivate(data);
                break;
            case SIGN_PUBLIC:
                result= new SodiumKey.SignPublic(data);
                break;
            case SIGN_PRIVATE:
                result= new SodiumKey.SignPrivate(data);
                break;
            default:
                throw new UnsupportedOperationException();
        }
        return RU.cast(result);
    }

    @Override
    public <T extends AKey> T createKey(String data) {
        var parts = data.split(":");
        if (parts.length != 3 || !parts[0].equals(getCryptoLibName())) {
            throw new IllegalArgumentException("Invalid key string for this provider.");
        }
        KeyType keyType = KeyType.valueOf(parts[1]);
        byte[] bytes = HexUtils.hexToBytes(parts[2]);
        return RU.cast(createKey(keyType, bytes));
    }


    @Override
    public Sign createSign(String data) {
        var parts = data.split(":");
        if (parts.length != 2 || !parts[0].equals(getCryptoLibName())) {
            throw new IllegalArgumentException("Invalid sign string for this provider.");
        }
        return new SodiumSign(HexUtils.hexToBytes(parts[1]));
    }

    @Override
    public Sign createSign(byte[] data) {
        return new SodiumSign(data);
    }
}

