package io.aether.crypto.sodium;

import com.goterl.lazysodium.SodiumJava;
import io.aether.crypto.AKey;
import io.aether.crypto.CryptoEngine;
import io.aether.crypto.CryptoProvider;
import io.aether.crypto.DecryptException;
import io.aether.crypto.EncryptException;
import org.jetbrains.annotations.NotNull;

public class SodiumAsymmetricEngine implements CryptoEngine {
    private static final SodiumJava SODIUM = new SodiumJava();
    private final AKey.AsymmetricPublic publicKey;
    private final AKey.AsymmetricPrivate privateKey;

    public SodiumAsymmetricEngine(AKey.AsymmetricPublic publicKey) {
        this.publicKey = publicKey;
        this.privateKey = null;
    }

    public SodiumAsymmetricEngine(AKey.AsymmetricPrivate privateKey, AKey.AsymmetricPublic publicKey) {
        this.privateKey = privateKey;
        this.publicKey = publicKey;
    }

    @Override
    public byte @NotNull [] encrypt(byte @NotNull [] dataBytes) {
        if (dataBytes == null || dataBytes.length == 0) return new byte[0];
        
        int outputSize = KeySize.SODIUM_CURVE_SEALBYTES + dataBytes.length;
        byte[] output = new byte[outputSize];
        var keyData = publicKey.getData();
        
        int result = SODIUM.crypto_box_seal(output, dataBytes, dataBytes.length, keyData);
        if (result != 0) {
            throw new EncryptException("Encrypt exception");
        }
        return output;
    }

    @Override
    public byte @NotNull [] decrypt(byte @NotNull [] dataBytes) {
        if (privateKey == null) {
            throw new UnsupportedOperationException("This engine is not configured for decryption.");
        }
        if (dataBytes == null || dataBytes.length == 0) return new byte[0];
        
        var outputSize = dataBytes.length - KeySize.SODIUM_CURVE_SEALBYTES;
        byte[] output = new byte[outputSize];
        var result = SODIUM.crypto_box_seal_open(output, dataBytes, dataBytes.length, publicKey.getData(), privateKey.getData());
        if (result != 0) {
            throw new DecryptException("Decrypt exception");
        }
        return output;
    }

    @Override
    public String getProviderName() {
        return "SODIUM";
    }

    @Override
    public CryptoProvider getCryptoProvider() {
        return SodiumCryptoProvider.INSTANCE;
    }
}

