package io.aether.crypto.sodium;

import com.goterl.lazysodium.LazySodiumJava;
import io.aether.crypto.AKey;
import io.aether.crypto.Sign;
import io.aether.crypto.Signer;
import io.aether.crypto.CryptoProvider;

public class SodiumSigner implements Signer {
    private final AKey.SignPublic publicKey;
    private final AKey.SignPrivate privateKey;
    private final LazySodiumJava lazySodium;

    public SodiumSigner(AKey.SignPublic publicKey, AKey.SignPrivate privateKey, LazySodiumJava lazySodium) {
        this.publicKey = publicKey;
        this.privateKey = privateKey;
        this.lazySodium = lazySodium;
    }

    @Override
    public Sign sign(byte[] data) {
        if (privateKey == null) {
            throw new IllegalStateException("This Signer instance is not configured with a private key for signing.");
        }
        var res = new byte[KeySize.SODIUM_SIGN_BYTES];
        lazySodium.cryptoSignDetached(res, data, data.length, privateKey.getData());
        return new SodiumSign(res);
    }

    @Override
    public boolean checkSign(byte[] data, Sign sign) {
        if (!(sign instanceof SodiumSign)) {
            throw new IllegalArgumentException("Sign must be a SodiumSign instance");
        }
        if (publicKey == null) {
            throw new IllegalStateException("This Signer instance is not configured with a public key for checking.");
        }
        return lazySodium.cryptoSignVerifyDetached(sign.getSignData(), data, data.length, publicKey.getData());
    }

    @Override
    public AKey.SignPublic getPublicKey() {
        return publicKey;
    }

    @Override
    public String getProviderName() {
        return "SODIUM";
    }

    @Override
    public CryptoProvider getCryptoProvider() {
        return SodiumCryptoProvider.INSTANCE;
    }
}

