package io.aether.crypto.sodium;

import com.goterl.lazysodium.SodiumJava;
import io.aether.crypto.AKey;
import io.aether.crypto.CryptoEngine;
import io.aether.crypto.CryptoProvider;
import io.aether.crypto.DecryptException;
import io.aether.crypto.EncryptException;
import io.aether.utils.DataUtils;
import org.jetbrains.annotations.NotNull;
import java.util.Arrays;

public class SodiumSymmetricEngine implements CryptoEngine {
    private static final SodiumJava SODIUM = new SodiumJava();
    private final AKey.Symmetric key;
    private final Nonce nonce;

    public SodiumSymmetricEngine(AKey.Symmetric key) {
        this.key = key;
        this.nonce = new Nonce();
    }

    @Override
    public byte @NotNull [] encrypt(byte @NotNull [] data) {
        if (data == null || data.length == 0) return new byte[0];
        
        var outSize = data.length + KeySize.SODIUM_CHACHA20POLY1305_ABYTES;
        byte[] output = new byte[outSize + KeySize.SODIUM_CHACHA20POLY1305_NONCE_LEN];
        
        var currentNonce = nonce.incAndGet();
        byte[] nonceBuf = new byte[KeySize.SODIUM_CHACHA20POLY1305_NONCE_LEN];
        DataUtils.writeLongLE(nonceBuf, 0, currentNonce);
        
        var res = SODIUM.crypto_aead_chacha20poly1305_encrypt(output, new long[1], data, data.length, null, 0, null, nonceBuf, key.getData());
        if (res != 0) {
            throw new EncryptException("Encrypt exception");
        }
        
        System.arraycopy(nonceBuf, 0, output, outSize, KeySize.SODIUM_CHACHA20POLY1305_NONCE_LEN);
        return output;
    }

    @Override
    public byte @NotNull [] decrypt(byte @NotNull [] data) {
        if (data == null || data.length == 0) return new byte[0];
        
        var outSize = data.length - KeySize.SODIUM_CHACHA20POLY1305_ABYTES - KeySize.SODIUM_CHACHA20POLY1305_NONCE_LEN;
        byte[] output = new byte[outSize];
        var len = new long[1];
        
        long nonceValue = DataUtils.readLongLE(data, data.length - KeySize.SODIUM_CHACHA20POLY1305_NONCE_LEN);
        byte[] nonceBuf = new byte[KeySize.SODIUM_CHACHA20POLY1305_NONCE_LEN];
        DataUtils.writeLongLE(nonceBuf, 0, nonceValue);
        
        var res = SODIUM.crypto_aead_chacha20poly1305_decrypt(output, len, null, data, data.length - KeySize.SODIUM_CHACHA20POLY1305_NONCE_LEN, null, 0, nonceBuf, key.getData());
        if (res == -1) {
            throw new DecryptException("Decrypt exception");
        }
        
        if (output.length != len[0]) {
            return Arrays.copyOf(output, (int) len[0]);
        }
        return output;
    }

    @Override
    public String getProviderName() {
        return "SODIUM";
    }

    @Override
    public CryptoProvider getCryptoProvider() {
        return SodiumCryptoProvider.INSTANCE;
    }
}

