// File: FastMeta.java

package io.aether.net.fastMeta;

import io.aether.utils.dataio.DataIn;
import io.aether.utils.dataio.DataOut;
import io.aether.utils.flow.Flow;
import it.unimi.dsi.fastutil.longs.LongOpenHashSet;

import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.Set;
import java.util.UUID;

public interface FastMeta {

    // --- Single Primitives ---

    FastMetaType<Boolean> META_BOOLEAN = new FastMetaType<>() {
        @Override
        public void serialize(FastFutureContext ctx, Boolean obj, DataOut out) {
            out.writeBoolean(obj);
        }

        @Override
        public Boolean deserialize(FastFutureContext ctx, DataIn in) {
            return in.readBoolean();
        }
    };

    FastMetaType<Byte> META_BYTE = new FastMetaType<>() {
        @Override
        public void serialize(FastFutureContext ctx, Byte obj, DataOut out) {
            out.writeByte(obj);
        }

        @Override
        public Byte deserialize(FastFutureContext ctx, DataIn in) {
            return in.readByte();
        }
    };

    FastMetaType<Short> META_SHORT = new FastMetaType<>() {
        @Override
        public void serialize(FastFutureContext ctx, Short obj, DataOut out) {
            out.writeShort(obj);
        }

        @Override
        public Short deserialize(FastFutureContext ctx, DataIn in) {
            return in.readShort();
        }
    };

    FastMetaType<Integer> META_INT = new FastMetaType<>() {
        @Override
        public void serialize(FastFutureContext ctx, Integer obj, DataOut out) {
            out.writeInt(obj);
        }

        @Override
        public Integer deserialize(FastFutureContext ctx, DataIn in) {
            return in.readInt();
        }
    };

    FastMetaType<Long> META_LONG = new FastMetaType<>() {
        @Override
        public void serialize(FastFutureContext ctx, Long obj, DataOut out) {
            out.writeLong(obj);
        }

        @Override
        public Long deserialize(FastFutureContext ctx, DataIn in) {
            return in.readLong();
        }
    };

    FastMetaType<Float> META_FLOAT = new FastMetaType<>() {
        @Override
        public void serialize(FastFutureContext ctx, Float obj, DataOut out) {
            out.writeFloat(obj);
        }

        @Override
        public Float deserialize(FastFutureContext ctx, DataIn in) {
            return in.readFloat();
        }
    };

    FastMetaType<Double> META_DOUBLE = new FastMetaType<>() {
        @Override
        public void serialize(FastFutureContext ctx, Double obj, DataOut out) {
            out.writeDouble(obj);
        }

        @Override
        public Double deserialize(FastFutureContext ctx, DataIn in) {
            return in.readDouble();
        }
    };

    // New FastMetaType for Date
    FastMetaType<Date> META_DATE = new FastMetaType<>() {
        @Override
        public void serialize(FastFutureContext ctx, Date obj, DataOut out) {
            out.writeLong(obj.getTime());
        }

        @Override
        public Date deserialize(FastFutureContext ctx, DataIn in) {
            return new Date(in.readLong());
        }
    };


    // --- Packed Integer (for `intpack`) ---

    FastMetaType<Long> META_PACK = new FastMetaType<>() {
        @Override
        public void serialize(FastFutureContext ctx, Long obj, DataOut out) {
            SerializerPackNumber.INSTANCE.put(out, obj);
        }

        @Override
        public Long deserialize(FastFutureContext ctx, DataIn in) {
            return DeserializerPackNumber.INSTANCE.put(in).longValue();
        }
    };

    // --- Common Object Types ---

    FastMetaType<String> META_STRING = new FastMetaType<>() {
        @Override
        public void serialize(FastFutureContext ctx, String obj, DataOut out) {
            var ar = obj.getBytes(StandardCharsets.UTF_8);
            SerializerPackNumber.INSTANCE.put(out, ar.length);
            out.write(ar);
        }

        @Override
        public String deserialize(FastFutureContext ctx, DataIn in) {
            var len = DeserializerPackNumber.INSTANCE.put(in).intValue();
            byte[] ar = new byte[len];
            in.read(ar);
            return new String(ar, StandardCharsets.UTF_8);
        }
    };

    // =================================================================
    // ИЗМЕНЕННЫЙ БЛОК META_UUID (Big Endian, побайтовая запись)
    // =================================================================
    FastMetaType<UUID> META_UUID = new FastMetaType<>() {
        @Override
        public void serialize(FastFutureContext ctx, UUID obj, DataOut out) {
            long msb = obj.getMostSignificantBits();
            long lsb = obj.getLeastSignificantBits();

            // Записываем MostSignificantBits в Big Endian (8 байт)
            out.writeByte((byte) (msb >>> 56));
            out.writeByte((byte) (msb >>> 48));
            out.writeByte((byte) (msb >>> 40));
            out.writeByte((byte) (msb >>> 32));
            out.writeByte((byte) (msb >>> 24));
            out.writeByte((byte) (msb >>> 16));
            out.writeByte((byte) (msb >>> 8));
            out.writeByte((byte) msb);

            // Записываем LeastSignificantBits в Big Endian (8 байт)
            out.writeByte((byte) (lsb >>> 56));
            out.writeByte((byte) (lsb >>> 48));
            out.writeByte((byte) (lsb >>> 40));
            out.writeByte((byte) (lsb >>> 32));
            out.writeByte((byte) (lsb >>> 24));
            out.writeByte((byte) (lsb >>> 16));
            out.writeByte((byte) (lsb >>> 8));
            out.writeByte((byte) lsb);
        }

        @Override
        public UUID deserialize(FastFutureContext ctx, DataIn in) {
            // Читаем MostSignificantBits в Big Endian (8 байт)
            // (in.readByte() & 0xFF) - для предотвращения расширения знака, кроме первого байта
            long msb = ((long) in.readByte() << 56)
                       | ((long) (in.readByte() & 0xFF) << 48)
                       | ((long) (in.readByte() & 0xFF) << 40)
                       | ((long) (in.readByte() & 0xFF) << 32)
                       | ((long) (in.readByte() & 0xFF) << 24)
                       | ((long) (in.readByte() & 0xFF) << 16)
                       | ((long) (in.readByte() & 0xFF) << 8)
                       | ((long) (in.readByte() & 0xFF));

            // Читаем LeastSignificantBits в Big Endian (8 байт)
            long lsb = ((long) in.readByte() << 56)
                       | ((long) (in.readByte() & 0xFF) << 48)
                       | ((long) (in.readByte() & 0xFF) << 40)
                       | ((long) (in.readByte() & 0xFF) << 32)
                       | ((long) (in.readByte() & 0xFF) << 24)
                       | ((long) (in.readByte() & 0xFF) << 16)
                       | ((long) (in.readByte() & 0xFF) << 8)
                       | ((long) (in.readByte() & 0xFF));

            return new UUID(msb, lsb);
        }
    };
    // =================================================================
    // КОНЕЦ ИЗМЕНЕННОГО БЛОКА
    // =================================================================

    FastMetaType<URI> META_URI = new FastMetaType<>() {
        @Override
        public void serialize(FastFutureContext ctx, URI obj, DataOut out) {
            var ar = obj.toString().getBytes(StandardCharsets.UTF_8);
            SerializerPackNumber.INSTANCE.put(out, ar.length);
            out.write(ar);
        }

        @Override
        public URI deserialize(FastFutureContext ctx, DataIn in) {
            var len = DeserializerPackNumber.INSTANCE.put(in).intValue();
            byte[] ar = new byte[len];
            in.read(ar);
            String uriString = new String(ar, StandardCharsets.UTF_8);
            try {
                return new URI(uriString);
            } catch (URISyntaxException e) {
                throw new RuntimeException("Failed to deserialize URI: " + uriString, e);
            }
        }
    };

    // --- Primitive Arrays ---

    FastMetaType<byte[]> META_ARRAY_BYTE = new FastMetaType<>() {
        @Override
        public void serialize(FastFutureContext ctx, byte[] ar, DataOut out) {
            SerializerPackNumber.INSTANCE.put(out, ar.length);
            out.write(ar);
        }

        @Override
        public byte[] deserialize(FastFutureContext ctx, DataIn in) {
            var len = DeserializerPackNumber.INSTANCE.put(in).intValue();
            byte[] ar = new byte[len];
            in.read(ar);
            return ar;
        }
    };

    FastMetaType<boolean[]> META_ARRAY_BOOLEAN = new FastMetaType<>() {
        @Override
        public void serialize(FastFutureContext ctx, boolean[] ar, DataOut out) {
            SerializerPackNumber.INSTANCE.put(out, ar.length);
            for (boolean v : ar) out.writeBoolean(v);
        }

        @Override
        public boolean[] deserialize(FastFutureContext ctx, DataIn in) {
            var len = DeserializerPackNumber.INSTANCE.put(in).intValue();
            boolean[] ar = new boolean[len];
            for (int i = 0; i < len; i++) ar[i] = in.readBoolean();
            return ar;
        }
    };
    FastMetaType<short[]> META_ARRAY_SHORT = new FastMetaType<>() {
        @Override
        public void serialize(FastFutureContext ctx, short[] ar, DataOut out) {
            SerializerPackNumber.INSTANCE.put(out, ar.length);
            for (short v : ar) out.writeShort(v);
        }

        @Override
        public short[] deserialize(FastFutureContext ctx, DataIn in) {
            var len = DeserializerPackNumber.INSTANCE.put(in).intValue();
            short[] ar = new short[len];
            for (int i = 0; i < len; i++) ar[i] = in.readShort();
            return ar;
        }
    };
    FastMetaType<int[]> META_ARRAY_INT = new FastMetaType<>() {
        @Override
        public void serialize(FastFutureContext ctx, int[] ar, DataOut out) {
            SerializerPackNumber.INSTANCE.put(out, ar.length);
            for (int v : ar) out.writeInt(v);
        }

        @Override
        public int[] deserialize(FastFutureContext ctx, DataIn in) {
            var len = DeserializerPackNumber.INSTANCE.put(in).intValue();
            int[] ar = new int[len];
            for (int i = 0; i < len; i++) ar[i] = in.readInt();
            return ar;
        }
    };
    FastMetaType<long[]> META_ARRAY_LONG = new FastMetaType<>() {
        @Override
        public void serialize(FastFutureContext ctx, long[] ar, DataOut out) {
            SerializerPackNumber.INSTANCE.put(out, ar.length);
            for (long v : ar) out.writeLong(v);
        }

        @Override
        public long[] deserialize(FastFutureContext ctx, DataIn in) {
            var len = DeserializerPackNumber.INSTANCE.put(in).intValue();
            long[] ar = new long[len];
            for (int i = 0; i < len; i++) ar[i] = in.readLong();
            return ar;
        }
    };
    FastMetaType<Set<Long>> META_SET_LONG = new FastMetaType<>() {
        @Override
        public void serialize(FastFutureContext ctx, Set<Long> ar, DataOut out) {
            META_ARRAY_LONG.serialize(ctx, Flow.flow(ar).mapToLong(Long::longValue).toArray(), out);
        }

        @Override
        public Set<Long> deserialize(FastFutureContext ctx, DataIn in) {
            var ar = META_ARRAY_LONG.deserialize(ctx, in);
            return new LongOpenHashSet(ar);
        }
    };
    FastMetaType<UUID[]> META_ARRAY_UUID = new FastMetaType<>() {
        @Override
        public void serialize(FastFutureContext ctx, UUID[] ar, DataOut out) {
            SerializerPackNumber.INSTANCE.put(out, ar.length);
            for (var v : ar) {
                META_UUID.serialize(ctx, v, out);
            }
        }

        @Override
        public UUID[] deserialize(FastFutureContext ctx, DataIn in) {
            var len = DeserializerPackNumber.INSTANCE.put(in).intValue();
            UUID[] ar = new UUID[len];
            for (int i = 0; i < len; i++) {
                ar[i] = META_UUID.deserialize(ctx, in);
            }
            return ar;
        }
    };
    FastMetaType<URI[]> META_ARRAY_URI = new FastMetaType<>() {
        @Override
        public void serialize(FastFutureContext ctx, URI[] ar, DataOut out) {
            SerializerPackNumber.INSTANCE.put(out, ar.length);
            for (var v : ar) {
                META_URI.serialize(ctx, v, out);
            }
        }

        @Override
        public URI[] deserialize(FastFutureContext ctx, DataIn in) {
            var len = DeserializerPackNumber.INSTANCE.put(in).intValue();
            URI[] ar = new URI[len];
            for (int i = 0; i < len; i++) {
                ar[i] = META_URI.deserialize(ctx, in);
            }
            return ar;
        }
    };

    // --- Internal Types ---

    FastMetaType<Integer> META_REQUEST_ID = new FastMetaType<>() {
        @Override
        public void serialize(FastFutureContext ctx, Integer ar, DataOut out) {
            out.writeInt(ar);
        }

        @Override
        public Integer deserialize(FastFutureContext ctx, DataIn in) {
            return in.readInt();
        }
    };

    FastMetaType<Integer> META_COMMAND = new FastMetaType<>() {
        @Override
        public void serialize(FastFutureContext ctx, Integer ar, DataOut out) {
            out.writeByte(ar);
        }

        @Override
        public Integer deserialize(FastFutureContext ctx, DataIn in) {
            return in.readUByte();
        }
    };

}