package io.aether.net.fastMeta.netty; // (или ваш пакет)

import io.aether.utils.dataio.*;
import io.netty.buffer.ByteBuf;

/**
 * Реализация DataIO, оборачивающая Netty ByteBuf.
 * Избегает копирования данных в/из byte[].
 *
 * ВАЖНО: Эта реализация использует Little-Endian (LE) методы Netty
 * (readIntLE, writeIntLE и т.д.), чтобы соответствовать
 * порядку байтов, определенному в DataIn.java и DataOut.java.
 */
public class ByteBufDataIO implements DataIO {

    private final ByteBuf buf;

    public ByteBufDataIO(ByteBuf buf) {
        this.buf = buf;
    }

    // --- DataIn Implementation ---

    @Override
    public int getSizeForRead() {
        return buf.readableBytes();
    }

    @Override
    public int read(byte[] b, int offset, int len) {
        int readable = buf.readableBytes();
        if (readable == 0) return (len == 0 ? 0 : -1);
        int readLen = Math.min(len, readable);
        buf.readBytes(b, offset, readLen);
        return readLen;
    }

    @Override
    public int read(int[] b, int offset, int len) {
        int endIndex = Math.min(offset + len, b.length);
        for (int i = offset; i < endIndex; i++) {
            if (buf.readableBytes() < 4) {
                return i - offset; // Вернуть кол-во прочитанных int
            }
            b[i] = readInt();
        }
        return endIndex - offset;
    }

    @Override
    public void skipBytes(int n) {
        buf.skipBytes(n);
    }

    @Override
    public boolean readBoolean() {
        return buf.readByte() != 0;
    }

    @Override
    public byte readByte() {
        return buf.readByte();
    }

    @Override
    public int readUByte() {
        // & 0xFF для преобразования знакового short (из Netty) в беззнаковый int
        return buf.readUnsignedByte() & 0xFF;
    }

    @Override
    public DataIO readSubData(int length) {
        // readSlice создает "вид" буфера без копирования
        return new ByteBufDataIO(buf.readSlice(length));
    }

    @Override
    public short readShort() {
        return buf.readShortLE();
    }

    @Override
    public int readUShort() {
        // & 0xFFFF для преобразования знакового int (из Netty) в беззнаковый int
        return buf.readUnsignedShortLE() & 0xFFFF;
    }

    @Override
    public char readChar() {
        // Соответствует DataIn.readChar()
        return (char) readUByte();
    }

    @Override
    public int readInt() {
        return buf.readIntLE();
    }

    @Override
    public long readUInt() {
        // & 0xFFFFFFFFL для преобразования знакового long в беззнаковый long
        return buf.readUnsignedIntLE() & 0xFFFFFFFFL;
    }

    @Override
    public long readLong() {
        return buf.readLongLE();
    }

    @Override
    public float readFloat() {
        return buf.readFloatLE();
    }

    @Override
    public double readDouble() {
        return buf.readDoubleLE();
    }

    @Override
    public String readString1() {
        int len = readUByte();
        byte[] data = new byte[len];
        read(data);
        return new String(data); // (Charset по умолчанию?)
    }

    @Override
    public int indexOf(int limit, byte val) {
        int searchLength = Math.min(limit, buf.readableBytes());
        // buf.indexOf ищет от readerIndex
        return buf.indexOf(buf.readerIndex(), buf.readerIndex() + searchLength, val);
    }

    // --- DataOut Implementation ---

    @Override
    public int write(byte[] b, int off, int len) {
        int writeLen = Math.min(len, buf.writableBytes());
        buf.writeBytes(b, off, writeLen);
        return writeLen;
    }

    @Override
    public int write(int[] b, int off, int len) {
        int endIndex = Math.min(off + len, b.length);
        for (int i = off; i < endIndex; i++) {
            if (buf.writableBytes() < 4) {
                return i - off; // Вернуть кол-во записанных int
            }
            writeInt(b[i]);
        }
        return endIndex - off;
    }

    @Override
    public void writeBoolean(boolean v) {
        buf.writeByte(v ? 1 : 0);
    }

    @Override
    public void writeByte(int v) {
        buf.writeByte(v);
    }

    @Override
    public void writeShort(short v) {
        buf.writeShortLE(v);
    }

    @Override
    public void writeShort(int v) {
        buf.writeShortLE(v);
    }

    @Override
    public void writeChar(char v) {
        // Соответствует DataOut.writeChar()
        writeByte((byte) v);
    }

    @Override
    public void writeInt(int v) {
        buf.writeIntLE(v);
    }

    @Override
    public void writeLong(long v) {
        buf.writeLongLE(v);
    }

    @Override
    public void writeFloat(float v) {
        buf.writeFloatLE(v);
    }

    @Override
    public void writeDouble(double v) {
        buf.writeDoubleLE(v);
    }

    @Override
    public boolean isWritable() {
        return buf.isWritable();
    }

    @Override
    public int getSizeForWrite() {
        return buf.writableBytes();
    }

    @Override
    public void write(DataInOut data) {
        //
        write(data.data, data.readPos, data.getSizeForRead());
        data.readPos = data.writePos; // Очищаем DataInOut, как в DataOut.write(DataInOut)
    }

    @Override
    public void write(DataInOutStatic data) {
        //
        write(data.data, data.readPos, data.getSizeForRead());
        data.readPos = data.writePos;
    }

    @Override
    public void write(DataIn data) {
        // Медленный путь, если это не ByteBufDataIO
        if (data instanceof ByteBufDataIO) {
            buf.writeBytes(((ByteBufDataIO) data).buf);
        } else {
            // Копирование, как определено в DataOut.java
            DataIO.super.write(data);
        }
    }
}