// [file name]: NettyFastMetaClient.java
package io.aether.net.fastMeta.netty;

import io.aether.logger.LNode;
import io.aether.logger.Log;
import io.aether.net.fastMeta.*;
import io.aether.utils.dataio.DataInOut;
import io.aether.utils.futures.AFuture;
import io.aether.utils.interfaces.AFunction;
import io.netty.bootstrap.Bootstrap;
import io.netty.buffer.ByteBuf;
import io.netty.channel.*;
import io.netty.channel.socket.DatagramChannel;
import io.netty.channel.socket.DatagramPacket;
import io.netty.channel.socket.SocketChannel;

import java.net.URI;
import java.util.List; // <--- ДОБАВЛЕН ИМПОРТ
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer; // <--- ДОБАВЛЕН ИМПОРТ

import io.netty.handler.codec.MessageToMessageDecoder; // <--- ДОБАВЛЕН ИМПОРТ
import io.netty.handler.codec.MessageToMessageEncoder; // <--- ДОБАВЛЕН ИМПОРТ
import io.netty.handler.codec.http.HttpClientCodec;
import io.netty.handler.codec.http.HttpObjectAggregator;
import io.netty.handler.codec.http.FullHttpResponse; // <--- ДОБАВЛЕН ИМПОРТ
import io.netty.handler.codec.http.websocketx.BinaryWebSocketFrame; // <--- ДОБАВЛЕН ИМПОРТ
import io.netty.handler.codec.http.websocketx.WebSocketClientHandshaker;
import io.netty.handler.codec.http.websocketx.WebSocketClientHandshakerFactory;
import io.netty.handler.codec.http.websocketx.WebSocketVersion;
import io.netty.handler.codec.http.websocketx.WebSocketFrame; // <--- ДОБАВЛЕН ИМПОРТ
import io.netty.handler.codec.http.DefaultHttpHeaders;

@ChannelHandler.Sharable
class NettyFastMetaClient<LT, RT extends RemoteApi> extends ChannelInboundHandlerAdapter implements FastMetaClient<LT, RT> {
    private final boolean isWebSocket;

    private static final int RECONNECT_DELAY_SECONDS = 1;

    private final NettyFastMetaNet nettyNet;
    private final URI uri;
    private final boolean isUdp;
    private final FastMetaApi<LT, ?> localApiMeta;
    private final FastMetaNet.WritableConsumer writableConsumer;
    private final LNode logContext;
    private final Bootstrap bootstrap;

    private final FastApiContext context;
    private final RT remoteApi;
    private final LT localApi;

    private volatile ChannelHandlerContext ctx; // The active Netty context
    private volatile boolean active = true;     // Controls the reconnect loop

    NettyFastMetaClient(NettyFastMetaNet nettyNet, URI uri,
                        FastMetaApi<LT, ?> lt, FastMetaApi<?, RT> rt,
                        AFunction<RT, LT> localApiProvider,
                        FastMetaNet.WritableConsumer writableConsumer) {
        this.nettyNet = nettyNet;
        this.uri = uri;
        this.isUdp = "udp".equals(uri.getScheme());
        this.localApiMeta = lt;
        this.writableConsumer = writableConsumer;
        this.logContext = Log.of("uri", uri, "socket", "client netty");
        this.isWebSocket = "ws".equals(uri.getScheme()) || "wss".equals(uri.getScheme());

        // 1. Create the context
        this.context = new FastApiContext() {
            @Override
            public void flush(AFuture sendFuture) {
                handleFlush(sendFuture);
            }
        };

        // 2. Create the remote proxy
        this.remoteApi = rt.makeRemote(this.context);

        // 3. Create the local API
        this.localApi = localApiProvider.apply(this.remoteApi);

        // 4. Configure the Netty Bootstrap
        this.bootstrap = new Bootstrap();
        bootstrap.group(nettyNet.getWorkerGroup())
                .option(ChannelOption.SO_KEEPALIVE, true);

        if (isUdp) {
            // --- UDP Pipeline ---
            Class<? extends DatagramChannel> datagramChannelClass = getDatagramChannelClass();
            bootstrap.channel(datagramChannelClass)
                    .handler(new ChannelInitializer<DatagramChannel>() {
                        @Override
                        protected void initChannel(DatagramChannel ch) {
                            ch.pipeline().addLast("handler", NettyFastMetaClient.this);
                        }
                    });
        } else {
            // --- TCP / WS Pipeline ---
            bootstrap.channel(nettyNet.getClientChannelClass())
                    .handler(new ChannelInitializer<SocketChannel>() {
                        @Override
                        protected void initChannel(SocketChannel ch) {

                            if (isWebSocket) {
                                // --- WS/WSS PIPELINE (ИСПРАВЛЕННЫЙ ПОРЯДОК) ---
                                try (var _l = logContext.context()) {
                                    Log.info("Initializing WebSocket client pipeline");

                                    WebSocketClientHandshaker handshaker = WebSocketClientHandshakerFactory.newHandshaker(
                                            uri, WebSocketVersion.V13, null, true, new DefaultHttpHeaders());

                                    // Этот хэндлер управляет рукопожатием
                                    final NettyWebSocketClientHandshakerHandler handshakerHandler =
                                            new NettyWebSocketClientHandshakerHandler(handshaker, logContext,
                                                    (ctx) -> NettyFastMetaClient.this.channelActive(ctx), // onHandshakeComplete
                                                    () -> { if (NettyFastMetaClient.this.ctx != null) NettyFastMetaClient.this.channelInactive(NettyFastMetaClient.this.ctx); } // onChannelInactive
                                            );

                                    ch.pipeline().addLast(
                                            new HttpClientCodec(),
                                            new HttpObjectAggregator(8192),
                                            // WebSocketClientCompressionHandler.INSTANCE, // Компрессию пока отключаем
                                            handshakerHandler
                                    );

                                    // --- Inbound (Порядок: Сверху Вниз) ---
                                    // 1. Адаптер: BinaryWebSocketFrame -> ByteBuf
                                    ch.pipeline().addLast("ws_to_buf_decoder", new MessageToMessageDecoder<BinaryWebSocketFrame>() {
                                        @Override
                                        protected void decode(ChannelHandlerContext ctx, BinaryWebSocketFrame msg, List<Object> out) {
                                            out.add(msg.content().retain());
                                        }
                                    });
                                    // 2. Декодер заголовка: ByteBuf -> ByteBuf (payload)
                                    ch.pipeline().addLast("frameDecoder", new FastMetaFrameDecoder());

                                    // --- Outbound (Порядок: Снизу Вверх - ОБРАТНЫЙ) ---
                                    // 2. Адаптер: ByteBuf -> BinaryWebSocketFrame (Добавляем ПЕРВЫМ)
                                    ch.pipeline().addLast("buf_to_ws_encoder", new MessageToMessageEncoder<ByteBuf>() {
                                        @Override
                                        protected void encode(ChannelHandlerContext ctx, ByteBuf msg, List<Object> out) {
                                            out.add(new BinaryWebSocketFrame(msg.retain()));
                                        }
                                    });
                                    // 1. Энкодер заголовка: byte[] -> ByteBuf (Добавляем ВТОРЫМ)
                                    ch.pipeline().addLast("frameEncoder", new FastMetaFrameEncoder());

                                    // 3. (In) / 3. (Out) Главный обработчик (Добавляем ПОСЛЕДНИМ)
                                    ch.pipeline().addLast("handler", NettyFastMetaClient.this);
                                }
                            } else {
                                // --- PLAIN TCP PIPELINE ---
                                ch.pipeline().addLast("frameDecoder", new FastMetaFrameDecoder());
                                ch.pipeline().addLast("frameEncoder", new FastMetaFrameEncoder());
                                ch.pipeline().addLast("handler", NettyFastMetaClient.this);
                            }
                        }
                    });
        }

        // 5. Start the connection process
        tryConnect();
    }

    /**
     * Gets the appropriate DatagramChannel class based on the selected transport.
     */
    private Class<? extends DatagramChannel> getDatagramChannelClass() {
        Class<? extends io.netty.channel.Channel> clientChannelClass = nettyNet.getClientChannelClass();

        if (clientChannelClass == io.netty.channel.epoll.EpollSocketChannel.class) {
            return io.netty.channel.epoll.EpollDatagramChannel.class;
        } else if (clientChannelClass == io.netty.channel.kqueue.KQueueSocketChannel.class) {
            return io.netty.channel.kqueue.KQueueDatagramChannel.class;
        } else {
            return io.netty.channel.socket.nio.NioDatagramChannel.class;
        }
    }

    private void tryConnect() {
        if (!active) return;

        try (var _l = logContext.context()) {
            Log.info("Attempting to connect...");
            bootstrap.connect(uri.getHost(), uri.getPort()).addListener((ChannelFuture future) -> {
                try (var _l_listener =logContext.context()) {
                    if (future.isSuccess()) {
                        // channelActive() будет вызван Netty (TCP) или HandshakerHandler (WS)
                        Log.info("Connection established", "local", future.channel().localAddress());
                    } else {
                        scheduleReconnect();
                    }
                }
            });
        }
    }

    private void scheduleReconnect() {
        if (!active) return;

        writableConsumer.apply(false); // Notify user we are not writable

        try (var _l = logContext.context()) {
            Log.info("Scheduling reconnect...", "delaySeconds", RECONNECT_DELAY_SECONDS);
        }

        nettyNet.getWorkerGroup().schedule(this::tryConnect, RECONNECT_DELAY_SECONDS, TimeUnit.SECONDS);
    }

    // --- ChannelInboundHandlerAdapter overrides (Netty Pipeline) ---

    @Override
    public void channelActive(ChannelHandlerContext ctx) {
        try (var _l = logContext.context()) {
            // ВАЖНО: для WS это вызывается *после* рукопожатия
            this.ctx = ctx;
            writableConsumer.apply(true);
            Log.info("Netty client connected");
        }
    }

    @Override
    public void channelInactive(ChannelHandlerContext ctx) {
        try (var _l = logContext.context()) {
            this.ctx = null;
            Log.warn("Netty client connection lost");
            scheduleReconnect(); // Try to reconnect
        }
    }

    @Override
    public void channelRead(ChannelHandlerContext ctx, Object msg) {
        try (var _l = logContext.context()) {
            io.netty.buffer.ByteBuf payload = null;

            if (msg instanceof io.netty.buffer.ByteBuf) {
                // TCP И WS путь (уже прошел через FastMetaFrameDecoder)
                payload = (io.netty.buffer.ByteBuf) msg;
            } else if (msg instanceof DatagramPacket) {
                // UDP path
                payload = ((DatagramPacket) msg).content();
            } else {
                Log.warn("Received unexpected message type, ignoring.", "type", msg.getClass().getName());
                io.netty.util.ReferenceCountUtil.release(msg);
                return;
            }

            ByteBufDataIO in = null;
            try {
                in = new ByteBufDataIO(payload);
                localApiMeta.makeLocal(context, in, localApi);
            } catch (Exception e) {
                Log.error("Error during makeLocal (packet processing).", e);
                ctx.close();
            } finally {
                if (payload != null) {
                    payload.release(); // Release the ByteBuf
                }
            }
        }
    }

    @Override
    public void channelWritabilityChanged(ChannelHandlerContext ctx) {
        writableConsumer.apply(ctx.channel().isWritable());
    }

    @Override
    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) {
        try (var _l = logContext.context()) {
            Log.warn("Netty client exception caught", cause);
            ctx.close(); // This will trigger channelInactive
        }
    }

    // --- FastMetaClient / FastMetaNet.Connection Implementation ---

    @Override
    public void flush(AFuture sendFuture) {
        context.flush(sendFuture);
    }

    private void handleFlush(AFuture sendFuture) {
        try (var _l = logContext.context()) {
            ChannelHandlerContext currentCtx = this.ctx;
            if (!isWritable() || currentCtx == null) {
                sendFuture.tryCancel();
                return;
            }

            DataInOut combinedData = new DataInOut();
            context.remoteDataToArray(combinedData);

            if (combinedData.getSizeForRead() == 0) {
                sendFuture.tryDone(); // Nothing to flush
                return;
            }

            byte[] payload = combinedData.toArray();

            Object dataToSend;
            if (isUdp) {
                ByteBuf buf = currentCtx.alloc().buffer(payload.length);
                buf.writeBytes(payload);
                dataToSend = new DatagramPacket(buf, new java.net.InetSocketAddress(uri.getHost(), uri.getPort()));
            } else {
                dataToSend = payload;
            }

            currentCtx.writeAndFlush(dataToSend).addListener(future -> {
                try (var _l_listener = logContext.context()) {
                    if (future.isSuccess()) {
                        sendFuture.tryDone();
                    } else {
                        sendFuture.tryError(future.cause());
                    }
                }
            });
        }
    }

    @Override
    public void read() {
    }

    @Override
    public void stopRead() {
    }

    @Override
    public AFuture write(byte[] data) {
        try (var _l = logContext.context()) {
            ChannelHandlerContext currentCtx = this.ctx;
            if (isWritable() && currentCtx != null) {
                AFuture res = AFuture.make();

                Object dataToSend;
                if (isUdp) {
                    ByteBuf buf = currentCtx.alloc().buffer(data.length);
                    buf.writeBytes(data);
                    dataToSend = new DatagramPacket(buf, new java.net.InetSocketAddress(uri.getHost(), uri.getPort()));
                } else {
                    dataToSend = data;
                }

                currentCtx.writeAndFlush(dataToSend).addListener(future -> {
                    try (var _l_listener = logContext.context()) {
                        if (future.isSuccess()) {
                            res.tryDone();
                        } else {
                            Log.warn("Netty client raw write failed", future.cause());
                            res.tryError(future.cause());
                        }
                    }
                });
                return res;
            }
            return AFuture.canceled();
        }
    }

    @Override
    public LT getLocalApi() {
        return localApi;
    }

    @Override
    public RT getRemoteApi() {
        return remoteApi;
    }

    @Override
    public boolean isWritable() {
        ChannelHandlerContext currentCtx = this.ctx;
        return active && currentCtx != null && currentCtx.channel().isWritable();
    }

    @Override
    public FastFutureContext getMetaContext() {
        return context;
    }

    @Override
    public AFuture destroy(boolean force) {
        try (var _l = logContext.context()) {
            Log.info("Destroying Netty client");
        }
        active = false; // Stop all reconnect attempts

        AFuture res = AFuture.make();
        ChannelHandlerContext currentCtx = this.ctx;
        if (currentCtx != null) {
            currentCtx.close().addListener(future -> {
                if (future.isSuccess()) {
                    res.tryDone();
                } else {
                    res.tryError(future.cause());
                }
            });
        } else {
            res.tryDone();
        }
        res.to(() -> {
            nettyNet.releaseSharedResources();
            Log.debug("Netty client destroy complete, released resources.");
        });
        return res;
    }

    private static class NettyWebSocketClientHandshakerHandler extends ChannelInboundHandlerAdapter {

        private final WebSocketClientHandshaker handshaker;
        private final LNode logContext;
        private final Consumer<ChannelHandlerContext> onHandshakeComplete;
        private final Runnable onChannelInactive;

        public NettyWebSocketClientHandshakerHandler(WebSocketClientHandshaker handshaker,
                                                     LNode logContext,
                                                     Consumer<ChannelHandlerContext> onHandshakeComplete,
                                                     Runnable onChannelInactive) {
            this.handshaker = handshaker;
            this.logContext = logContext;
            this.onHandshakeComplete = onHandshakeComplete;
            this.onChannelInactive = onChannelInactive;
        }

        @Override
        public void channelActive(ChannelHandlerContext ctx) {
            // Канал готов, начинаем рукопожатие
            handshaker.handshake(ctx.channel());
        }

        @Override
        public void channelInactive(ChannelHandlerContext ctx) {
            try (var _l = logContext.context()) {
                Log.info("WebSocket connection closed");
                onChannelInactive.run();
            }
        }

        @Override
        public void channelRead(ChannelHandlerContext ctx, Object msg) {
            try (var _l = logContext.context()) {
                if (!handshaker.isHandshakeComplete()) {
                    // Завершаем рукопожатие
                    try {
                        handshaker.finishHandshake(ctx.channel(), (FullHttpResponse) msg);
                        Log.info("WebSocket handshake completed");
                        // Сообщаем NettyFastMetaClient, что соединение активно
                        onHandshakeComplete.accept(ctx);
                    } catch (Exception e) {
                        Log.error("WebSocket handshake failed", e);
                        ctx.close();
                    } finally {
                        ((FullHttpResponse) msg).release();
                    }
                    return;
                }

                if (msg instanceof FullHttpResponse) {
                    FullHttpResponse response = (FullHttpResponse) msg;
                    Log.error("Unexpected FullHttpResponse after handshake", "status", response.status());
                    response.release();
                    ctx.close();
                    return;
                }

                // Передаем все фреймы (Binary, Close, etc.) дальше по пайплайну
                if (msg instanceof WebSocketFrame) {
                    ctx.fireChannelRead(msg);
                } else {
                    Log.warn("Received non-WebSocketFrame message after handshake", "type", msg.getClass().getName());
                    io.netty.util.ReferenceCountUtil.release(msg);
                }
            }
        }

        @Override
        public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) {
            try (var _l = logContext.context()) {
                Log.error("WebSocket client error", cause);
            }
            if (!handshaker.isHandshakeComplete()) {
                Log.error("WebSocket handshake failed during exception", cause);
            }
            ctx.close();
        }
    }
}