package io.aether.net.fastMeta.netty;

import io.aether.logger.Log;
import io.aether.net.fastMeta.*;
import io.aether.utils.interfaces.AFunction;
import io.netty.channel.EventLoopGroup;
import io.netty.channel.ServerChannel;
import io.netty.channel.epoll.Epoll;
import io.netty.channel.epoll.EpollEventLoopGroup;
import io.netty.channel.epoll.EpollServerSocketChannel;
import io.netty.channel.epoll.EpollSocketChannel;
import io.netty.channel.kqueue.KQueue;
import io.netty.channel.kqueue.KQueueEventLoopGroup;
import io.netty.channel.kqueue.KQueueServerSocketChannel;
import io.netty.channel.kqueue.KQueueSocketChannel;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.nio.NioServerSocketChannel;
import io.netty.channel.socket.nio.NioSocketChannel;
import io.netty.util.concurrent.Future;

import java.net.URI;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;

/**
 * Netty-based implementation of the FastMetaNet factory with
 * auto-selection of the optimal transport (Epoll/KQueue/NIO)
 * AND reference counting for shared EventLoopGroups.
 * <p>
 * Uses non-blocking AtomicReference operations to manage group lifecycles.
 */
public class NettyFastMetaNet implements FastMetaNet, AutoCloseable {

    private static final String TRANSPORT_NAME;
    private static final Class<? extends ServerChannel> SERVER_CHANNEL_CLASS;
    private static final Class<? extends io.netty.channel.Channel> CLIENT_CHANNEL_CLASS;

    /**
     * Manages the shared resource lifecycle using reference counting.
     */
    private final AtomicInteger refCounter = new AtomicInteger(0);

    /**
     * Shared group for I/O operations (clients + server child sockets).
     * Managed via AtomicReference for non-blocking, thread-safe lazy initialization.
     */
    private final AtomicReference<EventLoopGroup> workerGroupRef = new AtomicReference<>();

    /**
     * Group for accepting server connections (server-only).
     * Lazily initialized only when requested by a server.
     */
    private final AtomicReference<EventLoopGroup> bossGroupRef = new AtomicReference<>();


    /**
     * Initializes the NettyFastMetaNet.
     * EventLoopGroups are NOT created here; they are created lazily on first use.
     */
    public NettyFastMetaNet() {
        /** Constructor is empty. Groups are created lazily. */
    }

    /**
     * Creates a new worker EventLoopGroup instance.
     *
     * @return A new EventLoopGroup configured for worker I/O.
     */
    private EventLoopGroup createWorkerGroup() {
        Log.info("Initializing Netty Worker EventLoopGroup (first use)...", "transport", TRANSPORT_NAME);
        ThreadFactory factory = new ThreadFactory() {
            private final AtomicInteger threadNumber = new AtomicInteger(1);
            @Override
            public Thread newThread(Runnable r) {
                Thread t = new Thread(r, "fastmeta-netty-worker-" + threadNumber.getAndIncrement());
                t.setDaemon(true);
                return t;
            }
        };

        if (Epoll.isAvailable()) return new EpollEventLoopGroup(0, factory);
        if (KQueue.isAvailable()) return new KQueueEventLoopGroup(0, factory);
        return new NioEventLoopGroup(0, factory);
    }

    /**
     * Creates a new boss EventLoopGroup instance.
     *
     * @return A new EventLoopGroup configured for server boss (acceptor) threads.
     */
    private EventLoopGroup createBossGroup() {
        Log.info("Initializing Netty Boss EventLoopGroup (first server use)...", "transport", TRANSPORT_NAME);
        ThreadFactory factory = new ThreadFactory() {
            private final AtomicInteger threadNumber = new AtomicInteger(1);
            @Override
            public Thread newThread(Runnable r) {
                Thread t = new Thread(r, "fastmeta-netty-boss-" + threadNumber.getAndIncrement());
                t.setDaemon(true);
                return t;
            }
        };

        if (Epoll.isAvailable()) return new EpollEventLoopGroup(1, factory);
        if (KQueue.isAvailable()) return new KQueueEventLoopGroup(1, factory);
        return new NioEventLoopGroup(1, factory);
    }


    /**
     * Thread-safely ensures the worker EventLoopGroup is initialized and returns it.
     * Uses a CAS (compare-and-set) loop to handle concurrent initialization races.
     *
     * @return The initialized worker EventLoopGroup.
     */
    private EventLoopGroup ensureWorkerInitialized() {
        while (true) {
            EventLoopGroup currentGroup = workerGroupRef.get();
            if (currentGroup != null) {
                return currentGroup;
            }

            EventLoopGroup newGroup = createWorkerGroup();

            if (workerGroupRef.compareAndSet(null, newGroup)) {
                Log.info("Netty WorkerGroup started using " + TRANSPORT_NAME + " transport");
                return newGroup;
            } else {
                Log.debug("Lost CAS race, shutting down redundant WorkerGroup");
                shutdownGroup(newGroup);
            }
        }
    }

    /**
     * Thread-safely ensures the boss EventLoopGroup is initialized and returns it.
     * This is called lazily, only by servers.
     *
     * @return The initialized boss EventLoopGroup.
     */
    private EventLoopGroup ensureBossInitialized() {
        while (true) {
            EventLoopGroup currentGroup = bossGroupRef.get();
            if (currentGroup != null) {
                return currentGroup;
            }

            EventLoopGroup newGroup = createBossGroup();
            if (bossGroupRef.compareAndSet(null, newGroup)) {
                Log.info("Netty BossGroup started using " + TRANSPORT_NAME + " transport");
                return newGroup;
            } else {
                Log.debug("Lost CAS race, shutting down redundant BossGroup");
                shutdownGroup(newGroup);
            }
        }
    }

    /**
     * Shuts down a given EventLoopGroup gracefully if it is not null.
     *
     * @param group The EventLoopGroup to shut down.
     */
    private void shutdownGroup(EventLoopGroup group) {
        if (group == null) {
            return;
        }
        Log.info("Shutting down EventLoopGroup: " + group.getClass().getSimpleName());
        Future<?> shutdownFuture = group.shutdownGracefully(0, 7, TimeUnit.SECONDS);
        try {
            shutdownFuture.await();
            Log.info("EventLoopGroup shut down successfully.");
        } catch (InterruptedException e) {
            Log.warn("Interrupted while waiting for Netty shutdown", e);
            Thread.currentThread().interrupt();
        }
    }

    /**
     * Called by clients/servers on creation to increment the ref count
     * and ensure the worker group is initialized.
     */
    public void acquireSharedResources() {
        int users = refCounter.incrementAndGet();
        Log.debug("Acquired Netty resources. Active users: " + users);
        ensureWorkerInitialized();
    }

    /**
     * Called by clients/servers on destroy() to decrement the ref count
     * and shut down groups if the count reaches zero.
     */
    public void releaseSharedResources() {
        int users = refCounter.decrementAndGet();
        Log.debug("Released Netty resources. Active users: " + users);

        if (false) {//
            Log.info("All users released Netty resources. Attempting shutdown...");

            EventLoopGroup worker = workerGroupRef.getAndSet(null);
            EventLoopGroup boss = bossGroupRef.getAndSet(null);

            shutdownGroup(worker);
            shutdownGroup(boss);
        }
    }


    /**
     * Shuts down the Netty I/O thread pools if this instance is closed.
     * Per AutoCloseable interface.
     */
    @Override
    public void close() {
        Log.warn("NettyFastMetaNet.close() called on singleton. Releasing one reference.");
        releaseSharedResources();
    }

    /**
     * Gets the appropriate Netty ServerChannel class based on the detected transport.
     *
     * @return The ServerChannel class.
     */
    public Class<? extends ServerChannel> getServerChannelClass() {
        return SERVER_CHANNEL_CLASS;
    }

    /**
     * Gets the appropriate Netty Channel class for clients based on the detected transport.
     *
     * @return The Channel class.
     */
    public Class<? extends io.netty.channel.Channel> getClientChannelClass() {
        return CLIENT_CHANNEL_CLASS;
    }

    /**
     * Gets the shared worker EventLoopGroup, initializing it if necessary.
     * Called by both clients and servers.
     *
     * @return The worker EventLoopGroup.
     */
    public EventLoopGroup getWorkerGroup() {
        return ensureWorkerInitialized();
    }

    /**
     * Gets the boss EventLoopGroup, initializing it if necessary.
     * Called ONLY by servers.
     *
     * @return The boss EventLoopGroup.
     */
    public EventLoopGroup getBossGroup() {
        return ensureBossInitialized();
    }

    @Override
    public <LT, RT extends RemoteApi> FastMetaClient<LT, RT> makeClient(
            URI uri,
            FastMetaApi<LT, ?> lt,
            FastMetaApi<?, RT> rt,
            AFunction<RT, LT> localApi,
            WritableConsumer writableConsumer) {

        acquireSharedResources();
        Log.info("Creating Netty FastMeta client", "uri", uri);
        try {
            return createClientInstance(uri, lt, rt, localApi, writableConsumer);
        } catch (Exception e) {
            releaseSharedResources();
            throw e;
        }
    }

    /**
     * Instantiates the protocol-specific client.
     * <p>
     * This method passes 'this' (the NettyFastMetaNet instance) to the client constructor.
     * This allows the client to call {@link #releaseSharedResources()} when it is destroyed.
     *
     * @param <LT>             The local API type.
     * @param <RT>             The remote API type.
     * @param uri              The URI defining the protocol and endpoint.
     * @param lt               The local API metadata.
     * @param rt               The remote API metadata.
     * @param localApi         The local API implementation provider.
     * @param writableConsumer The consumer for writable state changes.
     * @return A new {@link FastMetaClient} instance.
     * @throws IllegalArgumentException if the URI scheme is not supported.
     */
    private <LT, RT extends RemoteApi> FastMetaClient<LT, RT> createClientInstance(
            URI uri,
            FastMetaApi<LT, ?> lt,
            FastMetaApi<?, RT> rt,
            AFunction<RT, LT> localApi,
            WritableConsumer writableConsumer) {

        switch (uri.getScheme()) {
            case "tcp":
            case "udp":
            case "ws":
            case "wss":
                return new NettyFastMetaClient<>(this, uri, lt, rt, localApi, writableConsumer);
            default:
                throw new IllegalArgumentException("Unsupported scheme: " + uri.getScheme());
        }
    }

    @Override
    public <LT, RT extends RemoteApi> FastMetaServer<LT, RT> makeServer(
            URI uri,
            FastMetaApi<LT, ?> localApiMeta,
            FastMetaApi<?, RT> remoteApiMeta,
            FastMetaServer.Handler<LT, RT> handler) {

        acquireSharedResources();
        Log.info("Creating Netty FastMeta server", "uri", uri);
        try {
            return createServerInstance(uri, localApiMeta, remoteApiMeta, handler);
        } catch (Exception e) {
            releaseSharedResources();
            throw e;
        }
    }

    /**
     * Instantiates the protocol-specific server.
     * <p>
     * This method passes 'this' (the NettyFastMetaNet instance) to the server constructor.
     * This allows the server to call {@link #releaseSharedResources()} when it is destroyed.
     *
     * @param <LT>          The local API type.
     * @param <RT>          The remote API type.
     * @param uri           The URI defining the protocol and endpoint.
     * @param localApiMeta  The local API metadata.
     * @param remoteApiMeta The remote API metadata.
     * @param handler       The server connection handler.
     * @return A new {@link FastMetaServer} instance.
     * @throws IllegalArgumentException if the URI scheme is not supported.
     */
    private <LT, RT extends RemoteApi> FastMetaServer<LT, RT> createServerInstance(
            URI uri,
            FastMetaApi<LT, ?> localApiMeta,
            FastMetaApi<?, RT> remoteApiMeta,
            FastMetaServer.Handler<LT, RT> handler) {

        switch (uri.getScheme()) {
            case "tcp":
            case "udp":
            case "ws":
            case "wss":
                return new NettyFastMetaServer<>(this, uri, localApiMeta, remoteApiMeta, handler);
            default:
                throw new IllegalArgumentException("Unsupported scheme: ".concat(uri.getScheme()));
        }
    }

    /**
     * Static initializer block to detect the best available native transport
     * and register this class as the global FastMetaNet.INSTANCE.
     */
    static {
        if (Epoll.isAvailable()) {
            TRANSPORT_NAME = "Epoll";
            SERVER_CHANNEL_CLASS = EpollServerSocketChannel.class;
            CLIENT_CHANNEL_CLASS = EpollSocketChannel.class;
        } else if (KQueue.isAvailable()) {
            TRANSPORT_NAME = "KQueue";
            SERVER_CHANNEL_CLASS = KQueueServerSocketChannel.class;
            CLIENT_CHANNEL_CLASS = KQueueSocketChannel.class;
        } else {
            TRANSPORT_NAME = "NIO";
            SERVER_CHANNEL_CLASS = NioServerSocketChannel.class;
            CLIENT_CHANNEL_CLASS = NioSocketChannel.class;
        }

        FastMetaNet.INSTANCE.set(new NettyFastMetaNet());
        Log.info("NettyFastMetaNet registered: $nativeTransport", "nativeTransport", TRANSPORT_NAME);
    }
}