package io.aether.net.fastMeta.netty;

import io.aether.logger.LNode;
import io.aether.logger.Log;
import io.aether.net.fastMeta.*;
import io.aether.utils.ConcurrentHashSet;
import io.aether.utils.futures.AFuture;
import io.netty.bootstrap.Bootstrap;
import io.netty.bootstrap.ServerBootstrap;
import io.netty.buffer.ByteBuf;
import io.netty.channel.*;
import io.netty.channel.socket.DatagramChannel;
import io.netty.channel.socket.DatagramPacket;
import io.netty.channel.socket.SocketChannel;

import java.net.SocketAddress;
import java.net.URI;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import io.netty.handler.codec.MessageToMessageDecoder;
import io.netty.handler.codec.MessageToMessageEncoder;
import io.netty.handler.codec.http.HttpClientCodec;
import io.netty.handler.codec.http.HttpObjectAggregator;
import io.netty.handler.codec.http.HttpServerCodec;
import io.netty.handler.codec.http.websocketx.BinaryWebSocketFrame;
import io.netty.handler.codec.http.websocketx.WebSocketClientHandshaker;
import io.netty.handler.codec.http.websocketx.WebSocketClientHandshakerFactory;
import io.netty.handler.codec.http.websocketx.WebSocketServerProtocolHandler;
import io.netty.handler.codec.http.websocketx.WebSocketVersion;
import io.netty.handler.codec.http.websocketx.extensions.compression.WebSocketClientCompressionHandler;
import io.netty.handler.codec.http.websocketx.extensions.compression.WebSocketServerCompressionHandler;
import io.netty.handler.codec.http.DefaultHttpHeaders;
/**
 * Netty-based TCP and UDP implementation of {@link FastMetaServer}.
 *
 * This class switches behavior based on the URI scheme.
 *
 * <h2>TCP Mode ("tcp://")</h2>
 * Uses a {@link ServerBootstrap} to listen for connections and a
 * {@link ChannelInitializer} to set up the pipeline (codecs and handler)
 * for each new client {@link SocketChannel}.
 *
 * <h2>UDP Mode ("udp://")</h2>
 * Uses a {@link Bootstrap} to bind a single {@link DatagramChannel}.
 * A custom {@link UdpServerHandler} receives all {@link DatagramPacket}s
 * and manages a map of "virtual" connections ({@link NettyServerConnectionHandler})
 * based on the sender's {@link SocketAddress}.
 */
class NettyFastMetaServer<LT, RT extends RemoteApi> implements FastMetaServer<LT, RT> {

    private final LNode logContext;
    private final Channel serverChannel; // The main server channel (TCP or UDP)
    private final FastMetaServer.Handler<LT, RT> handler;
    private final boolean isUdp;
    private final NettyFastMetaNet nettyNet;

    /**
     * A thread-safe map of all active TCP connections. (Used in TCP mode only)
     */
    private final Map<FastMetaNet.Connection<LT, RT>, Boolean> tcpConnections = new ConcurrentHashMap<>();

    /**
     * The single handler for the UDP channel. (Used in UDP mode only)
     */
    private UdpServerHandler udpHandler;

    NettyFastMetaServer(NettyFastMetaNet nettyNet, URI uri,
                        FastMetaApi<LT, ?> localApiMeta,
                        FastMetaApi<?, RT> remoteApiMeta,
                        FastMetaServer.Handler<LT, RT> handler) {

        this.nettyNet = nettyNet;
        this.handler = handler;
        this.logContext = Log.of("uri", uri, "socket", "server netty");
        this.isUdp = "udp".equals(uri.getScheme());

        if (isUdp) {
            this.serverChannel = startUdpServer(uri, localApiMeta, remoteApiMeta, handler);
        } else {
            this.serverChannel = startTcpServer(uri, localApiMeta, remoteApiMeta, handler);
        }
    }

    /**
     * Starts the server in TCP mode.
     */
    private Channel startTcpServer(URI uri,
                                   FastMetaApi<LT, ?> localApiMeta,
                                   FastMetaApi<?, RT> remoteApiMeta,
                                   FastMetaServer.Handler<LT, RT> handler) {
        try (var _l = logContext.context()) {
            boolean isWebSocket = "ws".equals(uri.getScheme()) || "wss".equals(uri.getScheme());

            if (isWebSocket) {
                Log.info("Starting Netty WebSocket server...");
            } else {
                Log.info("Starting Netty TCP server...");
            }

            ServerBootstrap b = new ServerBootstrap();
            b.group(nettyNet.getBossGroup(), nettyNet.getWorkerGroup())
                    .channel(nettyNet.getServerChannelClass())
                    .option(ChannelOption.SO_BACKLOG, 128)
                    .childOption(ChannelOption.SO_KEEPALIVE, true)
                    .childHandler(new ChannelInitializer<SocketChannel>() {
                        @Override
                        public void initChannel(SocketChannel ch) {
                            try (var connectionLogContext = logContext.add("remote", ch.remoteAddress()).context()) {
                                Log.info("Accepted new connection");

                                // 1. Создаем connHandler (он будет в конце пайплайна)
                                NettyServerConnectionHandler<LT, RT> connHandler = new NettyServerConnectionHandler<>(
                                        ch,
                                        connectionLogContext.node(),
                                        localApiMeta,
                                        remoteApiMeta,
                                        handler,
                                        NettyFastMetaServer.this::onConnectionClosed
                                );

                                // 2. Добавляем хэндлеры в зависимости от протокола
                                if (isWebSocket) {
                                    // --- WS/WSS PIPELINE (ИСПРАВЛЕННЫЙ ПОРЯДОК) ---
                                    ch.pipeline().addLast(
                                            new HttpServerCodec(),
                                            new HttpObjectAggregator(65536),
                                            // new WebSocketServerCompressionHandler(), // Компрессию пока отключаем
                                            new WebSocketServerProtocolHandler("/", null, true)
                                    );

                                    // --- Inbound (Порядок: Сверху Вниз) ---
                                    // 1. Адаптер: BinaryWebSocketFrame -> ByteBuf
                                    ch.pipeline().addLast("ws_to_buf_decoder", new MessageToMessageDecoder<BinaryWebSocketFrame>() {
                                        @Override
                                        protected void decode(ChannelHandlerContext ctx, BinaryWebSocketFrame msg, List<Object> out) {
                                            out.add(msg.content().retain());
                                        }
                                    });
                                    // 2. Декодер заголовка: ByteBuf -> ByteBuf (payload)
                                    ch.pipeline().addLast("frameDecoder", new FastMetaFrameDecoder());

                                    // --- Outbound (Порядок: Снизу Вверх - ОБРАТНЫЙ) ---
                                    // 2. Адаптер: ByteBuf -> BinaryWebSocketFrame (Добавляем ПЕРВЫМ)
                                    ch.pipeline().addLast("buf_to_ws_encoder", new MessageToMessageEncoder<ByteBuf>() {
                                        @Override
                                        protected void encode(ChannelHandlerContext ctx, ByteBuf msg, List<Object> out) {
                                            out.add(new BinaryWebSocketFrame(msg.retain()));
                                        }
                                    });
                                    // 1. Энкодер заголовка: byte[] -> ByteBuf (Добавляем ВТОРЫМ)
                                    ch.pipeline().addLast("frameEncoder", new FastMetaFrameEncoder());

                                    // 3. (In) / 3. (Out) Главный обработчик (Добавляем ПОСЛЕДНИМ)
                                    ch.pipeline().addLast("handler", connHandler);

                                } else {
                                    // TCP pipeline (без изменений)
                                    ch.pipeline().addLast("frameDecoder", new FastMetaFrameDecoder());
                                    ch.pipeline().addLast("frameEncoder", new FastMetaFrameEncoder());
                                    ch.pipeline().addLast("handler", connHandler);
                                }

                                // 3. Add to active list
                                tcpConnections.put(connHandler, true);

                            } catch (Exception e) {
                                Log.warn("Failed to initialize new connection pipeline", e);
                                ch.close();
                            }
                        }
                    });

            // Bind and start to accept incoming connections.
            ChannelFuture bindFuture = b.bind(uri.getHost(), uri.getPort());
            Channel ch = bindFuture.sync().channel();

            String protocol = isWebSocket ? "WebSocket" : "TCP";
            Log.info("Netty " + protocol + " server started and listening on " +
                     uri.getHost() + ":" + uri.getPort());
            return ch;

        } catch (Exception e) {
            Log.error("Failed to start Netty server", e);
            throw new RuntimeException("Failed to start Netty server", e);
        }
    }
    /**
     * Starts the server in UDP mode.
     */
    private Channel startUdpServer(URI uri,
                                   FastMetaApi<LT, ?> localApiMeta,
                                   FastMetaApi<?, RT> remoteApiMeta,
                                   FastMetaServer.Handler<LT, RT> handler) {
        try (var _l = logContext.context()) {
            Log.info("Starting Netty UDP server...");
            this.udpHandler = new UdpServerHandler(localApiMeta, remoteApiMeta, handler);

            // For UDP, we need to use the appropriate DatagramChannel class based on transport
            Class<? extends DatagramChannel> datagramChannelClass = getDatagramChannelClass();

            Bootstrap b = new Bootstrap();
            b.group(nettyNet.getWorkerGroup())
                    .channel(datagramChannelClass)
                    .option(ChannelOption.SO_BROADCAST, false) // Typically false for servers
                    .handler(this.udpHandler); // Single handler for the one channel

            // Bind and start to accept incoming datagrams.
            ChannelFuture bindFuture = b.bind(uri.getHost(), uri.getPort());
            Channel ch = bindFuture.sync().channel();
            Log.info("Netty UDP server started and listening on " + uri.getHost() + ":" + uri.getPort());
            return ch;

        } catch (Exception e) {
            Log.error("Failed to start Netty UDP server", e);
            throw new RuntimeException("Failed to start Netty UDP server", e);
        }
    }

    /**
     * Gets the appropriate DatagramChannel class based on the selected transport.
     */
    private Class<? extends DatagramChannel> getDatagramChannelClass() {
        Class<? extends io.netty.channel.Channel> clientChannelClass = nettyNet.getClientChannelClass();

        // Map client channel class to corresponding datagram channel class
        if (clientChannelClass == io.netty.channel.epoll.EpollSocketChannel.class) {
            return io.netty.channel.epoll.EpollDatagramChannel.class;
        } else if (clientChannelClass == io.netty.channel.kqueue.KQueueSocketChannel.class) {
            return io.netty.channel.kqueue.KQueueDatagramChannel.class;
        } else {
            return io.netty.channel.socket.nio.NioDatagramChannel.class;
        }
    }

    /**
     * Callback for when a connection (TCP or UDP) is closed.
     * @param connection The connection that closed.
     */
    private void onConnectionClosed(FastMetaNet.Connection<LT, RT> connection) {
        if (tcpConnections.remove(connection) != null) {
            Log.info("TCP connection removed from server", ((NettyServerConnectionHandler)connection).getLogContext());
        }
        // For UDP, the removal from the map happens inside UdpServerHandler
    }

    @Override
    public AFuture stop() {
        return destroy(false);
    }

    @Override
    public AFuture destroy(boolean force) {
        Log.info("Destroying Netty server...", logContext, "isUdp", isUdp);
        AFuture res = AFuture.make();

        // Close all active client connections
        for (FastMetaNet.Connection<LT, RT> conn : handlers()) {
            conn.destroy(force);
        }
        if (isUdp) {
            udpHandler.clearConnections();
        } else {
            tcpConnections.clear();
        }

        // Close the main server channel
        serverChannel.close().addListener(future -> {
            if (future.isSuccess()) {
                Log.info("Netty server socket closed", logContext);
                res.tryDone();
            } else {
                Log.warn("Error closing Netty server socket", future.cause());
                res.tryError(future.cause());
            }
        });
        res.to(() -> {
            nettyNet.releaseSharedResources();
            Log.debug("Netty server destroy complete, released resources.");
        });
        return res;
    }

    @Override
    public Iterable<FastMetaNet.Connection<LT, RT>> handlers() {
        // Return a thread-safe snapshot
        if (isUdp) {
            return new ConcurrentHashSet<>(udpHandler.getConnections());
        } else {
            return new ConcurrentHashSet<>(tcpConnections.keySet());
        }
    }

    /**
     * Inner handler class for UDP server mode.
     * Manages all "virtual" connections over a single DatagramChannel.
     */
    private class UdpServerHandler extends ChannelInboundHandlerAdapter {

        private final Map<SocketAddress, NettyServerConnectionHandler<LT, RT>> udpConnections = new ConcurrentHashMap<>();
        private final FastMetaApi<LT, ?> localApiMeta;
        private final FastMetaApi<?, RT> remoteApiMeta;
        private final FastMetaServer.Handler<LT, RT> serverHandler;
        private ChannelHandlerContext mainCtx;

        UdpServerHandler(FastMetaApi<LT, ?> localApiMeta,
                         FastMetaApi<?, RT> remoteApiMeta,
                         FastMetaServer.Handler<LT, RT> serverHandler) {
            this.localApiMeta = localApiMeta;
            this.remoteApiMeta = remoteApiMeta;
            this.serverHandler = serverHandler;
        }

        @Override
        public void handlerAdded(ChannelHandlerContext ctx) {
            this.mainCtx = ctx;
        }

        @Override
        public void channelRead(ChannelHandlerContext ctx, Object msg) {
            if (!(msg instanceof DatagramPacket)) {
                return;
            }

            DatagramPacket packet = (DatagramPacket) msg;
            SocketAddress sender = packet.sender();

            try (var connectionLogContext = logContext.add("remote", sender).context()) {
                NettyServerConnectionHandler<LT, RT> conn = udpConnections.get(sender);

                if (conn == null) {
                    // New "connection"
                    Log.info("Accepted new UDP connection");

                    conn = new NettyServerConnectionHandler<>(
                            mainCtx,
                            sender,
                            connectionLogContext.node(),
                            localApiMeta,
                            remoteApiMeta,
                            serverHandler,
                            this::onUdpConnectionClosed
                    );
                    // Manually initialize the connection
                    conn.udpInit();
                    udpConnections.put(sender, conn);
                }

                // Forward the payload (ByteBuf) to the connection handler for processing.
                // We must retain() because the NettyServerConnectionHandler will release() it.
                conn.channelRead(ctx, packet.content().retain());

            } catch (Exception e) {
                Log.error("Error processing UDP datagram", e);
            } finally {
                packet.release(); // Release the DatagramPacket container
            }
        }

        @Override
        public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) {
            try (var _l = logContext.context()) {
                // Don't close the server, just log the error
                Log.warn("Netty UDP server handler exception caught", cause);
            }
        }

        /**
         * Callback for when a UDP connection is closed (e.g., by destroy()).
         */
        private void onUdpConnectionClosed(SocketAddress remoteAddress, FastMetaNet.Connection<LT, RT> connection) {
            if (udpConnections.remove(remoteAddress, connection)) {
                Log.info("UDP connection removed from server", ((NettyServerConnectionHandler)connection).getLogContext());
                // The handler.onConnectionClose() is called *inside* NettyServerConnectionHandler
                // before this callback is invoked.
                onConnectionClosed(connection); // Notify outer server (e.g., for logging)
            }
        }

        /**
         * Gets a snapshot of current connections.
         */
        public Collection<FastMetaNet.Connection<LT, RT>> getConnections() {
            return new HashSet<>(udpConnections.values());
        }

        /**
         * Clears all connections during server shutdown.
         */
        public void clearConnections() {
            udpConnections.clear();
        }
    }
}