package io.aether.net.fastMeta.netty;

import io.aether.logger.LNode;
import io.aether.logger.Log;
import io.aether.net.fastMeta.*;
import io.aether.utils.dataio.DataInOut;
import io.aether.utils.futures.AFuture;
import io.netty.buffer.ByteBuf;
import io.netty.channel.Channel;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelInboundHandlerAdapter;
import io.netty.channel.socket.DatagramPacket;

import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

/**
 * Manages a single server-side connection for FastMeta.
 * <p>
 * This class implements FastMetaNet.Connection and can operate in two modes:
 * <p>
 * 1.  TCP Mode: It is the final handler in a dedicated client
 * {@link io.netty.channel.socket.SocketChannel} pipeline.
 * <p>
 * 2.  UDP Mode: It is *not* in the pipeline. It is a state object managed by
 * {@link NettyFastMetaServer.UdpServerHandler} in a Map, keyed by
 * {@link SocketAddress}. It sends/receives data via the main server
 * DatagramChannel's context.
 */
class NettyServerConnectionHandler<LT, RT extends RemoteApi> extends ChannelInboundHandlerAdapter implements FastMetaNet.Connection<LT, RT> {

    // --- Mode-agnostic fields ---
    private final FastMetaApi<LT, ?> localApiMeta;
    private final FastMetaApi<?, RT> remoteApiMeta;
    private final FastMetaServer.Handler<LT, RT> serverHandler;
    private final LNode logContext; // <-- Теперь инициализируется из конструктора

    // --- TCP-specific fields ---
    private final Channel channel; // The dedicated client SocketChannel (null in UDP mode)
    private final Consumer<FastMetaNet.Connection<LT, RT>> tcpCloseCallback; // (null in UDP mode)

    // --- UDP-specific fields ---
    private final ChannelHandlerContext mainCtx; // The main DatagramChannel context (null in TCP mode)
    private final SocketAddress remoteAddress; // The remote client address (null in TCP mode)
    private final BiConsumer<SocketAddress, FastMetaNet.Connection<LT, RT>> udpCloseCallback; // (null in TCP mode)

    // --- State fields ---
    private ChannelHandlerContext ctx; // The context to use for writing (SocketChannel ctx or main DatagramChannel ctx)
    private FastApiContext context;
    private RT remoteApi;
    private LT localApi;

    /**
     * Constructor for TCP Mode.
     * Called by NettyFastMetaServer's ChannelInitializer.
     */
    NettyServerConnectionHandler(Channel channel,
                                 LNode parentLogContext, // <-- ИЗМЕНЕНО: Принимаем контекст
                                 FastMetaApi<LT, ?> localApiMeta,
                                 FastMetaApi<?, RT> remoteApiMeta,
                                 FastMetaServer.Handler<LT, RT> serverHandler,
                                 Consumer<FastMetaNet.Connection<LT, RT>> closeCallback) {
        this.channel = channel;
        this.localApiMeta = localApiMeta;
        this.remoteApiMeta = remoteApiMeta;
        this.serverHandler = serverHandler;
        this.tcpCloseCallback = closeCallback;
        this.logContext = parentLogContext; // <-- ИЗМЕНЕНО: Используем родительский контекст

        // Null out UDP fields
        this.mainCtx = null;
        this.remoteAddress = null;
        this.udpCloseCallback = null;
    }

    /**
     * Constructor for UDP Mode.
     * Called by NettyFastMetaServer's UdpServerHandler.
     */
    NettyServerConnectionHandler(ChannelHandlerContext mainCtx,
                                 SocketAddress remoteAddress,
                                 LNode parentLogContext, // <-- ИЗМЕНЕНО: Принимаем контекст
                                 FastMetaApi<LT, ?> localApiMeta,
                                 FastMetaApi<?, RT> remoteApiMeta,
                                 FastMetaServer.Handler<LT, RT> serverHandler,
                                 BiConsumer<SocketAddress, FastMetaNet.Connection<LT, RT>> udpCloseCallback) {
        this.mainCtx = mainCtx;
        this.remoteAddress = remoteAddress;
        this.localApiMeta = localApiMeta;
        this.remoteApiMeta = remoteApiMeta;
        this.serverHandler = serverHandler;
        this.udpCloseCallback = udpCloseCallback;
        this.logContext = parentLogContext; // <-- ИЗМЕНЕНО: Используем родительский контекст

        // Null out TCP fields
        this.channel = null;
        this.tcpCloseCallback = null;
    }


    public LNode getLogContext() {
        return logContext;
    }

    /**
     * Called when this handler is added to the pipeline (TCP Mode only).
     */
    @Override
    public void handlerAdded(ChannelHandlerContext ctx) {
        this.ctx = ctx;
        initializeApis();
    }

    /**
     * Called manually by UdpServerHandler (UDP Mode only).
     */
    public void udpInit() {
        this.ctx = this.mainCtx; // Use the main server's context for writing
        initializeApis();
    }

    /**
     * Common logic for initializing the API context, remote proxy, and local implementation.
     */
    private void initializeApis() {
        // 1. Create context
        this.context = new FastApiContext() {
            @Override
            public void flush(AFuture sendFuture) {
                handleFlush(sendFuture);
            }
        };

        // 2. Create remote proxy
        this.remoteApi = remoteApiMeta.makeRemote(this.context);

        // 3. Get the user's local API implementation for this connection
        // This is the critical hand-off
        try (var _l = logContext.context()) {
            this.localApi = serverHandler.onNewConnection(this);
        } catch (Exception e) {
            Log.error("Handler onNewConnection failed, closing connection", e, logContext);
            ctx.close(); // For TCP, this closes the client. For UDP, this closes the SERVER.
            // TODO: In UDP mode, ctx.close() is catastrophic.
            // We should just destroy() this "virtual" connection.
            if (this.remoteAddress != null) {
                destroy(true);
            }
        }
    }


    /**
     * Called when the channel becomes inactive (TCP Mode only).
     */
    @Override
    public void channelInactive(ChannelHandlerContext ctx) {
        runCloseLogic();
        tcpCloseCallback.accept(this);
    }

    /**
     * Common logic for notifying the user's handler of connection closure.
     */
    private void runCloseLogic() {
        try (var _l = logContext.context()) {
            Log.info("Connection closed");
            serverHandler.onConnectionClose(this);
        } catch (Exception e) {
            Log.warn("Handler onConnectionClose failed", e, logContext);
        }
    }

    @Override
    public void channelRead(ChannelHandlerContext ctx, Object msg) {
        // --- WRAP ENTIRE METHOD IN LOG CONTEXT ---
        if (msg instanceof ByteBuf payload) {
            try (var _l = logContext.context()) {
                // In TCP mode, msg is a ByteBuf (from FastMetaFrameDecoder)
                // In UDP mode, UdpServerHandler sends us the ByteBuf (from DatagramPacket.content())

                try {
                    // Wrap the payload for consumption by makeLocal
                    ByteBufDataIO in = new ByteBufDataIO(payload);

                    // localApiMeta.makeLocal expects DataIn
                    localApiMeta.makeLocal(context, in, localApi);

                } catch (Exception e) {
                    // Контекст лога уже установлен внешним try-with-resources
                    Log.error("Error during makeLocal (packet processing).", e);
                    if (channel != null) {
                        ctx.close(); // Close TCP connection
                    }
                    // In UDP mode, we just log and drop the packet.
                } finally {
                    if (payload != null) {
                        payload.release(); // Release the ByteBuf
                    }
                }
            }
        }
    }

    @Override
    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) {
        // --- WRAP ENTIRE METHOD IN LOG CONTEXT ---
        try (var _l = logContext.context()) {
            // This is only called in TCP mode
            Log.warn("Netty server connection exception caught", cause);
            ctx.close(); // This will trigger channelInactive
        }
    }

    /**
     * Called by the custom {@link FastApiContext} when {@code flush()} is invoked.
     * This runs on the *user's thread*.
     */
    private void handleFlush(AFuture sendFuture) {
        // --- WRAP ENTIRE METHOD IN LOG CONTEXT ---
        try (var _l = logContext.context()) {
            if (!isWritable()) {
                sendFuture.tryCancel();
                return;
            }

            DataInOut combinedData = new DataInOut();
            context.remoteDataToArray(combinedData);

            if (combinedData.getSizeForRead() == 0) {
                sendFuture.tryDone();
                return;
            }

            byte[] payload = combinedData.toArray();
            Object packetToWrite;

            if (remoteAddress == null) {
                // TCP Mode: Send raw payload, FrameEncoder will prefix it
                packetToWrite = payload;
            } else {
                // UDP Mode: Send DatagramPacket
                ByteBuf buf = ctx.alloc().buffer(payload.length);
                buf.writeBytes(payload);
                // --- FIX: Cast SocketAddress to InetSocketAddress ---
                packetToWrite = new DatagramPacket(buf, (InetSocketAddress) this.remoteAddress);
            }

            // ctx is the SocketChannel ctx in TCP mode, or the DatagramChannel ctx in UDP mode
            ctx.writeAndFlush(packetToWrite).addListener(future -> {
                // --- WRAP LISTENER IN LOG CONTEXT ---
                try (var _l_listener = logContext.context()) {
                    if (future.isSuccess()) {
                        sendFuture.tryDone();
                    } else {
                        sendFuture.tryError(future.cause());
                    }
                }
            });
        }
    }

    // --- FastMetaNet.Connection Implementation ---

    @Override
    public void read() { /* No-op */ }

    @Override
    public void stopRead() { /* No-op */ }

    @Override
    public AFuture write(byte[] data) {
        // --- WRAP ENTIRE METHOD IN LOG CONTEXT ---
        try (var _l = logContext.context()) {
            if (!isWritable()) {
                return AFuture.canceled();
            }

            AFuture res = AFuture.make();
            Object packetToWrite;

            if (remoteAddress == null) {
                // TCP Mode: Send raw payload, FrameEncoder will prefix it
                packetToWrite = data;
            } else {
                // UDP Mode: Send DatagramPacket
                ByteBuf buf = ctx.alloc().buffer(data.length);
                buf.writeBytes(data);
                // --- FIX: Cast SocketAddress to InetSocketAddress ---
                packetToWrite = new DatagramPacket(buf, (InetSocketAddress) this.remoteAddress);
            }

            ctx.writeAndFlush(packetToWrite).addListener(future -> {
                // --- WRAP LISTENER IN LOG CONTEXT ---
                try (var _l_listener = logContext.context()) {
                    if (future.isSuccess()) {
                        res.tryDone();
                    } else {
                        Log.warn("Netty server raw write failed", future.cause());
                        res.tryError(future.cause());
                    }
                }
            });
            return res;
        }
    }

    @Override
    public LT getLocalApi() {
        return localApi;
    }

    @Override
    public RT getRemoteApi() {
        return remoteApi;
    }

    @Override
    public boolean isWritable() {
        // TCP check: dedicated channel must be writable
        // UDP check: main server channel must be writable
        return (channel != null && channel.isOpen() && channel.isWritable())
               || (mainCtx != null && mainCtx.channel().isWritable());
    }

    @Override
    public FastFutureContext getMetaContext() {
        return context;
    }

    @Override
    public AFuture destroy(boolean force) {
        AFuture res = AFuture.make();
        if (remoteAddress == null) {
            // TCP Mode: Close the dedicated channel
            ctx.close().addListener(future -> {
                if (future.isSuccess()) res.tryDone();
                else res.tryError(future.cause());
            });
            // channelInactive will handle the rest
        } else {
            // UDP Mode: Run close logic and notify the UdpServerHandler
            runCloseLogic();
            udpCloseCallback.accept(this.remoteAddress, this);
            res.tryDone();
        }
        return res;
    }
}