package io.aether.net.fastMeta.nio;

import io.aether.logger.Log;
import io.aether.net.fastMeta.*;
import io.aether.utils.Destroyer;
import io.aether.utils.futures.AFuture;
import io.aether.utils.interfaces.AFunction;
import io.aether.utils.interfaces.Destroyable;

import java.io.IOException;
import java.net.URI;

/**
 * NIO-based implementation of the FastMetaNet factory.
 * This class creates and manages non-blocking clients and servers.
 * It runs a central I/O event loop (Reactor) in a dedicated thread
 * and uses a Destroyer to manage all created resources.
 */
public class NioFastMetaNet implements FastMetaNet, AutoCloseable {

    static {
        // Register this implementation as the global instance
        FastMetaNet.INSTANCE.set(new NioFastMetaNet());
        Log.info("NioFastMetaNet registered as FastMetaNet.INSTANCE"); //
    }

    private final NioReactor reactor;
    private final Thread reactorThread;
    private final Destroyer destroyer;

    /**
     * Initializes the NioFastMetaNet and starts the underlying I/O reactor thread.
     */
    public NioFastMetaNet() {
        this.destroyer = new Destroyer("NioFastMetaNet"); //
        try {
            this.reactor = new NioReactor();
            this.reactorThread = new Thread(this.reactor, "fastmeta-nio-reactor");
            this.reactorThread.setDaemon(true); // Allow JVM shutdown
            this.reactorThread.start();
            Log.info("NIO Reactor thread started"); //

            // Add reactor shutdown logic to the destroyer
            addReactorShutdownHook();

        } catch (IOException e) {
            Log.error("Failed to initialize NIO Reactor", e); //
            throw new RuntimeException("Failed to initialize NIO Reactor", e);
        }
    }

    /**
     * Adds a task to the destroyer to stop the reactor and join its thread.
     */
    private void addReactorShutdownHook() {
        this.destroyer.add(new Destroyable() { //
            @Override
            public String toString() {
                return "NioReactor-Shutdown";
            }

            @Override
            public AFuture destroy(boolean force) {
                Log.info("NioReactor stopping..."); //
                reactor.stop();
                try {
                    reactorThread.join(2000); // Wait for reactor thread to die
                } catch (InterruptedException e) {
                    Log.warn("Interrupted while waiting for reactor shutdown", e); //
                    Thread.currentThread().interrupt();
                }
                return AFuture.completed(); //
            }
        });
    }

    /**
     * Shuts down the NIO reactor thread and all created clients/servers.
     */
    @Override
    public void close() {
        Log.info("Shutting down NioFastMetaNet..."); //
        destroyer.destroy(false); //
    }

    @Override
    public <LT, RT extends RemoteApi> FastMetaClient<LT, RT> makeClient(
            URI uri,
            FastMetaApi<LT, ?> lt,
            FastMetaApi<?, RT> rt,
            AFunction<RT, LT> localApi,
            WritableConsumer writableConsumer) {

        Log.info("Creating FastMeta client", "uri", uri); //
        FastMetaClient<LT, RT> client;
        switch (uri.getScheme()) {
            case "tcp":
                client = new NioTcpFastMetaClient<>(reactor, uri, lt, rt, localApi, writableConsumer);
                break;
            case "udp":
                Log.error("UDP client is not yet supported", "uri", uri); //
                throw new UnsupportedOperationException("UDP client not yet implemented");
            case "ws":
                Log.error("WebSocket (ws) client is not yet supported", "uri", uri); //
                throw new UnsupportedOperationException("WebSocket client not yet implemented");
            default:
                Log.error("Unsupported client URI scheme", "scheme", uri.getScheme(), "uri", uri); //
                throw new IllegalArgumentException("Unsupported scheme: " + uri.getScheme());
        }

        destroyer.add(client); //
        return client;
    }

    @Override
    public <LT, RT extends RemoteApi> FastMetaServer<LT, RT> makeServer(
            URI uri,
            FastMetaApi<LT, ?> localApiMeta,
            FastMetaApi<?, RT> remoteApiMeta,
            FastMetaServer.Handler<LT, RT> handler) {

        Log.info("Creating FastMeta server", "uri", uri); //
        FastMetaServer<LT, RT> server;
        switch (uri.getScheme()) {
            case "tcp":
                server = new NioTcpFastMetaServer<>(reactor, uri, localApiMeta, remoteApiMeta, handler);
                break;
            case "udp":
                Log.error("UDP server is not yet supported", "uri", uri); //
                throw new UnsupportedOperationException("UDP server not yet implemented");
            case "ws":
                Log.error("WebSocket (ws) server is not yet supported", "uri", uri); //
                throw new UnsupportedOperationException("WebSocket server not yet implemented");
            default:
                Log.error("Unsupported server URI scheme", "scheme", uri.getScheme(), "uri", uri); //
                throw new IllegalArgumentException("Unsupported scheme: " + uri.getScheme());
        }

        destroyer.add(server); //
        return server;
    }
}