package io.aether.net.fastMeta.nio;

import io.aether.logger.LNode;
import io.aether.logger.Log;

import java.io.IOException;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.SelectableChannel;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.util.Iterator;
import java.util.concurrent.ConcurrentLinkedQueue;

/**
 * Implements the Reactor pattern for non-blocking I/O.
 * Runs on a single dedicated thread, dispatching I/O events (accept, connect, read, write)
 * to their respective {@link NioEventHandler}s.
 * <p>
 * All interactions with NIO channels (registering, changing ops) must be
 * executed on this thread via {@link #addTask(Runnable)}.
 */
class NioReactor implements Runnable {
    private final Selector selector;
    private final ConcurrentLinkedQueue<Runnable> tasks = new ConcurrentLinkedQueue<>();
    private volatile boolean running = true;
    private final LNode logContext = Log.of("socket", "client nio"); // Context for the reactor itself

    /**
     * Creates a new NioReactor and opens its selector.
     *
     * @throws IOException If the selector cannot be opened.
     */
    NioReactor() throws IOException {
        this.selector = Selector.open();
    }

    /**
     * Stops the reactor loop gracefully.
     */
    public void stop() {
        try (var _l = logContext.context()) {
            Log.info("NioReactor stopping...");
        }
        running = false;
        selector.wakeup(); // Force the select() to return
    }

    /**
     * Queues a task to be run on the reactor thread.
     * This is the only thread-safe way to interact with the selector
     * or channels managed by this reactor.
     *
     * @param task The task to execute on the reactor thread.
     */
    public void addTask(Runnable task) {
        tasks.add(task);
        selector.wakeup(); // Wake up the select() call to process the task
    }

    /**
     * The main event loop. Do not call directly; run in a dedicated thread.
     */
    @Override
    public void run() {
        // Set the context for the reactor thread itself
        try (var mainLoopContext = logContext.context()) {
            Log.info("NioReactor thread started.");
            while (running) {
                try {
                    // 1. Process all pending tasks (registrations, op changes, etc.)
                    // Tasks will run with their captured context thanks to Log.wrap()
                    runPendingTasks();

                    // 2. Wait for I/O events.
                    // This blocks until an event, a wakeup(), or an interrupt.
                    int selectedCount = selector.select(); // Timeout can be added here if needed

                    if (!running) {
                        break;
                    }

                    // Log if select returned without keys but wasn't woken up explicitly for tasks
                    // This might indicate spurious wakeups or issues.
                    if (selectedCount == 0 && tasks.isEmpty()) {
                        Log.trace("Selector woke up without keys or tasks.");
                        continue; // Go back to select
                    }


                    Iterator<SelectionKey> keys = selector.selectedKeys().iterator();
                    while (keys.hasNext()) {
                        SelectionKey key = keys.next();
                        keys.remove();

                        if (!key.isValid()) {
                            Log.trace("Skipping invalid key");
                            Object attachment = key.attachment();
                            if (attachment instanceof NioEventHandler) {
                                // Try to notify the handler even if the key is invalid, it might need cleanup
                                try {
                                    ((NioEventHandler) attachment).closeConnection(new IOException("SelectionKey became invalid"));
                                } catch (Exception ignored) { }
                            }
                            continue;
                        }

                        // 3. Dispatch the event to its handler
                        // Dispatch itself runs within the reactor's context
                        dispatch(key);
                    }
                } catch (IOException e) { // Catch select() errors
                    if (running) {
                        Log.error("NIO Reactor select() error", e);
                        // Potentially attempt to recover selector here, or shut down
                        running = false; // Example: Stop on select error
                    }
                } catch (Exception e) {
                    if (running) {
                        Log.error("NIO Reactor loop unexpected error", e);
                    }
                    // Continue running unless stopped
                }
            } // end while(running)
            Log.info("NioReactor loop finished.");
        } finally {
            // Cleanup selector even if loop context fails somehow
            try {
                selector.close();
                Log.info("NioReactor selector closed");
            } catch (IOException e) {
                Log.error("Error closing selector", e);
            }
        }
    }

    /**
     * Executes all tasks queued via {@link #addTask(Runnable)}.
     */
    private void runPendingTasks() {
        Runnable task;
        while ((task = tasks.poll()) != null) {
            try {
                // The task itself should have been wrapped by Log.wrap by the caller
                task.run();
            } catch (Exception e) {
                // Log error within the reactor's context
                try (var _l = logContext.context()) {
                    Log.error("Error executing reactor task", e);
                }
            }
        }
    }

    /**
     * Dispatches an I/O event to the handler attached to the selection key.
     *
     * @param key The key that has a pending event.
     */
    private void dispatch(SelectionKey key) {
        Object attachment = key.attachment();
        if (!(attachment instanceof NioEventHandler)) {
            // Log within the reactor's context
            Log.warn("SelectionKey has no NioEventHandler attachment", "keyInfo", key.toString());
            key.cancel(); // Cancel the key if it has no handler
            return;
        }

        NioEventHandler handler = (NioEventHandler) attachment;
        try {
            // The handler's handleEvent method should establish its own context
            handler.handleEvent(key);
        } catch (Exception e) {
            // Log within the reactor's context, including handler info
            Log.warn("NioEventHandler threw an exception during handleEvent", e, "handler", handler.toString());
            // Ask the handler to clean itself up
            try {
                handler.closeConnection(e);
            } catch (Exception closeEx) {
                Log.warn("Exception during handler.closeConnection after event error", closeEx, "handler", handler.toString());
            }
        }
    }

    /**
     * Safely registers a channel with this reactor.
     * This operation is queued and executed on the reactor thread.
     *
     * @param channel The channel to register.
     * @param ops     The interest operations (e.g., {@code SelectionKey.OP_READ}).
     * @param handler The handler to attach to this channel's key.
     */
    public void register(SelectableChannel channel, int ops, NioEventHandler handler) {
        addTask(Log.wrap(() -> {
            try {
                if (!channel.isOpen()) {
                    Log.warn("Attempted to register an already closed channel", "handler", handler);
                    if (handler != null) handler.closeConnection(new ClosedChannelException());
                    return;
                }
                channel.configureBlocking(false); // Should ideally be done before calling register
                SelectionKey key = channel.register(selector, ops, handler);
                Log.debug("Channel registered successfully", "ops", ops, "keyValid", key.isValid(), "handler", handler);
            } catch (ClosedChannelException e) {
                Log.warn("Channel was closed before registration could complete", e, "handler", handler);
                if (handler != null) handler.closeConnection(e); // Notify handler
            } catch (Exception e) {
                Log.error("Failed to register channel with selector", e, "handler", handler);
                if (handler != null) handler.closeConnection(e); // Notify handler of the failure
            }
        }));
    }
}