package io.aether.net.fastMeta.nio;

import io.aether.logger.LNode;
import io.aether.logger.Log;
import io.aether.net.fastMeta.*;
import io.aether.utils.dataio.DataIn;
import io.aether.utils.dataio.DataInOut;
import io.aether.utils.dataio.DataInOutStatic;
import io.aether.utils.futures.AFuture;
import io.aether.utils.interfaces.AFunction;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.URI;
import java.nio.ByteBuffer;
import java.nio.channels.SelectionKey;
import java.nio.channels.SocketChannel;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;

/**
 * Manages a single non-blocking TCP connection for FastMeta.
 * <p>
 * This class implements the framing logic for TCP:
 * 1.  **Reading:** Reads the packed size prefix, then reads the full payload.
 * 2.  **Writing:** Prefixes outgoing data with its packed size.
 * <p>
 * It also handles non-blocking write queues and read-buffer processing.
 */
class NioTcpConnection<LT, RT extends RemoteApi> implements FastMetaNet.Connection<LT, RT>, NioEventHandler {

    // --- Write Request Wrapper ---
    private static class WriteRequest {
        final ByteBuffer buffer;
        final AFuture future; // The future associated with this specific write operation

        WriteRequest(ByteBuffer buffer, AFuture future) {
            this.buffer = buffer;
            this.future = future;
        }
    }
    // ----------------------------

    private final NioReactor reactor;
    private final SocketChannel channel;
    private final FastMetaApi<LT, ?> localApiMeta;
    private final Consumer<NioTcpConnection<LT, RT>> closeCallback;
    private final LNode logContext;
    private final AtomicBoolean isWritableFlag = new AtomicBoolean(true); // Renamed to avoid confusion with method
    private final AtomicBoolean closing = new AtomicBoolean(false);

    private final FastApiContext context;
    private final RT remoteApi;
    private LT localApi; // Set by server/client after construction
    private SelectionKey selectionKey;

    // --- Read State ---
    private final ByteBuffer readBuffer = ByteBuffer.allocate(65536); // 64k
    private final DataInOut sizeBuffer = new DataInOut(16); // Buffer for decoding packet size
    private enum ReadState {
        READ_SIZE, READ_PAYLOAD
    }
    private ReadState readState = ReadState.READ_SIZE;
    private int payloadSize = -1;

    // --- Write State ---
    // --- ИЗМЕНЕНИЕ: Тип очереди ---
    private final Queue<WriteRequest> writeQueue = new ConcurrentLinkedQueue<>();
    // ----------------------------

    NioTcpConnection(NioReactor reactor, SocketChannel channel,
                     FastMetaApi<LT, ?> localApiMeta, FastMetaApi<?, RT> remoteApiMeta,
                     Consumer<NioTcpConnection<LT, RT>> closeCallback) {
        this.reactor = reactor;
        this.channel = channel;
        this.localApiMeta = localApiMeta;
        this.closeCallback = closeCallback;
        String remoteAddr = "unknown";
        try {
            remoteAddr = channel.getRemoteAddress().toString();
        } catch (IOException e) { /* ignore, may not be connected */ }
        this.logContext = Log.of("remote", remoteAddr, "socket", "server nio"); //

        // Create the context that will be used by the FastMeta API
        this.context = new FastApiContext() { //
            @Override
            public void flush(AFuture sendFuture) {
                // This is the hook. When the user calls flush() on the
                // RemoteApi, we intercept it here to queue the data.
                handleFlush(sendFuture);
            }
        };
        // Create the remote proxy using our custom context
        this.remoteApi = remoteApiMeta.makeRemote(this.context); //
    }

    /**
     * Sets the local API implementation.
     * Called by the server/client after construction.
     *
     * @param localApi The local API instance.
     */
    void setLocalApi(LT localApi) {
        this.localApi = localApi;
    }

    /**
     * Gets the logging context for this connection.
     *
     * @return The LNode context.
     */
    public LNode getLogContext() {
        return logContext;
    }

    // --- NioEventHandler Implementation ---

    @Override
    public void handleEvent(SelectionKey key) {
        this.selectionKey = key; // Cache the key for op changes

        try (var _l = logContext.context()) { //
            if (!key.isValid()) {
                // Ключ стал невалидным (например, канал закрылся)
                // Не пытаемся писать, просто закрываем соединение
                closeConnection(new IOException("Invalid selection key during event handling"));
                return;
            }
            // Check writable *before* readable to drain queue first
            if (key.isWritable()) {
                handleWrite();
            }
            // Проверяем валидность ключа *снова*, так как handleWrite мог его закрыть
            if (key.isValid() && key.isReadable()) {
                handleRead();
            }
        } catch (Exception e) {
            Log.warn("Error during event handling, closing connection", e); //
            closeConnection(e);
        }
    }

    /**
     * Handles an {@code OP_READ} event.
     */
    private void handleRead() {
        int bytesRead;
        try {
            bytesRead = channel.read(readBuffer);
        } catch (IOException e) {
            // e.g., "Connection reset by peer"
            Log.warn("Read error", e); //
            closeConnection(e);
            return;
        }

        if (bytesRead == -1) {
            // Remote disconnected gracefully
            Log.info("Remote closed connection (End of Stream)"); //
            closeConnection(new IOException("End of Stream"));
            return;
        }

        if (bytesRead > 0) {
            // Prepare the buffer for reading
            readBuffer.flip();
            // Process all complete packets in the buffer
            processReadBuffer();
            // Move any remaining partial data to the start
            readBuffer.compact();
        }
        // Если bytesRead == 0, ничего не делаем, ждем следующего события
    }

    /**
     * Processes the {@code readBuffer}, parsing size-prefixed packets.
     */
    private void processReadBuffer() {
        // Loop while there's data to process
        while (readBuffer.hasRemaining()) {
            if (readState == ReadState.READ_SIZE) {
                if (!tryReadPacketSize()) {
                    break; // Need more data for size
                }
            }

            if (readState == ReadState.READ_PAYLOAD) {
                if (readBuffer.remaining() >= payloadSize) {
                    // We have the full payload
                    byte[] payload = new byte[payloadSize];
                    readBuffer.get(payload);

                    // Dispatch the payload to FastMeta
                    dispatchPacket(payload);

                    // Reset for next packet
                    readState = ReadState.READ_SIZE;
                    payloadSize = -1;
                    sizeBuffer.clear(); // Очищаем буфер размера после успешного чтения пакета
                } else {
                    break; // Need more data for payload
                }
            }
        }
    }

    /**
     * Tries to parse a packet size from the {@code readBuffer}.
     * This method robustly handles a packed size that may be split
     * across multiple network reads by using an intermediate {@code sizeBuffer}.
     *
     * @return {@code true} if a size was successfully parsed, {@code false} if more data is needed.
     */
    private boolean tryReadPacketSize() {
        int initialReadPos = readBuffer.position();
        // Feed new data from network into our persistent size buffer
        // Читаем из readBuffer байт за байтом, пока не сможем распарсить размер или readBuffer не опустеет
        while (readBuffer.hasRemaining()) {
            byte b = readBuffer.get();
            sizeBuffer.writeByte(b); // Добавляем байт в буфер размера

            // Create a read-only *view* of the size buffer's underlying array
            DataInOutStatic sizeView = new DataInOutStatic(sizeBuffer.getData(), 0, sizeBuffer.getWritePos()); // Читаем с начала буфера

            try {
                // Attempt to deserialize the packed number from the view
                Number sizeNum = DeserializerPackNumber.INSTANCE.put(sizeView); //
                this.payloadSize = sizeNum.intValue();
                this.readState = ReadState.READ_PAYLOAD;

                // Размер успешно прочитан, очищаем буфер размера
                sizeBuffer.clear();

                return true;
            } catch (IndexOutOfBoundsException e) {
                // This happens if DeserializerPackNumber.INSTANCE.put() fails
                // because there isn't enough data in the sizeBuffer. Continue reading from readBuffer.
            } catch (Exception e) {
                // Any other exception during size parsing is likely fatal
                Log.error("Fatal error parsing packet size", e);
                closeConnection(e);
                return false; // Indicate failure
            }
        }

        // Если мы дошли сюда, значит readBuffer опустел, а размер так и не распарсили
        // Сбрасываем позицию readBuffer, так как мы "заглядывали" вперед, но не потребили байты для размера
        // (они все еще нужны и находятся в sizeBuffer)
        readBuffer.position(initialReadPos);
        return false; // Need more data
    }


    /**
     * Deserializes and dispatches a complete payload to the local API.
     *
     * @param payload The raw byte[] for a single FastMeta packet.
     */
    private void dispatchPacket(byte[] payload) {
        try {
            DataInOutStatic in = new DataInOutStatic(payload); //
            // This is the core dispatch logic
            localApiMeta.makeLocal(context, in, localApi); //
        } catch (Exception e) {
            Log.error("Error during makeLocal (packet processing). Closing connection.", e); //
            closeConnection(e);
        }
    }

    /**
     * Handles an {@code OP_WRITE} event.
     */
    private void handleWrite() {
        // --- ИЗМЕНЕНИЕ: Работаем с WriteRequest ---
        WriteRequest writeRequest = writeQueue.peek();

        while (writeRequest != null) {
            try {
                channel.write(writeRequest.buffer);
            } catch (IOException e) {
                Log.warn("Write error, closing connection", e); //
                // --- ИЗМЕНЕНИЕ: Отменяем future при ошибке записи ---
                writeRequest.future.tryCancel();
                // --------------------------------------------------
                closeConnection(e); // Закроет соединение и отменит остальные futures в очереди
                return;
            }

            if (writeRequest.buffer.hasRemaining()) {
                // Kernel buffer is full. Stop writing.
                isWritableFlag.set(false);
                // Keep OP_WRITE active and try again later.
                return;
            }

            // Wrote this buffer completely. Remove it and try the next one.
            writeQueue.poll();
            // --- ИЗМЕНЕНИЕ: Завершаем future после успешной записи ---
            writeRequest.future.tryDone();
            // --------------------------------------------------------
            writeRequest = writeQueue.peek();
        }

        // If we are here, the write queue is empty.
        isWritableFlag.set(true);
        // De-register OP_WRITE to stop busy-spinning.
        if (selectionKey.isValid()) {
            // Проверяем, действительно ли очередь пуста, прежде чем снимать OP_WRITE
            // (на случай, если handleFlush добавил что-то во время цикла)
            if (writeQueue.isEmpty()) {
                selectionKey.interestOps(selectionKey.interestOps() & ~SelectionKey.OP_WRITE);
            }
        }
        // -------------------------------------------------------------
    }


    /**
     * Called by the custom {@link FastApiContext} when {@code flush()} is invoked.
     * This runs on the *user's thread*.
     *
     * @param sendFuture The future to complete when data is queued *and sent*.
     */
    private void handleFlush(AFuture sendFuture) {
        try (var _l = logContext.context()) {
            // --- ИЗМЕНЕНИЕ: Проверка возможности записи ---
            if (closing.get()) {
                Log.trace("Flush called on closing connection, cancelling future.");
                sendFuture.tryCancel();
                return;
            }
            // Не проверяем isWritableFlag здесь, так как handleWrite его сбросит,
            // но мы все равно должны добавить в очередь. OP_WRITE будет активирован.
            // -------------------------------------------

            // 1. Get all queued data from the context
            DataInOut combinedData = new DataInOut();
            context.remoteDataToArray(combinedData); //

            if (combinedData.getSizeForRead() == 0) { //
                // Если нет данных от FastApiContext, но очередь записи НЕ пуста,
                // значит, предыдущий flush еще не завершился. Мы не можем
                // завершить этот sendFuture, пока очередь не опустеет.
                // Привяжем этот future к последнему элементу в очереди.
                if (!writeQueue.isEmpty()) {
                    // Это может быть не идеально потокобезопасно, но дает шанс.
                    // Лучше было бы иметь future последнего элемента.
                    Log.trace("Flush called with empty context data but non-empty write queue. Linking future.");
                    // Найдем последнюю задачу (без удаления)
                    WriteRequest lastRequest = null;
                    for (WriteRequest req : writeQueue) { // Итератор ConcurrentLinkedQueue слабо согласован
                        lastRequest = req;
                    }
                    if (lastRequest != null) {
                        // Связываем новый future с future последней задачи
                        lastRequest.future.to(sendFuture); // Завершится вместе с последней задачей
                    } else {
                        // Маловероятно, но возможно, если очередь опустела между проверкой и итерацией
                        sendFuture.tryDone();
                    }
                } else {
                    sendFuture.tryDone(); // Нет данных и очередь пуста
                }
                return;
            }


            // 2. Create the TCP packet (Size + Payload)
            byte[] payload = combinedData.toArray(); //
            DataInOut packet = new DataInOut(payload.length + 10); // 10 is max packed size
            SerializerPackNumber.INSTANCE.put(packet, payload.length); //
            packet.write(payload); //

            // --- ИЗМЕНЕНИЕ: Оборачиваем ByteBuffer и Future ---
            // 3. Create WriteRequest and add to queue
            ByteBuffer bufferToSend = ByteBuffer.wrap(packet.toArray());
            WriteRequest writeRequest = new WriteRequest(bufferToSend, sendFuture);
            writeQueue.add(writeRequest);
            // --- УДАЛЕНО: sendFuture.tryDone(); ---
            // ----------------------------------------------------

            // 4. Register for write events on the reactor thread
            //    (Делаем это всегда, т.к. добавили данные в очередь)
            if (selectionKey != null && selectionKey.isValid()) {
                reactor.addTask(() -> {
                    try (var _l_task = logContext.context()) {
                        // Проверяем ключ снова внутри задачи реактора
                        if (selectionKey.isValid()) {
                            // Добавляем OP_WRITE, если он еще не установлен
                            int currentOps = selectionKey.interestOps();
                            if ((currentOps & SelectionKey.OP_WRITE) == 0) {
                                selectionKey.interestOps(currentOps | SelectionKey.OP_WRITE);
                                // Может понадобиться reactor.wakeup(), если поток реактора спит
                                // reactor.wakeup();
                            }
                        }
                    }
                });
            } else {
                // Если ключ невалиден, отменяем future
                Log.warn("SelectionKey invalid during flush scheduling, cancelling future.");
                sendFuture.tryCancel();
                // Очищаем очередь, так как записать уже не сможем
                clearWriteQueueAndCancelFutures();
            }
        }
    }


    /**
     * Closes the connection and notifies the parent (server or client).
     *
     * @param e The exception that caused the closure, or null.
     */
    @Override
    public void closeConnection(Exception e) {
        if (!closing.compareAndSet(false, true)) {
            return; // Already closing
        }

        LNode reasonNode = LNode.of("reason", e != null ? e.getMessage() : "shutdown");
        Log.info("Closing connection", logContext, reasonNode);

        // --- ИЗМЕНЕНИЕ: Отменяем все ожидающие futures ---
        clearWriteQueueAndCancelFutures();
        // ----------------------------------------------

        // 1. Notify the parent (server/client) for cleanup/reconnect
        //    Делаем это *после* отмены futures, чтобы избежать гонки состояний
        try {
            closeCallback.accept(this);
        } catch (Exception cbEx) {
            Log.error("Exception in close callback", cbEx, logContext);
        }

        // 2. Schedule the actual channel closure on the reactor thread
        reactor.addTask(() -> {
            if (selectionKey != null) {
                selectionKey.cancel();
            }
            try {
                channel.close();
            } catch (IOException ex) {
                Log.warn("Error closing socket channel", ex, logContext); //
            }
        });
    }

    // --- ДОБАВЛЕНО: Метод для очистки очереди записи ---
    private void clearWriteQueueAndCancelFutures() {
        Log.trace("Clearing write queue and cancelling pending futures.", logContext);
        WriteRequest req;
        while ((req = writeQueue.poll()) != null) {
            req.future.tryCancel();
        }
        isWritableFlag.set(false); // Считаем канал не записываемым
    }
    // ---------------------------------------------


    // --- FastMetaNet.Connection Implementation ---

    @Override
    public void read() {
        // No-op.
    }

    @Override
    public void stopRead() {
        // No-op.
    }

    @Override
    public AFuture write(byte[] data) {
        // Этот метод для "сырой" записи, не через FastMetaApi.flush()
        try (var _l = logContext.context()) {
            if (closing.get()) {
                Log.warn("Attempting write on a closing connection.", logContext);
                return AFuture.canceled(); // Используем статический метод
            }

            Log.warn("Direct write() called on connection, framing data...", logContext); //
            DataInOut packet = new DataInOut(data.length + 10);
            SerializerPackNumber.INSTANCE.put(packet, data.length); //
            packet.write(data); //

            // --- ИЗМЕНЕНИЕ: Создаем Future и WriteRequest ---
            AFuture writeFuture = AFuture.make();
            ByteBuffer bufferToSend = ByteBuffer.wrap(packet.toArray());
            WriteRequest writeRequest = new WriteRequest(bufferToSend, writeFuture);
            writeQueue.add(writeRequest);
            // ----------------------------------------------

            // Register for write
            if (selectionKey != null && selectionKey.isValid()) {
                reactor.addTask(() -> {
                    try (var _l_task = logContext.context()) {
                        if (selectionKey.isValid()) {
                            int currentOps = selectionKey.interestOps();
                            if ((currentOps & SelectionKey.OP_WRITE) == 0) {
                                selectionKey.interestOps(currentOps | SelectionKey.OP_WRITE);
                            }
                        }
                    }
                });
            } else {
                Log.warn("SelectionKey invalid during direct write scheduling, cancelling future.");
                writeFuture.tryCancel(); // Отменяем future, если ключ невалиден
            }
            // --- ИЗМЕНЕНИЕ: Возвращаем созданный Future ---
            return writeFuture;
            // ----------------------------------------------
        }
    }


    @Override
    public LT getLocalApi() {
        return localApi;
    }

    @Override
    public RT getRemoteApi() {
        return remoteApi;
    }

    @Override
    public boolean isWritable() {
        // Проверяем флаг и состояние закрытия
        return isWritableFlag.get() && !closing.get();
    }

    @Override
    public FastFutureContext getMetaContext() {
        return context;
    }

    @Override
    public AFuture destroy(boolean force) {
        // Используем существующий метод закрытия
        closeConnection(new IOException("Destroy called" + (force ? " (forced)" : "")));
        // Метод closeConnection асинхронный, поэтому возвращаем completed
        return AFuture.completed();
    }
}