package io.aether.net.fastMeta.nio;

import io.aether.logger.LNode;
import io.aether.logger.Log;
import io.aether.net.fastMeta.*;
import io.aether.utils.Destroyer;
import io.aether.utils.futures.AFuture;
import io.aether.utils.interfaces.AFunction;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.URI;
import java.nio.channels.SelectionKey;
import java.nio.channels.SocketChannel;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;

/**
 * NIO-based TCP implementation of {@link FastMetaClient}.
 * Manages a single {@link NioTcpConnection} and handles
 * automatic reconnection logic with increasing delay. Uses a {@link Destroyer}
 * to manage all connection and scheduler resources.
 */
class NioTcpFastMetaClient<LT, RT extends RemoteApi> implements FastMetaClient<LT, RT>, NioEventHandler {

    private static final long INITIAL_RECONNECT_DELAY_MS = 300;
    private static final long RECONNECT_DELAY_INCREMENT_MS = 100;
    private static final long MAX_RECONNECT_DELAY_MS = 2000;

    private long currentReconnectDelayMs = INITIAL_RECONNECT_DELAY_MS;

    private final NioReactor reactor;
    private final URI uri;
    private final FastMetaApi<LT, ?> localApiMeta;
    private final FastMetaApi<?, RT> remoteApiMeta;
    private final AFunction<RT, LT> localApiProvider;
    private final FastMetaNet.WritableConsumer writableConsumer;
    private final LNode logContext;
    private final Destroyer destroyer;

    private final AtomicReference<NioTcpConnection<LT, RT>> connection = new AtomicReference<>();
    private final ScheduledExecutorService reconnectScheduler;
    private volatile boolean active = true; // Controls the reconnect loop
    private volatile SelectionKey connectKey; // For the OP_CONNECT event

    NioTcpFastMetaClient(NioReactor reactor, URI uri,
                         FastMetaApi<LT, ?> lt, FastMetaApi<?, RT> rt,
                         AFunction<RT, LT> localApiProvider,
                         FastMetaNet.WritableConsumer writableConsumer) {
        this.reactor = reactor;
        this.uri = uri;
        this.localApiMeta = lt;
        this.remoteApiMeta = rt;
        this.localApiProvider = localApiProvider;
        this.writableConsumer = writableConsumer;
        this.logContext = Log.of("clientUri", uri, "socket","client nio");
        this.destroyer = new Destroyer("NioTcpFastMetaClient-" + uri.getPort());

        this.reconnectScheduler = Executors.newSingleThreadScheduledExecutor(r -> {
            String threadName = "fastmeta-client-reconnect-" + uri.getPort();
            Thread t = new Thread(r, threadName);
            t.setDaemon(true);
            return t;
        });

        this.destroyer.add((AutoCloseable) () -> {
            reconnectScheduler.shutdownNow();
        });

        tryConnect();
    }

    /**
     * Initiates a connection attempt on the reactor thread.
     */
    private void tryConnect() {
        if (!active) return;

        reactor.addTask(() -> {
            if (!active || connection.get() != null) {
                return;
            }

            try (var _l = logContext.context()) {
                Log.info("Attempting to connect...");
                SocketChannel channel = SocketChannel.open();
                channel.configureBlocking(false);

                boolean connecting = channel.connect(new InetSocketAddress(uri.getHost(), uri.getPort()));

                if (connecting) {
                    Log.info("Connected immediately");
                    finishConnection(channel);
                } else {
                    Log.trace("Connection pending...");
                    reactor.register(channel, SelectionKey.OP_CONNECT, this);
                }
            } catch (Exception e) {
                scheduleReconnect();
            }
        });
    }

    /**
     * Schedules the next reconnection attempt with calculated delay.
     */
    private void scheduleReconnect() {
        if (!active) return;

        writableConsumer.apply(false); // Уведомляем пользователя

        // --- НАЧАЛО ИЗМЕНЕНИЙ: Расчет и планирование задержки ---
        long delayToUse = currentReconnectDelayMs; // Задержка для *этой* попытки

        // Рассчитываем задержку для *следующей* попытки
        currentReconnectDelayMs = Math.min(
                currentReconnectDelayMs + RECONNECT_DELAY_INCREMENT_MS,
                MAX_RECONNECT_DELAY_MS
        );

        Log.info("Scheduling reconnect...", logContext, "delayMs", delayToUse, "nextDelayMs", currentReconnectDelayMs);
        reconnectScheduler.schedule(this::tryConnect, delayToUse, TimeUnit.MILLISECONDS);
        // --- КОНЕЦ ИЗМЕНЕНИЙ ---
    }

    /**
     * Handles the {@code OP_CONNECT} event from the reactor.
     *
     * @param key The {@link SelectionKey} from the reactor.
     */
    @Override
    public void handleEvent(SelectionKey key) {
        try (var _l = logContext.context()) {
            this.connectKey = key; // Сохраняем ключ на случай ошибки
            if (!key.isValid()) return;

            if (key.isConnectable()) {
                SocketChannel channel = (SocketChannel) key.channel();
                try {
                    if (channel.finishConnect()) {
                        // Успех!
                        key.interestOps(0); // Больше не интересуемся событиями этого ключа
                        // Не отменяем ключ (key.cancel()), так как канал используется дальше
                        this.connectKey = null; // Сбрасываем сохраненный ключ
                        finishConnection(channel);
                    }
                    // Если finishConnect() вернул false (маловероятно), просто ждем следующего события OP_CONNECT
                } catch (Exception e) {
                    Log.warn("Failed to finish connection", e, logContext);
                    closeConnection(e); // Закрываем канал и планируем переподключение
                }
            }
        }
    }

    /**
     * Finalizes the connection setup after the socket is connected.
     *
     * @param channel The successfully connected {@link SocketChannel}.
     */
    private void finishConnection(SocketChannel channel) {
        try (var _l = logContext.context()) {
            Log.info("Connection established", "local", channel.getLocalAddress());

            // --- НАЧАЛО ИЗМЕНЕНИЙ: Сброс задержки при успехе ---
            currentReconnectDelayMs = INITIAL_RECONNECT_DELAY_MS; // Сбрасываем задержку
            // --- КОНЕЦ ИЗМЕНЕНИЙ ---

            NioTcpConnection<LT, RT> conn = new NioTcpConnection<>(
                    reactor,
                    channel,
                    localApiMeta,
                    remoteApiMeta,
                    this::onConnectionClosed // Передаем колбэк для переподключения
            );

            LT localApi = localApiProvider.apply(conn.getRemoteApi());
            conn.setLocalApi(localApi);

            // Атомарно устанавливаем активное соединение
            NioTcpConnection<LT, RT> oldConn = connection.getAndSet(conn);
            if (oldConn != null) {
                Log.warn("Replacing an existing connection unexpectedly.");
                destroyer.remove(oldConn); // Удаляем старое из destroyer
                oldConn.destroy(true);     // Принудительно закрываем старое
            }
            this.destroyer.add(conn); // Добавляем новое

            writableConsumer.apply(true); // Уведомляем, что можно писать

            // Регистрируем *соединение* (conn) как обработчик OP_READ
            reactor.register(channel, SelectionKey.OP_READ, conn);

        } catch (Exception e) {
            Log.error("Failed to finalize connection setup", e);
            try {
                channel.close();
            } catch (IOException ignored) {
            }
            scheduleReconnect(); // Пробуем снова
        }
    }

    /**
     * Callback passed to {@link NioTcpConnection} to handle disconnects.
     *
     * @param closedConn The connection that was closed.
     */
    private void onConnectionClosed(NioTcpConnection<LT, RT> closedConn) {
        // Эта проверка гарантирует, что мы запланируем только одно переподключение
        if (connection.compareAndSet(closedConn, null)) {
            Log.warn("Client connection lost", logContext);
            scheduleReconnect(); // Планируем переподключение
        }
        // closedConn остается в destroyer
    }

    /**
     * Cleans up a failed connection attempt (e.g., from {@code finishConnect} failure).
     *
     * @param e The exception that caused the closure.
     */
    @Override
    public void closeConnection(Exception e) {
        SelectionKey key = this.connectKey; // Берем сохраненный ключ
        this.connectKey = null; // Сбрасываем
        if (key != null) {
            key.cancel();
            try {
                key.channel().close();
            } catch (IOException ex) {
                Log.warn("Error closing channel during connect failure", ex, logContext);
            }
        }
        scheduleReconnect(); // Планируем переподключение
    }

    @Override
    public void flush(AFuture sendFuture) {
        NioTcpConnection<LT, RT> conn = connection.get();
        if (conn != null) {
            conn.getMetaContext().flush(sendFuture);
        } else {
            sendFuture.tryCancel();
        }
    }

    @Override
    public void read() {
        // No-op
    }

    @Override
    public void stopRead() {
        // No-op
    }

    @Override
    public AFuture write(byte[] data) {
        NioTcpConnection<LT, RT> conn = connection.get();
        return (conn != null) ? conn.write(data) : AFuture.canceled();
    }

    @Override
    public LT getLocalApi() {
        NioTcpConnection<LT, RT> conn = connection.get();
        return (conn != null) ? conn.getLocalApi() : null;
    }

    @Override
    public RT getRemoteApi() {
        NioTcpConnection<LT, RT> conn = connection.get();
        return (conn != null) ? conn.getRemoteApi() : null;
    }

    @Override
    public boolean isWritable() {
        NioTcpConnection<LT, RT> conn = connection.get();
        return (conn != null) && conn.isWritable();
    }

    @Override
    public FastFutureContext getMetaContext() {
        NioTcpConnection<LT, RT> conn = connection.get();
        return (conn != null) ? conn.getMetaContext() : FastFutureContext.STUB;
    }

    @Override
    public AFuture destroy(boolean force) {
        Log.info("Destroying client", logContext);
        active = false; // Останавливаем все попытки переподключения

        // Закрываем отложенное подключение, если оно есть
        SelectionKey key = this.connectKey;
        this.connectKey = null;
        if (key != null) {
            reactor.addTask(() -> {
                key.cancel();
                try { key.channel().close(); } catch (IOException ignored) {}
            });
        }

        // Destroyer позаботится о reconnectScheduler и активном соединении (если оно есть)
        return destroyer.destroy(force);
    }
}