package io.aether.net.fastMeta.nio;

import io.aether.logger.LNode;
import io.aether.logger.Log;
import io.aether.net.fastMeta.FastMetaApi;
import io.aether.net.fastMeta.FastMetaNet;
import io.aether.net.fastMeta.FastMetaServer;
import io.aether.net.fastMeta.RemoteApi;
import io.aether.utils.Destroyer;
import io.aether.utils.futures.AFuture;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.URI;
import java.nio.channels.SelectionKey;
import java.nio.channels.ServerSocketChannel;
import java.nio.channels.SocketChannel;
import java.util.HashSet;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * NIO-based TCP implementation of {@link FastMetaServer}.
 * Listens on a port and accepts incoming connections, wrapping each
 * in an {@link NioTcpConnection}. Uses a {@link Destroyer}
 * to manage all connection and channel resources.
 */
class NioTcpFastMetaServer<LT, RT extends RemoteApi> implements FastMetaServer<LT, RT>, NioEventHandler {

    private final NioReactor reactor;
    private final FastMetaApi<LT, ?> localApiMeta;
    private final FastMetaApi<?, RT> remoteApiMeta;
    private final FastMetaServer.Handler<LT, RT> handler;
    private final ServerSocketChannel serverChannel;
    private final LNode logContext;
    private final Destroyer destroyer;

    /**
     * A thread-safe map of all active connections managed by this server.
     * Required for the handlers() method.
     */
    private final Map<NioTcpConnection<LT, RT>, Boolean> connections = new ConcurrentHashMap<>();

    NioTcpFastMetaServer(NioReactor reactor, URI uri, FastMetaApi<LT, ?> localApiMeta,
                         FastMetaApi<?, RT> remoteApiMeta, FastMetaServer.Handler<LT, RT> handler) {
        this.reactor = reactor;
        this.localApiMeta = localApiMeta;
        this.remoteApiMeta = remoteApiMeta;
        this.handler = handler;
        this.logContext = Log.of("serverUri", uri,"socket","server nio"); //
        this.destroyer = new Destroyer("NioTcpFastMetaServer-" + uri.getPort()); //

        try (var _l = logContext.context()) { //
            Log.info("Starting TCP server..."); //
            this.serverChannel = ServerSocketChannel.open();
            this.serverChannel.bind(new InetSocketAddress(uri.getHost(), uri.getPort()));
            this.serverChannel.configureBlocking(false);

            // Add the server channel to the destroyer for cleanup
            this.destroyer.add(this.serverChannel); //

            reactor.register(serverChannel, SelectionKey.OP_ACCEPT, this);
            Log.info("TCP server started and listening"); //
        } catch (IOException e) {
            Log.error("Failed to start TCP server", e); //
            throw new RuntimeException("Failed to start TCP server", e);
        }
    }

    /**
     * Handles I/O events for the server socket (i.e., new connections).
     *
     * @param key The {@link SelectionKey} from the reactor.
     */
    @Override
    public void handleEvent(SelectionKey key) {
        // --- ADDED LOG CONTEXT WRAPPER ---
        try (var _l = logContext.context()) {
            if (key.isAcceptable()) {
                handleAccept();
            }
        }
    }

    /**
     * Called when a new client connection is ready to be accepted.
     */
    private void handleAccept() {
        try (var _l = logContext.context()) { //
            SocketChannel clientChannel = serverChannel.accept();
            if (clientChannel == null) {
                return; // Should not happen, but defensive check
            }

            clientChannel.configureBlocking(false);
            Log.info("Accepted new connection", "remote", clientChannel.getRemoteAddress()); //

            // 1. Create the connection wrapper
            NioTcpConnection<LT, RT> connection = new NioTcpConnection<>(
                    reactor,
                    clientChannel,
                    localApiMeta,
                    remoteApiMeta,
                    this::onConnectionClosed // Callback to remove from map
            );

            // 2. Get the user's API implementation for this specific connection
            LT localApi = handler.onNewConnection(connection);
            connection.setLocalApi(localApi); // Complete the connection's setup

            // 3. Add to active list and destroyer
            connections.put(connection, true);
            destroyer.add(connection); //

            // 4. Register the new connection with the reactor for reading
            reactor.register(clientChannel, SelectionKey.OP_READ, connection);

        } catch (Exception e) {
            Log.warn("Failed to accept new connection", e); //
        }
    }

    /**
     * Callback passed to {@link NioTcpConnection} to handle cleanup.
     * This is called when a connection closes *on its own*.
     *
     * @param connection The connection that was closed.
     */
    private void onConnectionClosed(NioTcpConnection<LT, RT> connection) {
        if (connections.remove(connection) != null) {
            Log.info("Connection removed from server", connection.getLogContext()); //
            handler.onConnectionClose(connection);
        }
        // We do *not* remove it from the destroyer.
        // The destroyer will attempt to destroy it again on shutdown,
        // but the connection's idempotent `destroy` will handle this.
    }

    /**
     * Called by the reactor if the *server socket* itself fails.
     *
     * @param e The exception that caused the closure.
     */
    @Override
    public void closeConnection(Exception e) {
        Log.error("Server socket failed, stopping server...", e, logContext); //
        destroy(true); // Force destroy
    }

    @Override
    public AFuture stop() {
        //
        return destroy(false); // Graceful shutdown
    }

    @Override
    public AFuture destroy(boolean force) {
        //
        Log.info("Destroying TCP server...", logContext); //

        // Clear the map so handlers() returns empty.
        // The actual connections are cleaned up by the destroyer.
        connections.clear();

        // The destroyer holds the serverChannel and all connections
        return destroyer.destroy(force); //
    }

    @Override
    public Iterable<FastMetaNet.Connection<LT, RT>> handlers() {
        // Return a snapshot to prevent ConcurrentModificationException
        return new HashSet<>(connections.keySet());
    }
}