package io.aether.logger;

import io.aether.utils.AString;
import io.aether.utils.RU;
import io.aether.utils.flow.Flow;
import io.aether.utils.interfaces.ABiConsumer;
import it.unimi.dsi.fastutil.objects.Object2ObjectArrayMap;
import it.unimi.dsi.fastutil.objects.ObjectArrayList;

import java.lang.ref.Reference;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

public abstract class LNode {
    public static final LNode EMPTY = new LNode() {
        @Override
        public int count() {
            return 0;
        }


        @Override
        public void foreach(Set<String> exclude, ABiConsumer<String, Object> c) {

        }

        @Override
        protected Object get(String key, Set<LNode> old) {
            return null;
        }

        @Override
        public Object get(String key) {
            return null;
        }

    };

    public final long id = LogImpl.ID_COUNTER.getAndIncrement();

    public boolean contains(String key) {
        return get(key) != null;
    }

    public boolean msgContains(String c) {
        return getMsg().contains(c);
    }

    public boolean messageContains(String c) {
        return getMessage().indexOf(c) >= 0;
    }

    public AString getMessage() {
        return getMessage(Set.of());
    }

    public String getMsg() {
        return String.valueOf(get(Log.MSG));
    }

    public AString getMessage(Set<String> keys) {
        var sb = AString.of();
        printMessage(sb, keys, null, null);
        return sb;
    }

    public void printMessage(AString w, Set<String> keys, String color, String defaultColor) {
        try {
            String msg = getCast(Log.MSG);
            if (msg == null) return;
            w.addVars(msg, k -> {
                var kk = k.toString();
                keys.add(k.toString());
                return get(kk);
            });
            var e = getException();
            if (e != null) {
                w.add(" [");
                w.addStackTrace(e);
                w.add("]");
            }
        } catch (Exception e) {
            w.add("error print message: ").addStackTrace(e);
        }
    }

    public Map<String, Object> toMap() {
        Map<String, Object> map = new Object2ObjectArrayMap<>();
        Set<String> s = new HashSet<>();
        foreach(s, map::put);
        return map;
    }

    @Override
    public String toString() {
        var sb = AString.of();
        Log.Level l = getCast(Log.LEVEL);
        if (l != null) {
            sb.add(String.valueOf(get(Log.LEVEL)));
            sb.add(" ");
        }
        Set<String> exclude = new HashSet<>();
        exclude.add(Log.LEVEL);
        exclude.add(Log.MSG);

        printMessage(sb, exclude, null, null);
        sb.add(" ");
        foreach(exclude, (k, v) -> {
            sb.add(k).add(" = ");
            sb.add(v);
            sb.add(", ");
        });
        return sb.toString();
    }

    public <T> T getCast(String key) {
        return RU.cast(get(key));
    }

    public Object getSystemComponent() {
        return get(Log.SYSTEM_COMPONENT);
    }

    public boolean checkSystemComponent(Object val) {
        return Objects.equals(getSystemComponent(), val);
    }

    protected abstract Object get(String key, Set<LNode> old);

    public Object get(String key) {
        return get(key, new HashSet<>());
    }

    public abstract int count();

    public abstract void foreach(Set<String> exclude, ABiConsumer<String, Object> c);

    public boolean isTrace() {
        return check(Log.LEVEL, Log.Level.TRACE);
    }

    public boolean isDebug() {
        return check(Log.LEVEL, Log.Level.DEBUG);
    }

    public boolean isInfo() {
        return check(Log.LEVEL, Log.Level.INFO);
    }

    public boolean isWarn() {
        return check(Log.LEVEL, Log.Level.WARN);
    }

    public boolean isError() {
        return check(Log.LEVEL, Log.Level.ERROR);
    }

    public boolean check(Object key, Object value) {
        return Objects.equals(get(String.valueOf(key)), value);
    }

    public boolean isEmpty() {
        return count() == 0;
    }

    public Throwable getException() {
        Throwable res = getCast(Log.EXCEPTION_STR);
        if (res == null) {
            res = getCast("error");
        }
        return res;
    }

    public Log.Level getLevel() {
        return getCast(Log.LEVEL);
    }

    public void set(String key, Object value) {
        throw new UnsupportedOperationException();
    }

    public static LNode of(LNode data) {
        return data;
    }

    public static LNode of(Object... data) {
        if (data.length == 0) {
            return EMPTY;
        }
        assert data.length % 2 == 0;
        for (int i = 0; i < data.length; i += 2) {
            if (data[i] == null) {
                List<Object> l = new ObjectArrayList<>(data.length);
                for (int i2 = 0; i2 < data.length; i2 += 2) {
                    if (data[i] != null && data[i + 1] != null) {
                        l.add(data[i]);
                        l.add(data[i + 1]);
                    }
                }
                return of(l.toArray());
            }
            if (!(data[i] instanceof String)) {
                throw new IllegalStateException();
            }
        }
        return new LNode() {
            @Override
            public void foreach(Set<String> exclude, ABiConsumer<String, Object> c) {
                for (int i = 0; i < data.length; i += 2) {
                    var k = String.valueOf(data[i]);
                    if (exclude.add(k)) {
                        c.accept(k, data[i + 1]);
                    }
                }
            }

            @Override
            protected Object get(String key, Set<LNode> old) {
                for (int i = 0; i < data.length; i += 2) {
                    var k = String.valueOf(data[i]);
                    if (Objects.equals(k, key)) {
                        var res = data[i + 1];
                        if (res instanceof Reference) {
                            res = ((Reference<?>) res).get();
                        }
                        return res;
                    }
                }
                return null;
            }

            @Override
            public int count() {
                return data.length / 2;
            }

        };
    }

    public static LNode ofMulti(LNode... data) {
        int count = 0;
        if (data == null || data.length == 0) return EMPTY;
        if (data.length == 1) {
            if (data[0] == null) return EMPTY;
            return data[0];
        }
        for (var e : data) {
            if (e == null || e == EMPTY) {
                data = Flow.flow(data)
                        .filterExclude(EMPTY)
                        .filterNotNull()
                        .toArray(LNode.class);
                return ofMulti(Flow.flow(data).filterNotNull().toArray(LNode.class));
            }
            count += e.count();
        }
        int finalCount = count;
        LNode[] finalData1 = data;
        if (count == 0) {
            return LNode.EMPTY;
        }
        return new LNode() {
            @Override
            public void foreach(Set<String> exclude, ABiConsumer<String, Object> c) {
                for (var n : finalData1) {
                    n.foreach(exclude, c);
                }
            }

            @Override
            protected Object get(String key, Set<LNode> old) {
                for (var n : finalData1) {
                    if (old.add(n)) {
                        var res = n.get(key, old);
                        if (res instanceof Reference) {
                            res = ((Reference<?>) res).get();
                        }
                        if (res != null) {
                            return res;
                        }
                    }
                }
                return null;
            }

            @Override
            public int count() {
                return finalCount;
            }

        };
    }

    public static LNode ofMulti(Collection<LNode> data) {
        int count = 0;
        for (var e : data) {
            count += e.count();
        }
        int finalCount = count;
        return new LNode() {
            @Override
            public void foreach(Set<String> exclude, ABiConsumer<String, Object> c) {
                for (var n : data) {
                    n.foreach(exclude, c);
                }
            }

            @Override
            protected Object get(String key, Set<LNode> old) {
                for (var n : data) {
                    if (old.add(n)) {
                        var res = n.get(key, old);
                        if (res instanceof Reference) {
                            res = ((Reference<?>) res).get();
                        }
                        if (res != null) {
                            return res;
                        }
                    }
                }
                return null;
            }

            @Override
            public int count() {
                return finalCount;
            }

        };
    }

    public static LNode of2(Object[] keys, Object[] vals) {
        if (keys.length != vals.length) {
            Log.warn("bad node", "keys", keys, "vals", vals);
            return EMPTY;
        }
        return new LNode() {
            @Override
            public void foreach(Set<String> exclude, ABiConsumer<String, Object> c) {
                for (int i = 0; i < keys.length; i++) {
                    var k = String.valueOf(keys[i]);
                    if (exclude.add(k)) {
                        var v = vals[i];
                        c.accept(k, v);
                    }
                }
            }

            @Override
            protected Object get(String key, Set<LNode> old) {
                for (int i = 0; i < keys.length; i++) {
                    var k = String.valueOf(keys[i]);
                    if (Objects.equals(k, key)) {
                        var res = vals[i];
                        if (res instanceof Reference) {
                            res = ((Reference<?>) res).get();
                        }
                        return res;
                    }
                }
                return null;
            }

            @Override
            public int count() {
                return keys.length;
            }

        };
    }

    public static LNode of2(List<? extends CharSequence> keys, List<?> vals) {
        if (keys.size() != vals.size()) {
            Log.warn("bad node", "keys", keys, "vals", vals);
            return EMPTY;
        }
        return new LNode() {
            @Override
            public void foreach(Set<String> exclude, ABiConsumer<String, Object> c) {
                for (int i = 0; i < keys.size(); i++) {
                    var k = String.valueOf(keys.get(i));
                    if (exclude.add(k)) {
                        var v = vals.get(i);
                        c.accept(k, v);
                    }
                }
            }

            @Override
            protected Object get(String key, Set<LNode> old) {
                for (int i = 0; i < keys.size(); i++) {
                    var k = String.valueOf(keys.get(i));
                    if (Objects.equals(k, key)) {
                        var res = vals.get(i);
                        if (res instanceof Reference) {
                            res = ((Reference<?>) res).get();
                        }
                        return res;
                    }
                }
                return null;
            }

            @Override
            public int count() {
                return keys.size();
            }

        };
    }

    public static LNode ofMap(Map<String, ?> map) {
        return new NodeMap(map);
    }

    public static LNode ofMap() {
        return ofMap(new ConcurrentHashMap<>());
    }

    public static class NodeMap extends LNode {
        private final Map<String, ?> map;

        public NodeMap() {
            this(new ConcurrentHashMap<>());
        }

        public NodeMap(Map<String, ?> map) {
            this.map = map;
        }

        @Override
        public void foreach(Set<String> exclude, ABiConsumer<String, Object> c) {
            for (var e : map.entrySet()) {
                var k = e.getKey();
                if (exclude.add(k)) {
                    var v = e.getValue();
                    c.accept(k, v);
                }
            }
        }

        public void set(String key, Object val) {
            map.put(key, RU.cast(val));
        }

        @Override
        protected Object get(String key, Set<LNode> old) {
            return map.get(key);
        }

        @Override
        public Object get(String key) {
            return map.get(key);
        }

        @Override
        public int count() {
            return map.size();
        }

    }
}
