package io.aether.logger;

import io.aether.utils.AString;
import io.aether.utils.RU;
import io.aether.utils.interfaces.ABiConsumer;
import io.aether.utils.interfaces.AConsumer;
import io.aether.utils.interfaces.AFunction;
import it.unimi.dsi.fastutil.objects.ObjectOpenHashSet;

import java.util.Set;

public class LogPrinter implements AutoCloseable {
    public final LogFilter filter;
    final Column[] columns;
    final Set<String> keys;
    final AutoCloseable closeable;

    public LogPrinter(Column[] columns) {
        this(columns, new LogFilter());
    }

    public LogPrinter(Column[] columns, LogFilter filter) {
        this.columns = columns;
        this.filter = filter;
        keys = new ObjectOpenHashSet<>();
        for (var c : columns) {
            keys.add(c.getKey());
        }
        closeable = Log.addListener(filter, n -> {
            printRow(AString.of(), n);
        });
    }

    public void printRow(AString s, LNode n) {
        System.out.println(printNode(s, n));
    }

    @Override
    public void close() {
        try {
            closeable.close();
        } catch (Exception e) {
            RU.error(e);
        }
    }

    public AString printNode(AString s, LNode n) {
        Set<String> kk = new ObjectOpenHashSet<>();
        for (var c : columns) {
            c.print(this, s, n, kk);
        }
        if (n.isError()) {
            s.add("\n");
            var e = n.getException();
            if (e != null) {
                e.printStackTrace(s.toPrintWriter());
            }
        }
        return s;
    }

    public static Column col(String key, ColumnPrinter printer) {
        return new Column() {
            @Override
            public String getKey() {
                return key;
            }

            @Override
            public void print(LogPrinter p, AString out, LNode value, Set<String> workKeys) {
                printer.print(out, value, workKeys);
                workKeys.add(key);
            }
        };
    }

    public static Column col(String key, ColumnPrinter2 printer) {
        return new Column() {
            @Override
            public String getKey() {
                return key;
            }

            @Override
            public void print(LogPrinter p, AString out, LNode value, Set<String> workKeys) {
                printer.print(out, value);
                workKeys.add(key);
            }
        };
    }

    public static Column col(String key) {
        return Column.of(key);
    }

    public static Column splitter(String text) {
        return new Column() {
            @Override
            public void print(LogPrinter printer, AString out, LNode value, Set<String> workKeys) {
                out.add(text);
            }

            @Override
            public String getKey() {
                return null;
            }
        };
    }

    public static <T> Column col(String key, AFunction<T, Object> mapper) {
        return Column.of(key, mapper);
    }

    public static Column colAll() {
        return new Column() {
            @Override
            public void print(LogPrinter printer, AString out, LNode value, Set<String> workKeys) {
                out.add('{');
                value.foreach(workKeys, new ABiConsumer<>() {
                    boolean first = true;

                    @Override
                    public void accept2(String k, Object v) throws Throwable {
                        if (first) {
                            first = false;
                        } else {
                            out.add(", ");
                        }
                        out.add(k);
                        out.add('=');
                        if (v instanceof String) {
                            out.add('"').add((String) v).add('"');
                        } else {
                            out.add(v);
                        }
                    }
                });
                out.add('}');
            }

            @Override
            public String getKey() {
                return null;
            }
        };
    }

    public interface ColumnPrinter {
        void print(AString out, LNode value, Set<String> workKeys);
    }

    public interface ColumnPrinter2 {
        void print(AString out, LNode node);
    }

    public interface Column {
        void print(LogPrinter printer, AString out, LNode value, Set<String> workKeys);

        String getKey();


        default Column quote(AConsumer<AString> begin, AConsumer<AString> end) {
            var self = this;
            return new Column() {
                @Override
                public void print(LogPrinter printer, AString out, LNode value, Set<String> workKeys) {
                    begin.accept(out);
                    self.print(printer, out, value, workKeys);
                    end.accept(out);
                }

                @Override
                public String getKey() {
                    return self.getKey();
                }
            };
        }

        default Column max(int i) {
            var self = this;
            return new Column() {
                @Override
                public String getKey() {
                    return self.getKey();
                }

                @Override
                public void print(LogPrinter printer, AString out, LNode value, Set<String> workKeys) {
                    self.print(printer, out.limit(i), value, workKeys);
                }
            };
        }

        default Column foregroundColorRGB(int r, int g, int b) {
            var self = this;
            return new Column() {
                @Override
                public String getKey() {
                    return self.getKey();
                }

                @Override
                public void print(LogPrinter printer, AString out, LNode value, Set<String> workKeys) {
                    out.styleForeground(null, r, g, b);
                    self.print(printer, out, value, workKeys);
                    out.styleClear();
                }
            };
        }

        default Column backgroundColorRGB(int r, int g, int b) {
            var self = this;
            return new Column() {
                @Override
                public String getKey() {
                    return self.getKey();
                }

                @Override
                public void print(LogPrinter printer, AString out, LNode value, Set<String> workKeys) {
                    out.styleBackground(null, r, g, b);
                    self.print(printer, out, value, workKeys);
                    out.styleClear();
                }
            };
        }

        default Column style(AString.Style style, AString.Color color) {
            var self = this;
            return new Column() {
                @Override
                public String getKey() {
                    return self.getKey();
                }

                @Override
                public void print(LogPrinter printer, AString out, LNode value, Set<String> workKeys) {
                    out.style(style, color);
                    self.print(printer, out, value, workKeys);
                    out.styleClear();
                }
            };
        }

        default Column minAuto() {
            var self = this;
            return new Column() {
                int min = 0;

                @Override
                public String getKey() {
                    return self.getKey();
                }

                @Override
                public void print(LogPrinter printer, AString out, LNode value, Set<String> workKeys) {
                    var l = out.length();
                    self.print(printer, out, value, workKeys);
                    var l2 = out.length();
                    var d = l2 - l;
                    if (min <= d) {
                        min = d;
                    } else {
                        for (var i = 0; i < min - d; i++) {
                            out.add(' ');
                        }
                    }

                }
            };
        }

        default Column min(int i) {
            var self = this;
            return new Column() {
                @Override
                public String getKey() {
                    return self.getKey();
                }

                @Override
                public void print(LogPrinter printer, AString out, LNode value, Set<String> workKeys) {
                    var l = out.length();
                    self.print(printer, out, value, workKeys);
                    var l2 = out.length();
                    var d = l2 - l;
                    if (d < i) {
                        for (int ii = 0; ii < (i - d); ii++) {
                            out.add(" ");
                        }
                    }
                }
            };
        }

        static <T> Column of(String key, AFunction<T, Object> mapper) {
            return new Column() {
                @Override
                public void print(LogPrinter printer, AString out, LNode value, Set<String> workKeys) {
                    out.add(mapper.apply(RU.cast(value.get(key))));
                    workKeys.add(key);
                }

                @Override
                public String getKey() {
                    return key;
                }
            };
        }

        static Column of(String key) {
            return new Column() {
                @Override
                public void print(LogPrinter printer, AString out, LNode node, Set<String> workKeys) {
                    var v = node.get(key);
                    if (v != null) {
                        out.add(v);
                    }
                    workKeys.add(key);
                }

                @Override
                public String getKey() {
                    return key;
                }
            };
        }
    }
}
